/*
 *  linux/drivers/video/vfb.c -- Virtual frame buffer device
 *
 *	Copyright (C) 1997 Geert Uytterhoeven
 *
 *  This file is subject to the terms and conditions of the GNU General Public
 *  License. See the file COPYING in the main directory of this archive for
 *  more details.
 */
#define ALLOW_OS_CODE 1

#include "kernelcalls.h"
#include "gbus.h"
#include "mambolfb.h"

#include <linux/module.h>
#include <linux/vmalloc.h>
#include <asm/uaccess.h>
#include <linux/fb.h>

#include <video/fbcon.h>
#include <video/fbcon-mfb.h>
#include <video/fbcon-cfb2.h>
#include <video/fbcon-cfb4.h>
#include <video/fbcon-cfb8.h>
#include <video/fbcon-cfb16.h>
#include <video/fbcon-cfb24.h>
#include <video/fbcon-cfb32.h>

#if 0
#define DEB(f) (f)
#else
#define DEB(f)
#endif

#define RM_OSDBUFWIDTH 640
#define RM_OSDBUFHEIGHT 480
#define RM_OSDBUFDEPTH 32

static u_long videomemory=0, videomemorysize=0, palette=0;
static char bound=0;
static char *mode;

MODULE_PARM(videomemorysize, "l");
MODULE_PARM_DESC(videomemorysize, "Size of the OSD buffer.\n");

MODULE_PARM(videomemory, "l");
MODULE_PARM_DESC(videomemory, "Address of the OSD buffer (DRAM address).\n");

MODULE_PARM(palette, "l");
MODULE_PARM_DESC(palette, "Address of the OSD palette (DRAM address).\n");

MODULE_PARM(mode, "s");
MODULE_PARM_DESC(mode, "Video mode : with:height:bpp.\n");

#ifdef MODULE_LICENSE
MODULE_LICENSE("GPL");
#endif

#define CHECK_BOUND()									\
if (!bound){										\
	printk(KERN_INFO "%s : Mambo/Tango frame buffer not bound yet.\n", __func__);	\
	return -EAGAIN;									\
}

static int currcon = 0;
static struct display disp;
static struct fb_info fb_info;

#if defined(FBCON_HAS_CFB16) || defined(FBCON_HAS_CFB24) || defined(FBCON_HAS_CFB32)
static union {
#ifdef FBCON_HAS_CFB16
	u16 cfb16[16];
#endif
#ifdef FBCON_HAS_CFB24
	u32 cfb24[16];
#endif
#ifdef FBCON_HAS_CFB32
	u32 cfb32[16];
#endif
} fbcon_cmap;
#endif /* FBCON_HAS_CFB16 || FBCON_HAS_CFB24 || FBCON_HAS_CFB32 */

static char mambolfb_name[32] = "Mambo/Tango Linux Frame Buffer";

/* Those values arr changed during init */
static struct fb_var_screeninfo mambolfb_default = {
	RM_OSDBUFWIDTH, RM_OSDBUFHEIGHT, RM_OSDBUFWIDTH, RM_OSDBUFHEIGHT, 0, 0, RM_OSDBUFDEPTH, 0,
	{16, 8, 0}, {8, 8, 0}, {0, 8, 0}, {24, 8, 0},
	0, 0, -1, -1, 0, 20000, 64, 64, 32, 32, 64, 2,
	0, FB_VMODE_NONINTERLACED
};

static struct osd_descriptor{
	RMuint32 region_index;          //First region used to map the OSD buffer
	RMuint32 region_count;          //How many regions for the OSD buffer
	RMuint32 offset;                //offset from the base address of the first region
} osd;

static struct llad *p_llad;
static struct gbus *p_gbus;

/* Some RedHat systems have CONFIG_HIGHPTE, so pte_offset is renamed */
#ifndef pte_offset
#define pte_offset pte_offset_kernel
#endif

/* This is used to get the physical address or cookie associated to an iomapped area 
 *  See http://www.xml.com/ldd/chapter/book/ch13.html */
#if (RMARCHID==RMARCHID_MIPS)
#define io_virt_to_phys(v) kc_virt_to_phys(v)
#else
#define io_virt_to_phys(v) (pte_val(*(pte_t *)pte_offset(pmd_offset(pgd_offset_k((v)), (v)), (v))) & PAGE_MASK) + ((v) & ~PAGE_MASK)
#endif

static int mambolfb_mmap(struct fb_info *info, struct file *file, struct vm_area_struct *vma)
{
	unsigned long phys;	/* physical address to map */

	CHECK_BOUND();

#if (RMPLATFORM==RMPLATFORMID_AOE6_SH4)
	phys = videomemory;
#elif (RMPLATFORM==RMPLATFORMID_JASPERMAMBO)
	phys = videomemory;
#elif (RMPLATFORM==RMPLATFORMID_IXDP425)
	phys = videomemory;
#else
	phys = io_virt_to_phys(videomemory);
#endif

	printk("mambolfb_mmap: OSD videomemory asked:0x%08lx\n",phys);

	/* This should really be kc_io_remap_range, but as long as we don't
	 * port on alpha or sparc, it's enough */
	if (kc_remap_page_range((struct kc_vm_area_struct *) vma,
	                        vma->vm_start, phys,
	                        vma->vm_end-vma->vm_start,
	                        (struct kc_pgprot_t *) &vma->vm_page_prot)) 
		return -EAGAIN;

	printk("mambolfb_mmap remapped %ld bytes in userland of process %d at address vma_start0x%08lx to phys=0x%08lx\n",
	       vma->vm_end-vma->vm_start,current->pid,vma->vm_start, phys); 

	return 0;
}


static u_long get_line_length(int xres_virtual, int bpp)
{
	u_long length;
    
	length = xres_virtual*bpp;
	length = (length+31)&-32;
	length >>= 3;
	return(length);
}

static void mambolfb_encode_fix(struct fb_fix_screeninfo *fix,
                             struct fb_var_screeninfo *var)
{
	memset(fix, 0, sizeof(struct fb_fix_screeninfo));
	strcpy(fix->id, mambolfb_name);
	fix->smem_start = videomemory;
	fix->smem_len = videomemorysize;
	fix->type = FB_TYPE_PACKED_PIXELS;
	fix->type_aux = 0;
	switch (var->bits_per_pixel) {
		case 1:
			fix->visual = FB_VISUAL_MONO01;
			break;
		case 2:
		case 4:
		case 8:
			fix->visual = FB_VISUAL_PSEUDOCOLOR;
			break;
		case 16:
		case 24:
		case 32:
			fix->visual = FB_VISUAL_TRUECOLOR;
			break;
	}
	fix->ywrapstep = 1;
	fix->xpanstep = 1;
	fix->ypanstep = 1;
	fix->line_length = get_line_length(var->xres_virtual, var->bits_per_pixel);
}

/*
 *  Get the Fixed Part of the Display
 */
static int mambolfb_get_fix(struct fb_fix_screeninfo *fix, int con,
                         struct fb_info *info)
{
	struct fb_var_screeninfo *var;

	CHECK_BOUND();

	DEB(printk("mambolfb_get_fix, con = %d\n",con));
	if (con == -1)
		var = &mambolfb_default;
	else
		var = &fb_display[con].var;
	mambolfb_encode_fix(fix, var);
	return 0;
}

static void set_color_bitfields(struct fb_var_screeninfo *var)
{
	switch (var->bits_per_pixel) {
		case 1:
		case 8:
			var->red.offset = 0;
			var->red.length = 8;
			var->green.offset = 0;
			var->green.length = 8;
			var->blue.offset = 0;
			var->blue.length = 8;
			var->transp.offset = 0;
			var->transp.length = 0;
			break;
		case 16:	/* RGB 565 */
			var->red.offset = 11;
			var->red.length = 5;
			var->green.offset = 5;
			var->green.length = 6;
			var->blue.offset = 0;
			var->blue.length = 5;
			var->transp.offset = 0;
			var->transp.length = 0;
			break;
		case 24:	/* RGB 888 */
			var->red.offset = 16;
			var->red.length = 8;
			var->green.offset = 8;
			var->green.length = 8;
			var->blue.offset = 0;
			var->blue.length = 8;
			var->transp.offset = 0;
			var->transp.length = 0;
			break;
		case 32:	/* ARGB 8888 */
			var->red.offset = 16;
			var->red.length = 8;
			var->green.offset = 8;
			var->green.length = 8;
			var->blue.offset = 0;
			var->blue.length = 8;
			var->transp.offset = 24;
			var->transp.length = 8;
			break;
	}
	var->red.msb_right = 0;
	var->green.msb_right = 0;
	var->blue.msb_right = 0;
	var->transp.msb_right = 0;
}


/*
 *  Get the User Defined Part of the Display
 */
static int mambolfb_get_var(struct fb_var_screeninfo *var, int con,
                         struct fb_info *info)
{
	CHECK_BOUND();

	if (con == -1)
		*var = mambolfb_default;
	else
		*var = fb_display[con].var;
	set_color_bitfields(var);
	return 0;
}


/*
 *  Read a single color register and split it into
 *  colors/transparent. Return != 0 for invalid regno.
 *  # on 84XX
 */
static int mambolfb_getcolreg(u_int regno, u_int *red, u_int *green, u_int *blue,
                           u_int *transp, struct fb_info *info)
{
	RMuint32 color ;
	
	if (regno > 255)
		return 1;
	DEB(printk("getcolreg %d\n",regno));

	color = gbus_read_uint32(p_gbus,palette+regno*4);
	*red = color >> 16 & 0xFF;
	*green = color >> 8 & 0xFF;
	*blue = color & 0xFF;
	*transp = 0xFF;
	return 0;
}

/*
 *  Set a single color register. The values supplied are already
 *  rounded down to the hardware's capabilities (according to the
 *  entries in the var structure). Return != 0 for invalid regno.
 */
static int mambolfb_setcolreg(u_int regno, u_int red, u_int green, u_int blue,
                           u_int transp, struct fb_info *info)
{
	RMuint32 color;

	CHECK_BOUND();
	
	if (regno > 255)
		return 1;
	DEB(printk("setcolreg reg %d, (%lx,%lx,%lx,%lx)\n",regno,red,green,blue,transp));
	red >>= 8;
	green >>= 8;
	blue >>= 8;
	
	color = 0xFF << 24 | red << 16 | green << 8 | blue;
	gbus_write_uint32(p_gbus,palette+regno*4,color);

	/* Set colors for fb console */
	if (regno < 16){
		switch(fb_display[currcon].var.bits_per_pixel){
			case 16:
#ifdef FBCON_HAS_CFB16
				red >>= 3;
				green >>= 2;
				blue >>= 3;
				fbcon_cmap.cfb16[regno] = (u16) red << 11 | (u16) green << 5 | (u16) blue ;
#endif
				break;
			case 24:
#ifdef FBCON_HAS_CFB24
				fbcon_cmap.cfb24[regno] = 0xFF << 24 | red << 16 | green << 8 | blue ;
#endif
				break;
			case 32:
#ifdef FBCON_HAS_CFB32
				fbcon_cmap.cfb32[regno] = 0xFF << 24 | red << 16 | green << 8 | blue ;
#endif
				break;
			default:
				printk(KERN_WARNING "Warning, mambolfb_setcolreg, unsupported resolution : %d\n",
				       info->var.bits_per_pixel);
		}
	}

	return 0;
}

static void do_install_cmap(int con, struct fb_info *info)
{
	if (con != currcon)
		return;
	if (fb_display[con].cmap.len)
		fb_set_cmap(&fb_display[con].cmap, 1, mambolfb_setcolreg, info);
	else
		fb_set_cmap(fb_default_cmap(1<<fb_display[con].var.bits_per_pixel), 1,
		            mambolfb_setcolreg, info);
}

/*
 *  Set the User Defined Part of the Display (different en 84XX)
 */
static int mambolfb_set_var(struct fb_var_screeninfo *var, int con,
		         struct fb_info *info)
{
	int err, activate = var->activate;
	int oldxres, oldyres, oldvxres, oldvyres, oldbpp;
	u_long line_length;
	struct display *display;

	CHECK_BOUND();

	DEB(printk("mambolfb_set_var, con = %d\n",con));

	if (con >= 0)
		display = &fb_display[con];
	else
		display = &disp;	/* used during initialization */

	/*
	 *  FB_VMODE_CONUPDATE and FB_VMODE_SMOOTH_XPAN are equal!
	 *  as FB_VMODE_SMOOTH_XPAN is only used internally
	 */

	if (var->vmode & FB_VMODE_CONUPDATE) {
		var->vmode |= FB_VMODE_YWRAP;
		var->xoffset = display->var.xoffset;
		var->yoffset = display->var.yoffset;
	}

	/*
	 *  Some very basic checks
	 */
	if (!var->xres)
		var->xres = 1;
	if (!var->yres)
		var->yres = 1;
	if (var->xres > var->xres_virtual)
		var->xres_virtual = var->xres;
	if (var->yres > var->yres_virtual)
		var->yres_virtual = var->yres;
	if (var->bits_per_pixel <= 1)
		var->bits_per_pixel = 1;
	else if (var->bits_per_pixel <= 8)
		var->bits_per_pixel = 8;
	else if (var->bits_per_pixel <= 16)
		var->bits_per_pixel = 16;
	/* fbcon doesn't support this (yet) */
	else if (var->bits_per_pixel <= 24)
		var->bits_per_pixel = 24;
	else if (var->bits_per_pixel <= 32)
		var->bits_per_pixel = 32;
	else
		return -EINVAL;

	/*
	 *  Memory limit
	 */
	line_length = get_line_length(var->xres_virtual, var->bits_per_pixel);
	DEB(printk("Line length = %d (%dx%d)\n",line_length,var->xres_virtual,var->bits_per_pixel));
	if (line_length*var->yres_virtual > videomemorysize){
		printk("Error, frame buffer doesn't have enough memory\n");
		return -ENOMEM;
	}

	set_color_bitfields(var);

	if ((activate & FB_ACTIVATE_MASK) == FB_ACTIVATE_NOW) {
		oldxres = display->var.xres;
		oldyres = display->var.yres;
		oldvxres = display->var.xres_virtual;
		oldvyres = display->var.yres_virtual;
		oldbpp = display->var.bits_per_pixel;
		display->var = *var;
		if (oldxres != var->xres || oldyres != var->yres ||
		    oldvxres != var->xres_virtual || oldvyres != var->yres_virtual ||
		    oldbpp != var->bits_per_pixel) {
			struct fb_fix_screeninfo fix;

			mambolfb_encode_fix(&fix, var);
			display->screen_base = (char *)videomemory;
			display->visual = fix.visual;
			display->type = fix.type;
			display->type_aux = fix.type_aux;
			display->ypanstep = fix.ypanstep;
			display->ywrapstep = fix.ywrapstep;
			display->line_length = fix.line_length;
			display->can_soft_blank = 0;
			display->inverse = 0;
			display->scrollmode = SCROLL_YREDRAW; // bovine: imagine what it does on RAM
			switch (var->bits_per_pixel) {
#ifdef FBCON_HAS_MFB
				case 1:
					display->dispsw = &fbcon_mfb;
					break;
#endif
#ifdef FBCON_HAS_CFB2
				case 2:
					display->dispsw = &fbcon_cfb2;
					break;
#endif
#ifdef FBCON_HAS_CFB4
				case 4:
					display->dispsw = &fbcon_cfb4;
					break;
#endif
#ifdef FBCON_HAS_CFB8
				case 8:
					display->dispsw = &fbcon_cfb8;
					break;
#endif
#ifdef FBCON_HAS_CFB16
				case 16:
					display->dispsw = &fbcon_cfb16;
					display->dispsw_data = fbcon_cmap.cfb16;
					break;
#endif
#ifdef FBCON_HAS_CFB24
				case 24:
					display->dispsw = &fbcon_cfb24;
					display->dispsw_data = fbcon_cmap.cfb24;
					break;
#endif
#ifdef FBCON_HAS_CFB32
				case 32:
					display->dispsw = &fbcon_cfb32;
					display->dispsw_data = fbcon_cmap.cfb32;
					break;
#endif
				default:
					display->dispsw = &fbcon_dummy;
					break;
			}
			if (fb_info.changevar)
				(*fb_info.changevar)(con);
		}
		if (oldbpp != var->bits_per_pixel) {
			if ((err = fb_alloc_cmap(&display->cmap, 0, 0)))
				return err;
			do_install_cmap(con, info);
		}
	}
	return 0;
}


/*
 *  Pan or Wrap the Display (same on 84XX and mambolfb)
 *
 *  This call looks only at xoffset, yoffset and the FB_VMODE_YWRAP flag
 */
static int mambolfb_pan_display(struct fb_var_screeninfo *var, int con, struct fb_info *info)
{
	CHECK_BOUND();

	if (var->vmode & FB_VMODE_YWRAP) {
		if (var->yoffset < 0 ||
				var->yoffset >= fb_display[con].var.yres_virtual ||
				var->xoffset)
			return -EINVAL;
	} else {
		if (var->xoffset+fb_display[con].var.xres >
				fb_display[con].var.xres_virtual ||
				var->yoffset+fb_display[con].var.yres >
				fb_display[con].var.yres_virtual)
			return -EINVAL;
	}
	fb_display[con].var.xoffset = var->xoffset;
	fb_display[con].var.yoffset = var->yoffset;
	if (var->vmode & FB_VMODE_YWRAP)
		fb_display[con].var.vmode |= FB_VMODE_YWRAP;
	else
		fb_display[con].var.vmode &= ~FB_VMODE_YWRAP;
	return 0;
}

/*
 *  Get the Colormap
 */
static int mambolfb_get_cmap(struct fb_cmap *cmap, int kspc, int con, struct fb_info *info)
{
	CHECK_BOUND();

	if (con == currcon) /* current console? */
		return fb_get_cmap(cmap, kspc, mambolfb_getcolreg, info);
	else if (fb_display[con].cmap.len) /* non default colormap? */
		fb_copy_cmap(&fb_display[con].cmap, cmap, kspc ? 0 : 2);
	else
		fb_copy_cmap(fb_default_cmap(1<<fb_display[con].var.bits_per_pixel),
		             cmap, kspc ? 0 : 2);
	return 0;
}

/*
 *  Set the Colormap
 */
static int mambolfb_set_cmap(struct fb_cmap *cmap, int kspc, int con, struct fb_info *info)
{
	int err;

	CHECK_BOUND();

	if (!fb_display[con].cmap.len) {	/* no colormap allocated? */
		if ((err = fb_alloc_cmap(&fb_display[con].cmap,
		                         1<<fb_display[con].var.bits_per_pixel, 0)))
			return err;
	}
	if (con == currcon)			/* current console? */
		return fb_set_cmap(cmap, kspc, mambolfb_setcolreg, info);
	else
		fb_copy_cmap(cmap, &fb_display[con].cmap, kspc ? 0 : 1);
	return 0;
}


/* almost same on 84XX */
static int mambolfbcon_switch(int con, struct fb_info *info)
{
	/* Do we have to save the colormap? */
	if (fb_display[currcon].cmap.len)
		fb_get_cmap(&fb_display[currcon].cmap, 1, mambolfb_getcolreg, info);

	currcon = con;
	/* Install new colormap */
	do_install_cmap(con, info);
	return 0;
}

/*
 *  Update the `var' structure (called by fbcon.c)
 */
static int mambolfbcon_updatevar(int con, struct fb_info *info)
{
	/* Nothing */
	return 0;
}

/*
 *  Blank the display.
 */
static void mambolfbcon_blank(int blank, struct fb_info *info)
{
	/* Nothing */
}

/** 
 * Unlock PCI regions locked to map the frame buffer, if any
 * @return < 0 on error
 */
static int unlock_current_mambolfb(void)
{
	int status = 0,i;
	
	if (bound){
		for (i=0;i<osd.region_count; i++){
			if (RMFAILED(gbus_unlock_region(p_gbus, osd.region_index+i)))
				status = -EINVAL;
		}
	}

	return status;
}

/**
 * Lock and map a memory space as the new frame buffer
 * @return < 0 on error
 */
static int lock_map_mambolfb(void)
{
	int status;

	memset((void *)&osd,0,sizeof(struct osd_descriptor));
		
	/* Lock the regions we need */
	status = gbus_lock_area(p_gbus, &osd.region_index, videomemory, videomemorysize, &osd.region_count, &osd.offset);
	if (RMFAILED(status)){
		printk("Error locking PCI chunks\n");
		return -EINVAL;
	}
	DEB(printk("locked %ld regions, starting from region %ld at offset 0x%08lx\n",
	           osd.region_count, osd.region_index, osd.offset));

	/* Update video address to the mapped OSD buffer */
	videomemory = (u_long) gbus_map_region(p_gbus, osd.region_index, osd.region_count);
	if (videomemory == (u_long) NULL){
		printk("Error mapping OSD buffer in kernel space\n");
		status = gbus_unlock_region(p_gbus, osd.region_index);
		if (RMFAILED(status))
			printk("Error unlocking regions\n");
		return -EINVAL;
	}
	videomemory += osd.offset;
	DEB(printk("OSD videomemory=0x%lx (%ld bytes)\n",videomemory,videomemorysize));

	return 0;
}

	

/**
 * Handle specific IOCTL
 * @param inode - inode of the underlying device
 * @param file - file of the underlying device
 * @param cmd - command
 * @param arg - pointer to the argument
 * @param con - tty or visible console attached to this fb
 * @param info - the frame buffer descriptor
 * @return < 0 on error
 *
 */
static int mambolfb_ioctl(struct inode *inode, struct file *file, unsigned int cmd,
		unsigned long arg, int con, struct fb_info *info)
{
	struct mambolfb_config_s config;
	
	switch(cmd){
	case MAMBOLFBIO_SETBUFFER:
		if (copy_from_user(&config, (void *)arg, sizeof(config)))
			return -EFAULT;
		
		/* Cleanup the previous buffer if any */
		if (unlock_current_mambolfb() < 0)
			/* Go on if it fails */
			printk(KERN_INFO "Error unlocking Mambo/Tango frame buffer.\n");

		/* Setup the frame buffer with the new config */
		mambolfb_default.xres = config.xres;
		mambolfb_default.yres = config.yres;
		mambolfb_default.xres_virtual = config.xres;
		mambolfb_default.yres_virtual = config.yres;
		mambolfb_default.bits_per_pixel = config.bits_per_pixel;
		mambolfb_default.activate = FB_ACTIVATE_NOW;
		videomemory = config.videomemory;
		videomemorysize = config.videomemorysize;
		palette = config.palette;

		if (mambolfb_default.xres == 0 ||
		    mambolfb_default.yres == 0 ||
		    mambolfb_default.bits_per_pixel == 0 ||
		    videomemory == 0 ||
		    videomemorysize == 0 ||
		    palette == 0){
			bound = 0;
			printk(KERN_INFO "Mambo/Tango frame buffer unbound.\n");
			return 0;
		} 

		bound = 1;
		if (lock_map_mambolfb() < 0){
			printk(KERN_INFO "Error locking new Mambo/Tango frame buffer.\n");
			return -EFAULT;
		}

		if (mambolfb_set_var(&mambolfb_default, -1, &fb_info) < 0){
			printk(KERN_INFO "Error setting new var.\n");
			return -EFAULT;
		}

		printk(KERN_INFO "fb%d: Mambo/Tango frame buffer device, using %ldK of video memory\n",
		       GET_FB_IDX(fb_info.node), videomemorysize>>10);
		return 0;

	case MAMBOLFBIO_GETBUFFER:
		/* TODO implement this one too */
		return -ENOSYS;
		
	default:
		printk(KERN_INFO "%s : unkown command 0x%x\n", __func__, cmd);
		return -EFAULT;
	}
		
	return -EINVAL;
}

static struct fb_ops mambolfb_ops = {
	owner:		THIS_MODULE,
	fb_get_fix:	mambolfb_get_fix,
	fb_get_var:	mambolfb_get_var,
	fb_set_var:	mambolfb_set_var,
	fb_get_cmap:	mambolfb_get_cmap,
	fb_set_cmap:	mambolfb_set_cmap,
	fb_pan_display:	mambolfb_pan_display,
	fb_mmap:        mambolfb_mmap,
	fb_ioctl:	mambolfb_ioctl
};

/*
 *  Initialisation
 */
static int __init mambolfb_init(void)
{
	memset((void *)&fb_info,0,sizeof(struct fb_info));
	memset((void *)&disp,0,sizeof(struct display));
	strcpy(fb_info.modename, mambolfb_name);
	fb_info.changevar = NULL;
	fb_info.node = -1;
	fb_info.fbops = &mambolfb_ops;
	fb_info.disp = &disp;
	fb_info.switch_con = &mambolfbcon_switch;
	fb_info.updatevar = &mambolfbcon_updatevar;
	fb_info.blank = &mambolfbcon_blank;
	fb_info.flags = FBINFO_FLAG_DEFAULT;
	strcpy(fb_info.fontname,"Acorn8x8");

	if (bound && mambolfb_set_var(&mambolfb_default, -1, &fb_info))
		return -EINVAL;

	DEB(printk("registering fb\n"));
	if (register_framebuffer(&fb_info) < 0) {
		return -EINVAL;
	}

	if (bound)
		printk(KERN_INFO "fb%d: Mambo/Tango frame buffer device, using %ldK of video memory\n",
		       GET_FB_IDX(fb_info.node), videomemorysize>>10);
	else
		printk(KERN_INFO "fb%d: Mambo/Tango frame buffer device, not bound to any physical buffer yet\n",
		       GET_FB_IDX(fb_info.node));

	return 0;
}


static int __init parse(char *video_mode)
{
	char *token;
	
	token = strsep(&video_mode, ":");
	if (!token || !video_mode)
		return 0;
	mambolfb_default.xres = simple_strtoul(token,&token,0);
	mambolfb_default.xres_virtual = mambolfb_default.xres;

	token = strsep(&video_mode, ":");
	if (!token || !video_mode)
		return 0;
	mambolfb_default.yres = simple_strtoul(token,&token,0);
	mambolfb_default.yres_virtual = mambolfb_default.yres;
	mambolfb_default.bits_per_pixel = simple_strtoul(video_mode,&video_mode,0);

	return 1;
}


static int __init init_mambolfb(void)
{
	if (videomemory && videomemorysize && palette)
		/* We have a physical buffer */
		bound=1;
	else 
		/* No buffer given yet */
		bound=0;
	
	if (mode){
		DEB(printk("mode = %s\n",mode));
		if (parse(mode) == 0){
			return -EINVAL;
		}
		DEB(printk("mode = %ldx%ld %ld bpp\n", mambolfb_default.xres, mambolfb_default.yres, 
		           mambolfb_default.bits_per_pixel));
	}
	
	/* TODO add chip number as a parameter */
	p_llad = llad_open("0");
	if (p_llad == NULL){
		printk("Error, cannot open llad 0\n");
		return -EINVAL;
	}

	p_gbus = gbus_open(p_llad);
	if (p_gbus == NULL){
		llad_close(p_llad);
		printk("Error, cannot open gbus 0\n");
		return -EINVAL;
	}

	if (bound && (lock_map_mambolfb() < 0))
		return -EINVAL;

	return mambolfb_init();
}

static void __exit cleanup_mambolfb(void)
{
	unregister_framebuffer(&fb_info);
	if (unlock_current_mambolfb() < 0)
		/* Go on if it fails */
		printk(KERN_INFO "Error unlocking Mambo/Tango frame buffer.\n");

	gbus_close(p_gbus);
	llad_close(p_llad);
}

module_init(init_mambolfb);
module_exit(cleanup_mambolfb);

/* vim:set shiftwidth=8 tabstop=8 noexpandtab: */ 
