/*
 * arch/arm/mach-em86xx/em86xxapi.c
 *
 * Copyright (C) 2003-2004 Sigma Designs, Inc
 *
 * by Ho Lee 03/07/2003
 */

#include <linux/config.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <asm/hardware.h>
#include <asm/io.h>
#include <asm/arch/em86xxapi.h>
#include <linux/irq.h>
#include <linux/sched.h>

//
// global variables
//

//
// init
//

int __init em86xx_init(void)
{
	em86xx_mbus_init();

#ifdef CONFIG_EM86XX_UART0_AS_GPIO_FULL
	printk("EM86XX: Setting UART0 as full GPIO.\n");
	__raw_writel(0x7f7f, REG_BASE_CPU + CPU_uart0_gpio_mode_w);
#endif
#ifdef CONFIG_EM86XX_UART0_AS_GPIO_PARTIAL
	printk("EM86XX: Setting UART0 as partial GPIO.\n");
	__raw_writel(0x7f6e, REG_BASE_CPU + CPU_uart0_gpio_mode_w);
#endif
#ifdef CONFIG_EM86XX_UART1_AS_GPIO_FULL
	printk("EM86XX: Setting UART1 as full GPIO.\n");
	__raw_writel(0x7f7f, REG_BASE_CPU + CPU_uart1_gpio_mode_w);
#endif
#ifdef CONFIG_EM86XX_UART1_AS_GPIO_PARTIAL
	printk("EM86XX: Setting UART1 as partial GPIO.\n");
	__raw_writel(0x7f6e, REG_BASE_CPU + CPU_uart1_gpio_mode_w);
#endif

	return 0;
}

__initcall(em86xx_init);

//
// Cache control
//

#if 0
void em86xx_get_cache_state(int *picache, int *pdcache, int *pwriteback)
{
	unsigned int reg;

	__asm__ __volatile__ (
		"mrc	p15, 0, %0, c1, c0, 0\n"
		: "=r" (reg)
	);

	if (picache)
		*picache = (reg & 0x1000);
	if (pdcache)
		*pdcache = (reg & 0x0004);
	if (pwriteback)
		*pwriteback = (reg & 0x0002);
}

void em86xx_enable_cache(int icache, int dcache, int writeback)
{
	unsigned int reg;

	__asm__ __volatile__ (
		"mrc	p15, 0, %0, c1, c0, 0\n"
		: "=r" (reg)
	);

	if (icache > 0)
		reg |= 0x1000;
	else if (icache == 0) {
		reg &= ~0x1000;
		em86xx_flush_cache_data();
	}
	
	if (dcache > 0)
		reg |= 0x0004;
	else if (dcache == 0) {
		reg &= ~0x0004;
		em86xx_flush_cache_data();
	}
	
	if (writeback > 0)
		reg |= 0x0002;
	else if (writeback == 0) {
		reg &= ~0x0002;
		em86xx_clean_cache_data();
	}

	__asm__ __volatile__ (
		"mcr	p15, 0, %0, c1, c0, 0\n"
		: : "r" (reg)
	);
}

void em86xx_clean_cache_data_region(unsigned int from, unsigned int to)
{
    unsigned int flag;

	from &= ~0x0f;
	to &= ~0x0f;

	__save_flags_cli(flag);
	for (; from <= to; from += 0x10) {
		__asm__ __volatile__ (
			"mcr	p15, 0, %0, c7, c10, 1\n"	// clean data
			: : "r" (from)
		);
	}
	__restore_flags(flag);
}

void em86xx_invalidate_cache_instruction(void)
{
    unsigned int flag;

    __save_flags_clif(flag);
	__asm__(
        "mcr    p15, 0, %0, c7, c10, 0\n"       // clean data
        "mcr    p15, 0, %0, c7, c5, 0\n"        // invalidate instruction
		: : "r" (0)
	);
    __restore_flags(flag);
}

void em86xx_invalidate_cache_instruction_region(unsigned int from, unsigned int to)
{
    unsigned int flag;

	from &= ~0x0f;
	to &= ~0x0f;
		
	__save_flags_cli(flag);
	for (; from <= to; from += 0x10) {
		__asm__ __volatile__ (
            "mcr    p15, 0, %0, c7, c10, 1\n"   // clean data
            "mcr    p15, 0, %0, c7, c5, 1\n"    // invalidate instruction
			: : "r" (from)
		);
	}
	__restore_flags(flag);
}

void em86xx_invalidate_cache_data(void)
{
	__asm__(
        "mcr    p15, 0, %0, c7, c6, 0\n"        // invalidate data
		: : "r" (0)
	);
}

void em86xx_invalidate_cache_data_region(unsigned int from, unsigned int to)
{
    unsigned int flag;

	from &= ~0x0f;
	to &= ~0x0f;
		
	__save_flags_cli(flag);
	for (; from <= to; from += 0x10) {
		__asm__ __volatile__ (
            "mcr    p15, 0, %0, c7, c6, 1\n"    // invalidate data
			: : "r" (from)
		);
	}
	__restore_flags(flag);
}

void em86xx_flush_cache_all(void)
{
    unsigned int flag;

    __save_flags_clif(flag);
	__asm__(
		"mcr	p15, 0, %0, c7, c10, 0\n"		// clean data cache
		"mcr	p15, 0, %0, c7, c5, 0\n"		// invalidate instruction cache
		"mcr	p15, 0, %0, c7, c6, 0\n"		// invalidate data cache
		: : "r" (0)
	);
    __restore_flags(flag);
}

void em86xx_flush_cache_data_region(unsigned int from, unsigned int to)
{
    unsigned int flag;

	from &= ~0x0f;
	to &= ~0x0f;
		
	__save_flags_cli(flag);
	for (; from <= to; from += 0x10) {
		__asm__ __volatile__ (
			"mcr	p15, 0, %0, c7, c10, 1\n"	// clean data cache
			"mcr	p15, 0, %0, c7, c6, 1\n"	// invalidate data cache
			: : "r" (from)
		);
	}
	__restore_flags(flag);
}
#endif

void em86xx_clean_cache_data(void)
{
	__asm__(
		"mcr	p15, 0, %0, c7, c10, 0\n"		// clean data
		: : "r" (0)
	);
}

void em86xx_flush_cache_data(void)
{
    unsigned int flag;

    __save_flags_clif(flag);
	__asm__(
		"mcr	p15, 0, %0, c7, c10, 0\n"		// clean data cache
		"mcr	p15, 0, %0, c7, c6, 0\n"		// invalidate data cache
		: : "r" (0)
	);
    __restore_flags(flag);
}

//
// Memory
//

unsigned int em86xx_get_pciregionsize(void)
{
	unsigned int regsize;

	// calculate PCI memory address space size in MB
   	regsize = 1 << (__raw_readl(REG_BASE_HOST + PCI_devcfg_reg3) & 0x07);
	// calculate each memory region size in byte
	regsize = (regsize << 20) >> 3;

	return regsize;
}

unsigned int em86xx_get_dmamemorysize(void)
{
	unsigned int dmamemorysize = (unsigned int)em86xx_kmemsize;

#ifdef CONFIG_SD_EM86XX_PCIDMA
	unsigned int regsize = em86xx_get_pciregionsize();
#ifdef CONFIG_SD_EM86XX_PCIDMA_MATCHADDR
	regsize *= 8;
#else
	regsize *= 6;
#endif
	// adjust DMA memory size information
	if (dmamemorysize > regsize)
		dmamemorysize = regsize;
#endif

	return dmamemorysize;
}

//
// switchbox
// 

static int g_sbox_map[SBOX_MAX + 1], g_sbox_lastmap[2];;
static int g_sbox_used[2] = { 0, 0 };

// Initialize SwitchBox 
int em86xx_sbox_init(void)
{
#ifdef CONFIG_BLK_DEV_EM86XX_BMIDEDMA
	em86xx_sbox_setup(SBOX_IDEDVD, SBOX_PCIMASTER);
#else
	em86xx_sbox_setup(SBOX_IDEFLASH, SBOX_PCIMASTER);
#endif
	em86xx_sbox_reset();

	return 0;
}

void em86xx_sbox_reset(void)
{
#if defined(CONFIG_ARCH_MAMBO)
	__raw_writel(0x1f1f1f1f, REG_BASE_HOST + SBOX_reset);
	__raw_writel(0x1f001f00, REG_BASE_HOST + SBOX_reset);
#elif defined(CONFIG_ARCH_TANGO)
#ifndef CONFIG_SD_KEEP_SBOX_R1W1
	__raw_writel(0x7f7fffff, REG_BASE_HOST + SBOX_reset);
	__raw_writel(0x7f00ff00, REG_BASE_HOST + SBOX_reset);
#else
	/* Leave W1/R1 alone. */
	__raw_writel(0x7d7dfdfd, REG_BASE_HOST + SBOX_reset);
	__raw_writel(0x7d00fd00, REG_BASE_HOST + SBOX_reset);
#endif
#endif
}

// connect R0/W0 and R1/W1 to specified interface if0, if1
int em86xx_sbox_setup(int if0, int if1)
{
	int i;
	unsigned int data;

#if defined(CONFIG_ARCH_MAMBO)
	g_sbox_map[0] = if0;
#ifndef CONFIG_SD_KEEP_SBOX_R1W1
	g_sbox_map[1] = if1;
#else
	g_sbox_map[1] = 0; /* Leave W1 alone */
#endif
	g_sbox_map[SBOX_IDEFLASH] = 7;
	g_sbox_map[SBOX_PCIMASTER] = 7;
	g_sbox_map[SBOX_PCISLAVE] = 7;

	g_sbox_map[if0] = 0;
#ifndef CONFIG_SD_KEEP_SBOX_R1W1
	g_sbox_map[if1] = 1;	/* Leave R1 alone. */
#endif
#elif defined(CONFIG_ARCH_TANGO)
	g_sbox_map[0] = if0 + 1;
#ifndef CONFIG_SD_KEEP_SBOX_R1W1
	g_sbox_map[1] = if1 + 1;
#else
	g_sbox_map[1] = 0; /* Leave W1 alone */
#endif
	g_sbox_map[SBOX_PCIMASTER] = 0xf;
	g_sbox_map[SBOX_PCISLAVE] = 0xf;
	g_sbox_map[SBOX_CIPHER] = 0xf;
	g_sbox_map[SBOX_IDEDVD] = 0xf;
	g_sbox_map[SBOX_IDEFLASH] = 0xf;
	g_sbox_map[SBOX_SFLASH] = 0xf;

	g_sbox_map[if0] = 1;
#ifndef CONFIG_SD_KEEP_SBOX_R1W1
	g_sbox_map[if1] = 2;	/* Leave R1 alone */
#endif
#else
#error Unknown Architecture
#endif

	for (i = SBOX_MAX, data = 0; i >= 0; --i)
		data = (data << 4) | g_sbox_map[i];

	__raw_writel(data, REG_BASE_HOST + SBOX_route);
	return 0;
}

int em86xx_sbox_connect(int iface)
{
	int port = -1;

#if defined(CONFIG_ARCH_MAMBO)
	if (iface == g_sbox_map[0] && g_sbox_map[iface] == 0)
		port = 0;
#elif defined(CONFIG_ARCH_TANGO)
	if (iface + 1 == g_sbox_map[0] && g_sbox_map[iface] == 1) 
		port = 0;
#else
#error Unknown Architecture
#endif

#if 1
	else
		printk("MBUS Error : MBUS channel allocation failed\n");
#else

#if defined(CONFIG_ARCH_MAMBO)
	else if (iface == g_sbox_map[1] && g_sbox_map[iface] == 1) {
		port = 1;
#elif defined(CONFIG_ARCH_TANGO)
	else if (iface + 1 == g_sbox_map[1] && g_sbox_map[iface] == 2) {
		port = 1;
#else
#error Unknown Architecture
#endif

	} else {
		int i, firstavail;

		// check if last used slot is available
		for (i = 0; i < 2; ++i) {
			if (g_sbox_lastmap[i] == iface && !g_sbox_used[i]) {
				port = i;
				break;
			}
		}

		// look for first available slot
		if (port < 0) {
			for (i = 0, firstavail = -1; i < 2; ++i) {
				if (!g_sbox_used[i]) {
					if (firstavail < 0)
						firstavail = i;
					if (g_sbox_lastmap[i] == 0) {
						port = i;
						break;
					}
				}
			}
			if (port < 0)
				port = firstavail;
		}

		if (port >= 0) {
			g_sbox_map[port] = iface;
			em86xx_sbox_setup(g_sbox_map[0], g_sbox_map[1]);
		}
	}
#endif
	
	if (port >= 0) {
		g_sbox_used[port] = 1;
		g_sbox_lastmap[port] = iface;
	}

	return port;
}

void em86xx_sbox_disconnect(int port)
{
	if (port >= 0)
		g_sbox_used[port] = 0;
}

//
// MBUS interface 
// 

static int s_mbus_irq_inited[4] = {0,0,0,0};
static mbus_irq_handler_t s_mbus_irq_handler[4] = {0,0,0,0};
static void *s_mbus_irq_handler_arg[4] = {NULL, NULL, NULL, NULL};

static void em86xx_mbus_init_irq(int irq);
static void em86xx_mbus_irq(int irq, void *devinfo, struct pt_regs *regs);

int em86xx_mbus_init(void)
{
	return 0;
}

// idx : [0..3]
void em86xx_mbus_init_irq(int idx)
{
	static char *s_irqname[] = {
		"EM86XX MBUS W0", 
		"EM86XX MBUS W1", 
		"EM86XX MBUS R0", 
		"EM86XX MBUS R1", 
	};
	
	if (!s_mbus_irq_inited[idx]) {
		request_irq(idx + IRQ_MBUS_W0, em86xx_mbus_irq, SA_SHIRQ, s_irqname[idx], (void *)idx);
		s_mbus_irq_inited[idx] = 1;
	}
}

void em86xx_mbus_irq(int irq, void *devinfo, struct pt_regs *regs)
{
	int idx = irq - IRQ_MBUS_W0;

	em86xx_irq_clr(irq);
	
	// printk("[MBUS %d, %d, %x, %d] ", idx, __raw_readl(REG_BASE_HOST + MIF_w0_base + idx * 0x40 + 0x0c), (__raw_readl(REG_BASE_CPU + CPU_edge_rawstat) >> 9) & 0x0f, __raw_readl(REG_BASE_HOST + PB_automode_control) & 0xffff);

	if (s_mbus_irq_handler[idx]) {
		(*s_mbus_irq_handler[idx])(irq, s_mbus_irq_handler_arg[idx]);
		s_mbus_irq_handler[idx] = NULL;
	}
}

// sbox : SBOX_xxx
// fromdev : 
//   0 : EM86XX => device (writing) : use Wx port
//   1 : device => EM86XX (reading) : use Rx port
unsigned int em86xx_mbus_alloc_dma(int sbox, int fromdev, unsigned int *pregbase, int *pirq)
{
	int port = em86xx_sbox_connect(sbox);
	int x = port + (fromdev ? 0 : 2);
	
	if (pirq) 
		*pirq = IRQ_MBUS_W0 + x;
	if (pregbase) 
		*pregbase = REG_BASE_HOST + MIF_w0_base + x * 0x40;

	return port;
}

void em86xx_mbus_free_dma(int port)
{
	em86xx_sbox_disconnect(port);
}

#define MBUS_LINEAR_MAX 	(0x2000 - 0x200)

int em86xx_mbus_setup_dma_common(unsigned int regbase, unsigned int addr, unsigned int count, mbus_irq_handler_t handler, void *arg)
{
	int idx = (regbase - REG_BASE_HOST - MIF_w0_base) / 0x40;
	int flags;

	if (em86xx_mbus_inuse(regbase) != 0) {
		printk("MBUS Error : previous command is pending\n");
		return 1;
	}

	if (handler) {
		em86xx_mbus_init_irq(idx);
		__save_flags_cli(flags);
		s_mbus_irq_handler[idx] = handler;
		s_mbus_irq_handler_arg[idx] = arg;
		__restore_flags(flags);

	}

	return 0;
}

void em86xx_mbus_setup_dma_linear(unsigned int regbase, unsigned int addr, unsigned int count)
{
	if (em86xx_mbus_inuse(regbase) != 0) {
		printk("MBUS Error : previous command is pending (%d)\n",
			__raw_readl(regbase + MIF_cmd_offset));
	}
	__raw_writel(addr, regbase + MIF_add_offset);
	__raw_writel(count, regbase + MIF_cnt_offset);
	__raw_writel(0x05, regbase + MIF_cmd_offset);
}

void em86xx_mbus_setup_dma_double(unsigned int regbase, unsigned int addr, unsigned int count, unsigned int addr2, unsigned int count2)
{
	if (em86xx_mbus_inuse(regbase) != 0) {
		printk("MBUS Error : previous command is pending (%d)\n",
			__raw_readl(regbase + MIF_cmd_offset));
	}

	__raw_writel(addr, regbase + MIF_add_offset);
	__raw_writel((count2 << 16) | count, regbase + MIF_cnt_offset);
	__raw_writel(addr2, regbase + MIF_add2_skip_offset);
	__raw_writel(0x06, regbase + MIF_cmd_offset);
}

void em86xx_mbus_setup_dma_rectangle(unsigned int regbase, unsigned int addr, unsigned int horiz, unsigned int lines, int skip)
{
	if (em86xx_mbus_inuse(regbase) != 0) {
		printk("MBUS Error : previous command is pending (%d)\n",
			__raw_readl(regbase + MIF_cmd_offset));
	}

	__raw_writel(addr, regbase + MIF_add_offset);
	__raw_writel((lines << 16) | horiz, regbase + MIF_cnt_offset);
	__raw_writel(horiz, regbase + MIF_add2_skip_offset);
	__raw_writel(0x07, regbase + MIF_cmd_offset);
}

int em86xx_mbus_setup_dma(unsigned int regbase, unsigned int addr, unsigned int count, mbus_irq_handler_t handler, void *arg)
{
	if (em86xx_mbus_setup_dma_common(regbase, addr, count, handler, arg))
		return 1;

	if (count <= MBUS_LINEAR_MAX) {
		// try linear
		// printk("Linear\n");
		em86xx_mbus_setup_dma_linear(regbase, addr, count);
	} else if (count <= (MBUS_LINEAR_MAX * 2)) {
		// try double
		// printk("Double (%04x, %004x)\n", MBUS_LINEAR_MAX, count - MBUS_LINEAR_MAX);
		em86xx_mbus_setup_dma_double(regbase, addr, MBUS_LINEAR_MAX, addr + MBUS_LINEAR_MAX, count - MBUS_LINEAR_MAX);
	} else {
		int i, horiz, lines;

		// try rectangle
		for (i = 0, horiz = 1; i < 10 && ((count & 0x01) == 0); ++i, horiz <<= 1) 
			;
		lines = count >> i;
		if (horiz > MBUS_LINEAR_MAX || lines > MBUS_LINEAR_MAX) {
			printk("MBUS Error : too big to transfer (%d, %d)\n", horiz, lines);
			return 1;
		}
		// printk("Rectangle (%d x %d)\n", horiz, lines);
		em86xx_mbus_setup_dma_rectangle(regbase, addr, horiz, lines, horiz);
	}

	return 0;
}

int em86xx_mbus_inuse(unsigned int regbase)
{
	return ((__raw_readl(regbase + MIF_cmd_offset) & 0x7) != 0) ? 1 : 0;
}

#if defined(CONFIG_ARCH_MAMBO)
static unsigned int sbox_reset_vals[1][4] = {
			{0x01010404, 0x02020404, 0x04040101, 0x04040202}};
static unsigned int sbox_unreset_vals[1][4] = {
			{0x01000400, 0x02000400, 0x04000100, 0x04000200}};
#elif defined(CONFIG_ARCH_TANGO)
/* Note: 06/24/2004 there're errors in the spec. The bit assignments should be
	(* indicates error)
	Bit 0/8: MBUS_R0_SBOX
	Bit 1/9: MBUS_R1_SBOX
	Bit 2/10: PCI_MASTER_SBOX
	Bit 3/11: PCI_SLAVE_SBOX
	Bit 4/12: CIPHER_SBOX
	Bit 5/13: IDE_ISA_SBOX*
	Bit 6/14: IDE_DVD_SBOX*
	Bit 7/15: SFLA_SBOX
	Bit 16/24: SBOX_MBUS_W0*
	Bit 17/25: SBOX_MBUS_W1*
	Bit 18/26: SBOX_PCI_MASTER*
	Bit 19/27: SBOX_PCI_SLAVE*
	Bit 20/28: SBOX_CIPHER*
	Bit 21/29: SBOX_ISA*
	Bit 22/30: SBOX_DVD*
*/
static unsigned int sbox_reset_vals[2][4] = {
			{0x01012020, 0x02022020, 0x20200101, 0x20200202},
			{0x01014040, 0x02024040, 0x40400101, 0x40400202}};
static unsigned int sbox_unreset_vals[2][4] = {
			{0x01002000, 0x02002000, 0x20000100, 0x20000200},
			{0x01004000, 0x02004000, 0x40000100, 0x40000200}};
#endif

static void em86xx_mbus_reset(unsigned int regbase, int sbox)
{
	/* Clear MBUS transaction */
	int midx = (regbase - REG_BASE_HOST - MIF_w0_base) / 0x40;
	int sidx = sbox - SBOX_IDEFLASH;

#if defined(CONFIG_ARCH_MAMBO)
	if (((midx < 0) || (midx > 3)) || ((sidx < 0) || (sidx > 1))) {
		printk("MBUS reset: out of range, midx %d, sidx %d\n", midx, sidx);
		return;
	}	
#elif defined(CONFIG_ARCH_TANGO)
	if (((midx < 0) || (midx > 3)) || ((sidx < 0) || (sidx > 2))) {
		printk("MBUS reset: out of range, midx %d, sidx %d\n", midx, sidx);
		return;
	}	
#endif
        __raw_writel(sbox_reset_vals[sidx][midx], REG_BASE_HOST + SBOX_reset);
        __raw_writel(sbox_unreset_vals[sidx][midx], REG_BASE_HOST + SBOX_reset);
}

int em86xx_mbus_wait(unsigned int regbase, int sbox)
{
	int timeout = 10000, try = 1;

	do {
		if (timeout == 0)
			timeout = 0x00001000;

		while ((em86xx_mbus_inuse(regbase) != 0) && (--timeout != 0))
			;

		if (timeout == 0) {
			printk("MBUS timeout : MBUS CMD = %d, PB Automode = %08x\n", 
				__raw_readl(regbase + MIF_cmd_offset) & 0x7, 
				__raw_readl(REG_BASE_HOST + PB_automode_control));
			printk("MBUS registers : %08x %08x %08x %08x\n", 
				__raw_readl(regbase + MIF_add_offset),
				__raw_readl(regbase + MIF_cnt_offset),
				__raw_readl(regbase + MIF_add2_skip_offset),
				__raw_readl(regbase + MIF_cmd_offset));
			if (--try == 0) {
				printk("MBUS fails, resetting ..\n");
				em86xx_mbus_reset(regbase, sbox);

				/* If not able to reset, return 1, so the DMA
				   can be disabled accordingly  */
				return (em86xx_mbus_inuse(regbase) != 0) ? 1 : 0;
			}
		} else {
			unsigned int automode = __raw_readl(REG_BASE_HOST + PB_automode_control);
			if ((automode & 0xffff) != 0) {
				printk("MBUS : automode register is not empty : %08x\n",
					__raw_readl(REG_BASE_HOST + PB_automode_control));
				while ((automode = __raw_readl(REG_BASE_HOST + PB_automode_control)) & 0xffff)
					;
				printk("DONE\n");
			}
		}
	} while (timeout == 0);

    return 0;
}

#if 0
//
// PCI Master
//

void em86xx_pcimaster_setup_read(unsigned int addr, unsigned int count)
{
	__raw_writel(0, REG_BASE_HOST + PCI_master_read_reverse);
	__raw_writel(addr, REG_BASE_HOST + PCI_master_read_addr);
	__raw_writel(count, REG_BASE_HOST + PCI_master_read_counter);
}

void em86xx_pcimaster_start_read(int start)
{
	__raw_writel(start ? 1 : 0, REG_BASE_HOST + PCI_master_read_enable);
}

void em86xx_pcimaster_setup_write(unsigned int addr, unsigned int count)
{
	__raw_writel(0, REG_BASE_HOST + PCI_master_read_reverse);
	__raw_writel(addr, REG_BASE_HOST + PCI_master_write_addr);
	__raw_writel(count, REG_BASE_HOST + PCI_master_write_counter);
}

void em86xx_pcimaster_start_write(int start)
{
	__raw_writel(start ? 1 : 0, REG_BASE_HOST + PCI_master_write_enable);
}
#endif

//
// GPIO
//

int em86xx_gpio_read(int gpio)
{
	return (__raw_readl(REG_BASE_SYSTEM + SYS_gpio_data) >> gpio) & 1;
}

void em86xx_gpio_write(int gpio, int data)
{
	__raw_writel(data ? GPIO_DATA_SET(gpio) : GPIO_DATA_CLEAR(gpio),
		REG_BASE_SYSTEM + SYS_gpio_data);
}

void em86xx_gpio_setdirection(int gpio, int dir)
{
	__raw_writel(dir ? GPIO_DIR_OUTPUT(gpio) : GPIO_DIR_INPUT(gpio),
		REG_BASE_SYSTEM + SYS_gpio_dir);
}

#if defined(CONFIG_EM86XX_UART0_AS_GPIO_FULL) || defined(CONFIG_EM86XX_UART0_AS_GPIO_PARTIAL)
int em86xx_uart0_gpio_read(int gpio)
{
	return (__raw_readl(REG_BASE_CPU + CPU_uart0_gpio_data) >> gpio) & 1;
}

void em86xx_uart0_gpio_write(int gpio, int data)
{
	__raw_writel(data ? UART_GPIO_DATA_SET(gpio) : UART_GPIO_DATA_CLEAR(gpio),
		REG_BASE_CPU + CPU_uart0_gpio_data);
}

void em86xx_uart0_gpio_setdirection(int gpio, int dir)
{
	__raw_writel(dir ? UART_GPIO_DIR_OUTPUT(gpio) : UART_GPIO_DIR_INPUT(gpio),
		REG_BASE_CPU + CPU_uart0_gpio_dir);
}
#endif

#if defined(CONFIG_EM86XX_UART1_AS_GPIO_FULL) || defined(CONFIG_EM86XX_UART1_AS_GPIO_PARTIAL)
int em86xx_uart1_gpio_read(int gpio)
{
	return (__raw_readl(REG_BASE_CPU + CPU_uart1_gpio_data) >> gpio) & 1;
}

void em86xx_uart1_gpio_write(int gpio, int data)
{
	__raw_writel(data ? UART_GPIO_DATA_SET(gpio) : UART_GPIO_DATA_CLEAR(gpio),
		REG_BASE_CPU + CPU_uart1_gpio_data);
}

void em86xx_uart1_gpio_setdirection(int gpio, int dir)
{
	__raw_writel(dir ? UART_GPIO_DIR_OUTPUT(gpio) : UART_GPIO_DIR_INPUT(gpio),
		REG_BASE_CPU + CPU_uart1_gpio_dir);
}
#endif

EXPORT_SYMBOL(em86xx_gpio_read);
EXPORT_SYMBOL(em86xx_gpio_write);
EXPORT_SYMBOL(em86xx_gpio_setdirection);

#if defined(CONFIG_EM86XX_UART0_AS_GPIO_FULL) || defined(CONFIG_EM86XX_UART0_AS_GPIO_PARTIAL)
EXPORT_SYMBOL(em86xx_uart0_gpio_read);
EXPORT_SYMBOL(em86xx_uart0_gpio_write);
EXPORT_SYMBOL(em86xx_uart0_gpio_setdirection);
#endif

#if defined(CONFIG_EM86XX_UART1_AS_GPIO_FULL) || defined(CONFIG_EM86XX_UART1_AS_GPIO_PARTIAL)
EXPORT_SYMBOL(em86xx_uart1_gpio_read);
EXPORT_SYMBOL(em86xx_uart1_gpio_write);
EXPORT_SYMBOL(em86xx_uart1_gpio_setdirection);
#endif

EXPORT_SYMBOL(em86xx_clean_cache_data);
EXPORT_SYMBOL(em86xx_flush_cache_data);
EXPORT_SYMBOL(em86xx_mbus_alloc_dma);
EXPORT_SYMBOL(em86xx_mbus_setup_dma);
EXPORT_SYMBOL(em86xx_mbus_setup_dma_common);
EXPORT_SYMBOL(em86xx_mbus_setup_dma_double);
EXPORT_SYMBOL(em86xx_mbus_setup_dma_linear);
EXPORT_SYMBOL(em86xx_mbus_wait);

