/*
 * arch/arm/mach-em86xx/irq.c
 *
 * Copyright (C) 2003-2004 Sigma Designs, Inc
 *
 * by Ho Lee 01/27/2003
 */

#include <linux/kernel.h>
#include <linux/init.h>

#include <asm/hardware.h>
#include <asm/irq.h>
#include <asm/io.h>

#include <asm/mach/irq.h>

#ifdef CONFIG_PCI_EM86XX_HOST_FPGA
#include <asm/arch/pcifpga.h>
#endif

//
// IRQ information
// 

char *em86xx_get_irqinfo(char *p)
{
    static char *irqstat_str[] = {
        "EDGE_STATUS", 
	"EDGE_RAWSTAT", 
#if defined(CONFIG_ARCH_TANGO15) && (CONFIG_ARCH_TANGO15_REV > 1)
        "EDGE_STATUS_HI", 
	"EDGE_RAWSTAT_HI", 
#endif
	"EDGE_RISE", 
	"EDGE_FALL",
#if defined(CONFIG_ARCH_TANGO15) && (CONFIG_ARCH_TANGO15_REV > 1)
	"EDGE_RISE_HI", 
	"EDGE_FALL_HI",
#endif
        "IRQ_STATUS", 
	"IRQ_RAWSTAT", 
	"IRQ_ENABLE", 
#if defined(CONFIG_ARCH_TANGO15) && (CONFIG_ARCH_TANGO15_REV > 1)
        "IRQ_STATUS_HI", 
	"IRQ_RAWSTAT_HI", 
	"IRQ_ENABLE_HI", 
#endif
#ifdef CONFIG_SD_FIQ
        "FIQ_STATUS", 
	"FIQ_RAWSTAT", 
	"FIQ_ENABLE",
#if defined(CONFIG_ARCH_TANGO15) && (CONFIG_ARCH_TANGO15_REV > 1)
        "FIQ_STATUS_HI", 
	"FIQ_RAWSTAT_HI", 
	"FIQ_ENABLE_HI",
#endif
#endif
    };
    static unsigned int irqstat_reg[] = {
        REG_BASE_CPU + CPU_edge_status,
        REG_BASE_CPU + CPU_edge_rawstat,
#if defined(CONFIG_ARCH_TANGO15) && (CONFIG_ARCH_TANGO15_REV > 1)
        REG_BASE_CPU + CPU_edge_status_hi,
        REG_BASE_CPU + CPU_edge_rawstat_hi,
#endif
#if defined(CONFIG_ARCH_MAMBO)
        // Mambo bug
        REG_BASE_CPU + CPU_edge_config_fall,
        REG_BASE_CPU + CPU_edge_config_rise,
#elif defined(CONFIG_ARCH_TANGO)
        REG_BASE_CPU + CPU_edge_config_rise,
        REG_BASE_CPU + CPU_edge_config_fall,
#if defined(CONFIG_ARCH_TANGO15) && (CONFIG_ARCH_TANGO15_REV > 1)
        REG_BASE_CPU + CPU_edge_config_rise_hi,
        REG_BASE_CPU + CPU_edge_config_fall_hi,
#endif
#else
#error Unknown Architecture
#endif
        REG_BASE_CPU + CPU_irq_status,
        REG_BASE_CPU + CPU_irq_rawstat,
        REG_BASE_CPU + CPU_irq_enableset,
#if defined(CONFIG_ARCH_TANGO15) && (CONFIG_ARCH_TANGO15_REV > 1)
        REG_BASE_CPU + CPU_irq_status_hi,
        REG_BASE_CPU + CPU_irq_rawstat_hi,
        REG_BASE_CPU + CPU_irq_enableset_hi,
#endif
#ifdef CONFIG_SD_FIQ
        REG_BASE_CPU + CPU_fiq_status,
        REG_BASE_CPU + CPU_fiq_rawstat,
        REG_BASE_CPU + CPU_fiq_enableset,
#if defined(CONFIG_ARCH_TANGO15) && (CONFIG_ARCH_TANGO15_REV > 1)
        REG_BASE_CPU + CPU_fiq_status_hi,
        REG_BASE_CPU + CPU_fiq_rawstat_hi,
        REG_BASE_CPU + CPU_fiq_enableset_hi,
#endif
#endif
        0
    };
        
    int i;
    unsigned int irqstat;

    p += sprintf(p, "\nEM86XX IRQ Registers :\n");

    for (i = 0; irqstat_reg[i] > 0; ++i) {
        irqstat = __raw_readl(irqstat_reg[i]);
        p += sprintf(p, "  %08x (%s) : %08x\n", irqstat_reg[i], irqstat_str[i], irqstat);
    }
    
    p += sprintf(p, "\nEM86XX IRQ/GPIO mapping :\n");
    irqstat = __raw_readl(REG_BASE_SYSTEM + SYS_gpio_int);

    for (i = 0; i < 4; ++i, irqstat >>= 8) 
        p += sprintf(p, "  IRQ %d = GPIO %d\n", IRQ_GPIO0 + i, irqstat & 0xff);

    return p;
}

//
// IRQ common
//

void em86xx_enter_irq(unsigned int irq)
{
    // if the interrupt is detected by edge detector not level detector 
    // set the edge rawstat bit to clear edge status bit (also irq status bit)
    // so the edge detector can trigger interrupt at next edge change
#if defined(CONFIG_ARCH_TANGO15) && (CONFIG_ARCH_TANGO15_REV > 1)
    if ((IRQMASKOF(irq) & (IRQMASK_RISINGEDGE | IRQMASK_FALLINGEDGE)) != 0) {
        __raw_writel((u32)(IRQMASKOF(irq) & 0xffffffff), REG_BASE_CPU + CPU_edge_rawstat);
        __raw_writel((u32)((IRQMASKOF(irq)>>32) & 0xffffffff), REG_BASE_CPU + CPU_edge_rawstat_hi);
    }
#else
    if ((IRQMASKOF(irq) & (IRQMASK_RISINGEDGE | IRQMASK_FALLINGEDGE)) != 0)
        __raw_writel(IRQMASKOF(irq), REG_BASE_CPU + CPU_edge_rawstat);
#endif
}

void em86xx_exit_irq(unsigned int irq)
{
    // by Ho Lee 12/10/2003
    // Edge detector triggers interrupt at rising edge even it has to 
    // trigger at only falling edge. Also it triggers falling edge when it
    // has to trigger rising edge. This code checks this false interrupt
    // report and clears interrupt flag in this case
    // configuration = falling edge, irq_status = 1, edge_status = 1, edge_rawstat = 1
    //               = rising edge, irq_status = 1, edge_status = 1, edge_rawstat = 0
    // Only applied to external devices' interrupts
#if defined(CONFIG_ARCH_TANGO15) && (CONFIG_ARCH_TANGO15_REV > 1)
    if ((IRQMASKOF(irq) & (IRQMASK_EXTERNALIRQS & (IRQMASK_FALLINGEDGE | IRQMASK_RISINGEDGE) &
          ((((u64)__raw_readl(REG_BASE_CPU + CPU_edge_status_hi))<<32) | 
	   __raw_readl(REG_BASE_CPU + CPU_edge_status)))) != 0) {
        u64 falling = IRQMASKOF(irq) & IRQMASK_FALLINGEDGE;
        u64 rawstat;
#if 0
        // to debug false interrupt report
        rawstat = ((((u64)__raw_readl(REG_BASE_CPU + CPU_edge_rawstat_hi))<<32) |
		__raw_readl(REG_BASE_CPU + CPU_edge_rawstat)) & IRQMASKOF(irq);

        if ((falling && rawstat != 0) || (!falling && rawstat == 0))
            printk("interrupt at %s edge\n", falling ? "rising" : "falling");
#endif
        // clear interrupt first
        __raw_writel((u32)(IRQMASKOF(irq) & 0xffffffff), REG_BASE_CPU + CPU_edge_rawstat);
        __raw_writel((u32)((IRQMASKOF(irq)>>32) & 0xffffffff), REG_BASE_CPU + CPU_edge_rawstat_hi);

        // set edge status again if line status is low
        // edge_status bit can not be cleared, but only be set. 
        // so setting a bit is safe
        rawstat = ((((u64)__raw_readl(REG_BASE_CPU + CPU_edge_rawstat_hi))<<32) |
		__raw_readl(REG_BASE_CPU + CPU_edge_rawstat)) & IRQMASKOF(irq);
        if ((falling && rawstat == 0) || (!falling && rawstat != 0)) {
            __raw_writel((u32)(IRQMASKOF(irq) & 0xffffffff), REG_BASE_CPU + CPU_edge_status);
            __raw_writel((u32)((IRQMASKOF(irq)>>32) & 0xffffffff), REG_BASE_CPU + CPU_edge_status_hi);
	}
    }
#else
    if ((IRQMASKOF(irq) & (IRQMASK_EXTERNALIRQS & (IRQMASK_FALLINGEDGE | IRQMASK_RISINGEDGE) &
          __raw_readl(REG_BASE_CPU + CPU_edge_status))) != 0) {
        int falling = IRQMASKOF(irq) & IRQMASK_FALLINGEDGE;
        int rawstat;
#if 0
        // to debug false interrupt report
        rawstat = __raw_readl(REG_BASE_CPU + CPU_edge_rawstat) & IRQMASKOF(irq);
        if ((falling && rawstat != 0) || (!falling && rawstat == 0))
            printk("interrupt at %s edge\n", falling ? "rising" : "falling");
#endif
        // clear interrupt first
        __raw_writel(IRQMASKOF(irq), REG_BASE_CPU + CPU_edge_rawstat);
        // set edge status again if line status is low
        // edge_status bit can not be cleared, but only be set. 
        // so setting a bit is safe
        rawstat = __raw_readl(REG_BASE_CPU + CPU_edge_rawstat) & IRQMASKOF(irq);
        if ((falling && rawstat == 0) || (!falling && rawstat != 0))
            __raw_writel(IRQMASKOF(irq), REG_BASE_CPU + CPU_edge_status);
    }
#endif
}

//
// IRQ handling
//

void em86xx_mask_irq(unsigned int irq)
{
    // disable IRQ
#if defined(CONFIG_ARCH_TANGO15) && (CONFIG_ARCH_TANGO15_REV > 1)
    __raw_writel((u32)(IRQMASKOF(irq) & 0xffffffff), REG_BASE_CPU + CPU_irq_enableclr);
    __raw_writel((u32)((IRQMASKOF(irq)>>32) & 0xffffffff), REG_BASE_CPU + CPU_irq_enableclr_hi);
#else
    __raw_writel(IRQMASKOF(irq), REG_BASE_CPU + CPU_irq_enableclr);
#endif
}

void em86xx_unmask_irq(unsigned int irq)
{
    // enable IRQ
#if defined(CONFIG_ARCH_TANGO15) && (CONFIG_ARCH_TANGO15_REV > 1)
    __raw_writel((u32)(IRQMASKOF(irq) & 0xffffffff), REG_BASE_CPU + CPU_irq_enableset);
    __raw_writel((u32)((IRQMASKOF(irq)>>32) & 0xffffffff), REG_BASE_CPU + CPU_irq_enableset_hi);
#else
    __raw_writel(IRQMASKOF(irq), REG_BASE_CPU + CPU_irq_enableset);
#endif
}

void em86xx_wait_irq(unsigned int irq)
{
#if defined(CONFIG_ARCH_TANGO15) && (CONFIG_ARCH_TANGO15_REV > 1)
    u64 mask = IRQMASKOF(irq);
    while ((((((u64)__raw_readl(REG_BASE_CPU + CPU_irq_rawstat_hi))<<32) | 
	    __raw_readl(REG_BASE_CPU + CPU_irq_rawstat)) & mask) == 0)
        ;
#else
    unsigned int mask = IRQMASKOF(irq);
    while ((__raw_readl(REG_BASE_CPU + CPU_irq_rawstat) & mask) == 0)
        ;
#endif
}

int em86xx_check_lost_irq(unsigned int irq)
{
#ifdef CONFIG_ARCH_MAMBO
    // by Ho Lee 09/12/2003
    // level detector issues interrupt only when the level is high
    // for the active low interrupts, edge detector is used instead
    // and there is possiblity of losing interrupt when many devices
    // drive the same interrupt line (O/D). PCI interrupt is the example. 
    // For such interrupts, check the level after processing the 
    // interrupt and run the interrupt handler again if the level is 
    // still low. It may cause the infinite loop when the device malfunctions
    // and keeps the interrupt line low

    unsigned int irqmask = IRQMASKOF(irq);

    if (irqmask & IRQMASK_LOWLEVEL) {
        if ((__raw_readl(REG_BASE_CPU + CPU_edge_rawstat) & irqmask) == 0)
            return 1;
    }
#endif

    return 0;
}

// 
// FIQ handling
//

void em86xx_mask_fiq(unsigned int fiq)
{
    // disable FIQ
#if defined(CONFIG_ARCH_TANGO15) && (CONFIG_ARCH_TANGO15_REV > 1)
    __raw_writel((u32)(IRQMASKOF(fiq) & 0xffffffff), REG_BASE_CPU + CPU_fiq_enableclr);
    __raw_writel((u32)((IRQMASKOF(fiq)>>32) & 0xffffffff), REG_BASE_CPU + CPU_fiq_enableclr_hi);
#else
    __raw_writel(IRQMASKOF(fiq), REG_BASE_CPU + CPU_fiq_enableclr);
#endif
}

void em86xx_unmask_fiq(unsigned int fiq)
{
    // enable FIQ
#if defined(CONFIG_ARCH_TANGO15) && (CONFIG_ARCH_TANGO15_REV > 1)
    __raw_writel((u32)(IRQMASKOF(fiq) & 0xffffffff), REG_BASE_CPU + CPU_fiq_enableset);
    __raw_writel((u32)((IRQMASKOF(fiq)>>32) & 0xffffffff), REG_BASE_CPU + CPU_fiq_enableset_hi);
#else
    __raw_writel(IRQMASKOF(fiq), REG_BASE_CPU + CPU_fiq_enableset);
#endif
}

#if defined(CONFIG_ARCH_TANGO15) && (CONFIG_ARCH_TANGO15_REV > 1)
u64 em86xx_getfiqnr(void)
{
    return ((((u64)__raw_readl(REG_BASE_CPU + CPU_fiq_status_hi))<<32) |
	    __raw_readl(REG_BASE_CPU + CPU_fiq_status));
}
#else
unsigned int em86xx_getfiqnr(void)
{
    return __raw_readl(REG_BASE_CPU + CPU_fiq_status);
}
#endif

//
// Software Interrupt handling
//

int em86xx_softirq_isset(int irq)
{
    return __raw_readl(REG_BASE_CPU + CPU_irq_softset) & (1 << irq);
}

void em86xx_softirq_set(int irq)
{
    __raw_writel(1 << irq, REG_BASE_CPU + CPU_irq_softset);
}

void em86xx_softirq_clr(int irq)
{
    __raw_writel(1 << irq, REG_BASE_CPU + CPU_irq_softclr);
}

void em86xx_irq_clr(int irq)
{
#if defined(CONFIG_ARCH_TANGO15) && (CONFIG_ARCH_TANGO15_REV > 1)
    __raw_writel((u32)(IRQMASKOF(irq) & 0xffffffff), REG_BASE_CPU + CPU_edge_rawstat);
    __raw_writel((u32)((IRQMASKOF(irq)>>32) & 0xffffffff), REG_BASE_CPU + CPU_edge_rawstat_hi);
#else
    __raw_writel(IRQMASKOF(irq), REG_BASE_CPU + CPU_edge_rawstat);
#endif
}

//
// Initialization
//

void __init em86xx_init_irq(void)
{
    unsigned int i;
#ifdef	CONFIG_ALPHA_STAND_ALONE_PCI_INTERRUPTS	
	/*
		GPIO 11 is for IO expander interrupt,
		but this causes SCART output hangs thesystem.
		We disable 11 as interrupt trigger until Sigma fixes the bugs.
	 */
	unsigned int irqstat=(8|(9<<8)|(10<<16));
	em86xx_gpio_setdirection(8,0);
	em86xx_gpio_setdirection(9,0);
	em86xx_gpio_setdirection(10,0);
	__raw_writel(irqstat,REG_BASE_SYSTEM + SYS_gpio_int);
#endif	// CONFIG_ALPHA_STAND_ALONE_PCI_INTERRUPTS

    for (i = 0; i < NR_IRQS; i++) {
        if (((1 << i) & IRQMASK_VALID) != 0) {
            irq_desc[i].valid = 1;
            irq_desc[i].probe_ok = 1;
            irq_desc[i].mask_ack = em86xx_mask_irq;
            irq_desc[i].mask = em86xx_mask_irq;
            irq_desc[i].unmask = em86xx_unmask_irq;
        }
    }

    /* Setup edge detector */
#if defined(CONFIG_ARCH_MAMBO)
    __raw_writel(IRQMASK_RISINGEDGE, REG_BASE_CPU + CPU_edge_config_rise);
    __raw_writel(IRQMASK_FALLINGEDGE, REG_BASE_CPU + CPU_edge_config_fall);
#elif defined(CONFIG_ARCH_TANGO10)
    __raw_writel(IRQMASK_RISINGEDGE | IRQMASK_LOWLEVEL, REG_BASE_CPU + CPU_edge_config_rise);
    __raw_writel(IRQMASK_FALLINGEDGE | IRQMASK_LOWLEVEL, REG_BASE_CPU + CPU_edge_config_fall);
#elif defined(CONFIG_ARCH_TANGO15)
#if (CONFIG_ARCH_TANGO15_REV > 1)
    __raw_writel((u32)((IRQMASK_RISINGEDGE | IRQMASK_LOWLEVEL) & 0xffffffff), REG_BASE_CPU + CPU_edge_config_rise);
    __raw_writel((u32)(((IRQMASK_RISINGEDGE | IRQMASK_LOWLEVEL)>>32) & 0xffffffff), REG_BASE_CPU + CPU_edge_config_rise_hi);
    __raw_writel((u32)((IRQMASK_FALLINGEDGE | IRQMASK_LOWLEVEL) & 0xffffffff), REG_BASE_CPU + CPU_edge_config_fall);
    __raw_writel((u32)(((IRQMASK_FALLINGEDGE | IRQMASK_LOWLEVEL)>>32) & 0xffffffff), REG_BASE_CPU + CPU_edge_config_fall_hi);
#else
    __raw_writel(IRQMASK_RISINGEDGE | IRQMASK_LOWLEVEL, REG_BASE_CPU + CPU_edge_config_rise);
    __raw_writel(IRQMASK_FALLINGEDGE | IRQMASK_LOWLEVEL, REG_BASE_CPU + CPU_edge_config_fall);
#endif
#endif

#if defined(CONFIG_ARCH_TANGO15) && (CONFIG_ARCH_TANGO15_REV > 1)
    /* Clear edge status flags */
    __raw_writel(__raw_readl(REG_BASE_CPU + CPU_edge_rawstat), REG_BASE_CPU + CPU_edge_rawstat);
    __raw_writel(__raw_readl(REG_BASE_CPU + CPU_edge_rawstat_hi), REG_BASE_CPU + CPU_edge_rawstat_hi);

    /* Disable all interrupts initially. */
    __raw_writel(0xffffffff, REG_BASE_CPU + CPU_irq_enableclr);
    __raw_writel(0xffffffff, REG_BASE_CPU + CPU_irq_enableclr_hi);
#else
    /* Clear edge status flags */
    __raw_writel(__raw_readl(REG_BASE_CPU + CPU_edge_rawstat), REG_BASE_CPU + CPU_edge_rawstat);

    /* Disable all interrupts initially. */
    __raw_writel(0xffffffff, REG_BASE_CPU + CPU_irq_enableclr);
#endif
}

asmlinkage u32 em86xx_getirqnr(void)
{
	static u32 em86xx_lastscan_irqnr = 0;
	u32 softset = __raw_readl(REG_BASE_CPU + CPU_irq_softset);
	u32 irqnr;

#if defined(CONFIG_ARCH_TANGO15) && (CONFIG_ARCH_TANGO15_REV > 1)
	u64 irqstat = (((u64)__raw_readl(REG_BASE_CPU + CPU_irq_status_hi))<<32) |
			__raw_readl(REG_BASE_CPU + CPU_irq_status);
#else
	u32 irqstat = __raw_readl(REG_BASE_CPU + CPU_irq_status);
#endif

	for (irqnr = em86xx_lastscan_irqnr, irqstat >>= irqnr; irqnr != NR_IRQS; ++irqnr, irqstat >>= 1) {
		if (irqstat & 1) {
			if ((irqnr == 0) && ((softset & SOFTIRQMASK_VALID) == 0))
				continue;
			em86xx_lastscan_irqnr = irqnr + 1;
			return irqnr;
		}
	}
	em86xx_lastscan_irqnr = 0;
	return irqnr;
}

