/*
 * arch/arm/mach-em86xx/pciem86xx.c
 *
 * Copyright (C) 2003-2004 Sigma Designs, Inc
 *
 * by Ho Lee 02/11/2003
 */

#include <linux/config.h>

#ifdef CONFIG_PCI_EM86XX_HOST_EM86XX

#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/sched.h>
#include <asm/hardware.h>
#include <asm/io.h>

#include <linux/pci.h>
#include <asm/mach/pci.h>
#include <asm/system.h>
#include <asm/irq.h>

static int em86xx_pci_read_config_byte(struct pci_dev *dev, int where, u8 *value);
static int em86xx_pci_read_config_word(struct pci_dev *dev, int where, u16 *value);
static int em86xx_pci_read_config_dword(struct pci_dev *dev, int where, u32 *value);
static int em86xx_pci_write_config_byte(struct pci_dev *dev, int where, u8 value);
static int em86xx_pci_write_config_word(struct pci_dev *dev, int where, u16 value);
static int em86xx_pci_write_config_dword(struct pci_dev *dev, int where, u32 value);

static struct pci_ops em86xx_pci_pci_ops = {
    em86xx_pci_read_config_byte,
    em86xx_pci_read_config_word,
    em86xx_pci_read_config_dword,
    em86xx_pci_write_config_byte,
    em86xx_pci_write_config_word,
    em86xx_pci_write_config_dword,
};

static void em86xx_pci_init(void *sysdata);
static int em86xx_pci_map_irq(struct pci_dev *dev, u8 slot, u8 pin);

struct hw_pci em86xx_pci = {
    init : em86xx_pci_init,
    setup_resources : NULL, 
    swizzle : no_swizzle, 
    map_irq : em86xx_pci_map_irq,
    mem_offset : MEMORY_BASE_PCI_MEMORY,    // offset between address on the GBUS and address on the PCI bus (PCI memory)
    io_offset : MEMORY_BASE_PCI_CONFIG,     // offset between address on the GBUS and address on the PCI bus (PCI I/O)
};

// bus address : PCI bus address
// phy address : low address 
unsigned int g_pcimem_busaddr = PCIBIOS_MIN_MEM_EM86XX - MEMORY_BASE_PCI_MEMORY;
unsigned int g_pcimem_phyaddr = em86xx_to_ncaddr(DRAM_BASE);
unsigned int g_pcimem_phyaddr_end = 0;

EXPORT_SYMBOL(g_pcimem_busaddr);
EXPORT_SYMBOL(g_pcimem_phyaddr);
EXPORT_SYMBOL(g_pcimem_phyaddr_end);

//
// PCI configuration area I/O
// 

// According to EM86XX PCI host spec, to access PCI configuration area of device
// it is needed to setup idsel bits in HOST_REG2 register. And the idsel bits
// are only two bits wide. So it supports only 4 PCI slot, and doesn't support 
// PCI bridge at all. There is no way to access PCI configuration area over
// the PCI bridge -- Mambo only 
//
// DEVFN : Linux uses devfn to specify device. 
//   PCI_SLOT(devfn) = idsel
//   PCI_FUNC(devfn) = function
//   generates type 0 configuration cycle

#if defined(CONFIG_ARCH_MAMBO) || defined(CONFIG_ARCH_TANGO10)
#define PCIEM86XX_IDSEL_MAX         0x04
#elif defined(CONFIG_ARCH_TANGO15)
#define PCIEM86XX_IDSEL_MAX         0x05
#endif

static int g_last_idsel = -1;
static int g_idsel_exist[PCIEM86XX_IDSEL_MAX];
static unsigned char g_pci_header_type[PCIEM86XX_IDSEL_MAX];

#if defined(CONFIG_ARCH_MAMBO)

#define PCIEM86XX_CONFIG_START                                              \
    unsigned long flags;                                                    \
    int idsel = PCI_SLOT(dev->devfn), func = PCI_FUNC(dev->devfn);          \
                                                                            \
    if (dev->bus->number != 0) {					    \
	printk("PCI: error accessing PCI config on bus %d.\n", dev->bus->number); \
        return PCIBIOS_DEVICE_NOT_FOUND;                                    \
    }									    \
    if (idsel >= PCIEM86XX_IDSEL_MAX || !g_idsel_exist[idsel])              \
        return PCIBIOS_DEVICE_NOT_FOUND;                                    \
                                                                            \
    local_irq_save(flags);                                                  \
                                                                            \
    if (g_last_idsel != idsel) {                                            \
        g_last_idsel = idsel;                                               \
        __raw_writew((idsel << 8), REG_BASE_HOST + PCI_host_host_reg2);     \
    }

#define PCIEM86XX_CONFIG_END                                                \
    local_irq_restore(flags);                                               \
    return PCIBIOS_SUCCESSFUL;                                          

#define PCIEM86XX_CONFIG_READ(x, type, size)                                \
    int em86xx_pci_read_config_##type(struct pci_dev *dev, int where, u##size *value) \
    {                                                                       \
        PCIEM86XX_CONFIG_START                                              \
        *value = __raw_read##x(MEMORY_BASE_PCI_CONFIG | (func << 8) | where);\
        if (func && where == 0) {                                           \
            unsigned int data = __raw_readb(REG_BASE_HOST + PCI_host_host_reg2 + 3) >> 1;   \
            if (data) { /* bus fault */                                     \
                __raw_writeb(0x01, REG_BASE_HOST + PCI_host_host_reg2 + 3); \
                __raw_writeb(0x00, REG_BASE_HOST + PCI_host_host_reg2 + 3); \
                return PCIBIOS_DEVICE_NOT_FOUND;                            \
            }                                                               \
        }                                                                   \
        PCIEM86XX_CONFIG_END                                                \
    } 

#define PCIEM86XX_CONFIG_WRITE(x, type, size)                               \
    int em86xx_pci_write_config_##type(struct pci_dev *dev, int where, u##size value) \
    {                                                                       \
        PCIEM86XX_CONFIG_START                                              \
        __raw_write##x(value, MEMORY_BASE_PCI_CONFIG | (func << 8) | where);    \
        PCIEM86XX_CONFIG_END                                                    \
    }

#elif defined(CONFIG_ARCH_TANGO)

// Tango supports setting IDSEL through ADDR/DATA[12:11]. If these bits are 0,
// then pci_host_reg2[9:8] decides IDSEL, otherwise these bits decide IDSEL.
// To make use of it, initially set pci_host_reg2[9:8] to 0, and never change
// these bits. And set ADDR/DATA[12:11] whenever kernel initiates configuration
// cycle. 

#define PCIEM86XX_CONFIG_START                                              \
    unsigned long flags;                                                    \
    int idsel = PCI_SLOT(dev->devfn), func = PCI_FUNC(dev->devfn);          \
                                                                            \
    if (idsel >= PCIEM86XX_IDSEL_MAX || !g_idsel_exist[idsel])              \
        return PCIBIOS_DEVICE_NOT_FOUND;                                    \
                                                                            \
    local_irq_save(flags);

#define PCIEM86XX_CONFIG_END                                                \
    local_irq_restore(flags);                                               \
    return PCIBIOS_SUCCESSFUL;                                          

#define PCIEM86XX_CONFIG_READ(x, type, size)                                \
    int em86xx_pci_read_config_##type(struct pci_dev *dev, int where, u##size *value) \
    {                                                                       \
        PCIEM86XX_CONFIG_START                                              \
        *value = __raw_read##x(MEMORY_BASE_PCI_CONFIG | (func << 8) | where | (idsel << 11) | (dev->bus->number << 16));\
        if (where == 0) {                                                   \
            unsigned int data = __raw_readb(REG_BASE_HOST + PCI_host_host_reg2 + 3) >> 1;   \
            if (data) { /* bus fault */                                     \
                __raw_writeb(0x01, REG_BASE_HOST + PCI_host_host_reg2 + 3); \
                __raw_writeb(0x00, REG_BASE_HOST + PCI_host_host_reg2 + 3); \
                return PCIBIOS_DEVICE_NOT_FOUND;                            \
            }                                                               \
        }                                                                   \
        PCIEM86XX_CONFIG_END                                                \
    } 

#define PCIEM86XX_CONFIG_WRITE(x, type, size)                               \
    int em86xx_pci_write_config_##type(struct pci_dev *dev, int where, u##size value) \
    {                                                                       \
        PCIEM86XX_CONFIG_START                                              \
        __raw_write##x(value, MEMORY_BASE_PCI_CONFIG | (func << 8) | where | (idsel << 11) | (dev->bus->number << 16));    \
        PCIEM86XX_CONFIG_END                                                    \
    }

#endif

PCIEM86XX_CONFIG_READ(b, byte, 8)
PCIEM86XX_CONFIG_READ(w, word, 16)
PCIEM86XX_CONFIG_READ(l, dword, 32)

PCIEM86XX_CONFIG_WRITE(b, byte, 8)
PCIEM86XX_CONFIG_WRITE(w, word, 16)
PCIEM86XX_CONFIG_WRITE(l, dword, 32)

//
// Initialization
//

static unsigned int em86xx_pci_select(int idsel);
static void em86xx_pci_busfault_interrupt(int irq, void *devinfo, struct pt_regs *regs);
// Debugging purpose 
//static void em86xx_pci_devirq(int irq, void *devinfo, struct pt_regs *regs);

#if (defined(CONFIG_ARCH_MAMBO) && (CONFIG_ARCH_MAMBO_REV == 1))
#define PCIEM86XX_BUSMASTERNUMBER       3
#define PCIEM86XX_ARBITER_CYCLE         0xff
#elif (defined(CONFIG_ARCH_MAMBO) && (CONFIG_ARCH_MAMBO_REV >= 2)) || defined(CONFIG_ARCH_TANGO)
#define PCIEM86XX_ARBITER_LEVEL         0x00
#define PCIEM86XX_ARBITER_GRANTTIMEOUT  0x10101010
#endif

void em86xx_pci_init(void *sysdata)
{
    int i, idsel;
    unsigned int pciid = 0;
    
    // EM86XX is Host
    __raw_writel(1, REG_BASE_HOST + PCI_chip_is_host);

    // PCI HOST registers initialization
#if (defined(CONFIG_ARCH_MAMBO) && (CONFIG_ARCH_MAMBO_REV == 1))
    // HOST_REG1 : 
    //   [31:16] : # of PCI retry cycle = 0xff (default)
    //   [15] : arbitration mode = 0 (used time slot) : now only time-slot mode is supported
    //   [14:13] : number of masters to arbitrate = 3 (all device can be master) or 1
    //   [12:0] : # of arbitration clock = 0xff (default)
    __raw_writel(0x00ff0000 | (PCIEM86XX_BUSMASTERNUMBER << 13) | PCIEM86XX_ARBITER_CYCLE, 
        REG_BASE_HOST + PCI_host_host_reg1);
#elif (defined(CONFIG_ARCH_MAMBO) && (CONFIG_ARCH_MAMBO_REV >= 2)) || defined(CONFIG_ARCH_TANGO)
    // HOST_REG1 : 
    //   [31:16] : # of PCI retry cycle = 0xffff (default = 0xff)
    //   [8] : host Super Request = 0
    //   [3:0] arbitration level = 0x00 (Level 1) 
    __raw_writel(0xffff0000 | PCIEM86XX_ARBITER_LEVEL, 
        REG_BASE_HOST + PCI_host_host_reg1);

    // setup Additional PCI control registers introduced by Rev B

    // PCI_CTRL1 : 
    //   [17] : enable "Memory Read Multiple" and "Memory Read Line"
    //   [16] [7:0] : enable "prefetch" for PCI slave regions 2..7
#ifdef CONFIG_ARCH_TANGO
    //   [17] : Always enabled
    //   [18] : additional bit for Tango. Long PCI memory read burst
//    __raw_writel(0x000700fc, REG_BASE_HOST + PCI_pcictrl_reg1);
    __raw_writel(0x00030000, REG_BASE_HOST + PCI_pcictrl_reg1);
#else
    __raw_writel(0x000300fc, REG_BASE_HOST + PCI_pcictrl_reg1);
#endif
    // PCI_CTRL2 :
    //   [18] : fast back-to-back capable = 0 (default)
    //   [17] : read FIFO level = 1 (8 level deep, default)
    //   [16] : discard timer enable = 1 (default)
    //   [15:8] : subs latency = 0x06 (default = 0x08)
    //   [7:0] : initial latency = 0x0d (default = 0x0b)
    __raw_writel(0x0003060d, REG_BASE_HOST + PCI_pcictrl_reg2);
    // PCI_CTRL3 : 
    //   [16] : slave abort clear = 0
    //   [10:8] : abort interrupt enable = 0 (default)
    //   [2:0] : abort status = 0
    __raw_writel(0x00000000, REG_BASE_HOST + PCI_pcictrl_reg3);
#endif

    // clear PCI bus fault
    if ((__raw_readb(REG_BASE_HOST + PCI_host_host_reg2 + 3) >> 1) != 0) {
        __raw_writeb(0x01, REG_BASE_HOST + PCI_host_host_reg2 + 3);
        __raw_writeb(0x00, REG_BASE_HOST + PCI_host_host_reg2 + 3);
    }

#if defined(CONFIG_ARCH_MAMBO)
    // detect each agent
    for (idsel = 1; idsel < PCIEM86XX_IDSEL_MAX; ++idsel)
        g_idsel_exist[idsel] = em86xx_pci_select(idsel) ? 1 : 0;
#elif defined(CONFIG_ARCH_TANGO)
    for (idsel = 1; idsel < PCIEM86XX_IDSEL_MAX; ++idsel)
        g_idsel_exist[idsel] = 1;
#endif

    // Scan PCI Bus
    pci_scan_bus(0, &em86xx_pci_pci_ops, sysdata);

#ifdef CONFIG_SD_EM86XX_PCIDMA
#if (defined(CONFIG_ARCH_MAMBO) && (CONFIG_ARCH_MAMBO_REV >= 2)) || defined(CONFIG_ARCH_TANGO)
    // grant timeout
    __raw_writel(PCIEM86XX_ARBITER_GRANTTIMEOUT, REG_BASE_HOST + PCI_host_host_reg3);

    // initialize arbiter
    __raw_writel(0, REG_BASE_HOST + PCI_host_host_reg4);
#endif

    // Initialize IDSEL 0 - EM86XX as a PCI slave
    for (i = 1; i <= 3; ++i) {
        if ((pciid = em86xx_pci_select(0)) != 0)
            break;
        printk("PCI: Probing EM86XX at IDSEL 0, retry = %d\n", i);
    }

    if (pciid == 0) {
        printk("PCI: Can't initialize EM86XX as a PCI slave\n");
    } else {
        unsigned int memsize, regsize, membase;
        
        // by Ho Lee 03/03/2004
        // set PCI memory size to maximum, so the PCI memory will cover the 
        // whole memory if the total DRAM size is smaller than 96MB. 
        // maximum = 128MB => 16MB per region => DMA memory = 16 * 6 = 96MB
        // But the current code doesn't care 2nd DRAM controller. 
        __raw_writeb(0x07, REG_BASE_HOST + PCI_devcfg_reg3);

        // pci command 
        __raw_writew(PCI_COMMAND_MEMORY | PCI_COMMAND_MASTER, MEMORY_BASE_PCI_CONFIG + PCI_COMMAND);
        // base address 0
        __raw_writel(PCIBIOS_MIN_MEM_EM86XX - MEMORY_BASE_PCI_MEMORY, MEMORY_BASE_PCI_CONFIG + PCI_BASE_ADDRESS_0);

        // PCI slave access
        // region 0 (R) : Configuration area 
        // region 1 : RISC memory 
        __raw_writel(MEMORY_BASE_RISC, REG_BASE_HOST + PCI_region_base + (1 * 4));

        // region 2 - : mapped to DRAM starting from DRAM_BASE
        memsize = __raw_readl(REG_BASE_HOST + PCI_devcfg_reg3) & 0x07;
        memsize = 1 << memsize;
        regsize = (memsize << 20) >> 3; // memory size / 8
        g_pcimem_busaddr += regsize * 2;
        printk("PCI: Configured EM86XX as a PCI slave with %dMB PCI memory\n", memsize);
#ifdef CONFIG_SD_EM86XX_PCIDMA_MATCHADDR
        membase = DRAM_BASE + regsize * 2;
#else
        membase = DRAM_BASE + 0x00000000;
#endif
        membase = em86xx_to_ncaddr(membase);
        g_pcimem_phyaddr = membase;
        g_pcimem_phyaddr_end = g_pcimem_phyaddr + em86xx_kmemsize;
        for (i = 2; i < 8; ++i) {
            // PCI region base registers should contain low address
            __raw_writel(EM86XX_DRAM_C2NC(membase), REG_BASE_HOST + PCI_region_base + (i * 4));
            membase += regsize;
        }
        if (g_pcimem_phyaddr_end > membase)
            g_pcimem_phyaddr_end = membase;
        printk("PCI: Each Region size is %dKB\n", regsize >> 10);
        printk("PCI: Reserved memory from 0x%08x to 0x%08x for DMA and mapped to 0x%08x\n", 
            g_pcimem_phyaddr, g_pcimem_phyaddr_end, g_pcimem_busaddr);
    }
#endif

    // PCI Bus Fault error handler
    request_irq(IRQ_PCILOCALBUSFAULT, em86xx_pci_busfault_interrupt, SA_INTERRUPT, "EM86XX PCI host", NULL);

    // Debugging purpose
    // request_irq(IRQ_PCIA, em86xx_pci_devirq, SA_SHIRQ, "EM86XX PCI", (void *) IRQ_PCIA);
}

// return 0 if device not found
// return vendorid:deviceid if device found
unsigned int em86xx_pci_select(int idsel)
{
    unsigned int data, pciid;

    g_last_idsel = idsel;
    __raw_writew((idsel << 8), REG_BASE_HOST + PCI_host_host_reg2);
    pciid = __raw_readl(MEMORY_BASE_PCI_CONFIG);
    data = __raw_readb(REG_BASE_HOST + PCI_host_host_reg2 + 3) >> 1;
        
    if (data == 0) {
        return pciid;
    } else {
        // clears PCI bus fault flags
        __raw_writeb(0x01, REG_BASE_HOST + PCI_host_host_reg2 + 3);
        __raw_writeb(0x00, REG_BASE_HOST + PCI_host_host_reg2 + 3);
        return 0;
    }
}

// Debugging purpose
//void em86xx_pci_devirq(int irq, void *devinfo, struct pt_regs *regs)
//{
    // printk("[%c]", irq - IRQ_PCIA + 'A');
//}

void em86xx_pci_busfault_interrupt(int irq, void *devinfo, struct pt_regs *regs)
{
#ifdef CONFIG_SD_CHECKERROR
#ifdef CONFIG_ALPHA_PRINT_GPF_STACK
unsigned int *stack,*stack_top;
#endif // CONFIG_ALPHA_PRINT_GPF_STACK
    static char *s_fault_reason[] = {
        "OK", "Master Abort", "Retry timer expired", "Unknown" };

    static int s_faultcount = 0;
    unsigned int data;

    data = __raw_readb(REG_BASE_HOST + PCI_host_host_reg2 + 3);

    if ((data >> 1) == 1) {
	printk("GPF: pid(%d, <%s>)", current->pid, current->comm);
#ifdef CONFIG_BINFMT_FLAT
    	if ((regs->ARM_cpsr & MODE_MASK) == USR_MODE)
		/* Print out the offset from text section */
		printk(" text_offset(0x%lx)", 
			regs->ARM_pc - current->mm->start_code - 4); 
#endif
    } else
	printk("PCI: PCI bus fault %d : %s", 
            ++s_faultcount, s_fault_reason[data >> 1]);

    printk(" (");

    switch (regs->ARM_cpsr & MODE_MASK) {
    case USR_MODE : 
        printk("USER");
        break;
    case FIQ_MODE : 
        printk("FIQ");
        break;
    case IRQ_MODE : 
        printk("IRQ");
        break;
    case SVC_MODE : 
        printk("SVC");
        break;
    case ABT_MODE : 
        printk("ABT");
        break;
    case UND_MODE : 
        printk("UND");
        break;
    default :
        printk("UNKNOWN");
        break;
    }
    printk(" pc=%08lx r0=%08lx r3=%08lx sp=%08lx lr=%08lx)",
        regs->ARM_pc, regs->ARM_r0, regs->ARM_r3, regs->ARM_sp, regs->ARM_lr);
    printk("\n");
#ifdef	CONFIG_ALPHA_DUMP_ARM_REGS
	printk(" r0=%08lx r1=%08lx r2=%08lx r3=%08lx r4=%08lx)\n",
		regs->ARM_r0, regs->ARM_r1, regs->ARM_r2, regs->ARM_r3, regs->ARM_r4);
	printk(" r5=%08lx r6=%08lx r7=%08lx r8=%08lx r9=%08lx)\n",
		regs->ARM_r5, regs->ARM_r6, regs->ARM_r7, regs->ARM_r8, regs->ARM_r9);
	printk(" r10=%08lx fp=%08lx ip=%08lx sp=%08lx lr=%08lx pc=%08lx)\n",
		regs->ARM_r10, regs->ARM_fp, regs->ARM_ip, regs->ARM_sp, regs->ARM_lr, regs->ARM_pc);
#endif	// CONFIG_ALPHA_DUMP_ARM_REGS
#ifdef	CONFIG_ALPHA_BT_GPF
	if ((regs->ARM_cpsr & MODE_MASK) == USR_MODE) {
		unsigned long ufp;
		unsigned long ulr;
		unsigned long ulfp;
		unsigned long s_sp;
		unsigned long e_sp;
		s_sp = regs->ARM_sp;
		e_sp = s_sp + CONFIG_ALPHA_USER_STACK_DEPTH;
		ufp = regs->ARM_fp;
		printk("\033[1;33mFor mode USR, do the back trace..\033[0m\n");
		printk("\033[1;33mBT for lr = 0x%lx,function address:0x%lx\033[0m\n", regs->ARM_lr,regs->ARM_lr-current->mm->start_code - 4 );
		for(;;) {
			if ( (ufp > e_sp) || (ufp < s_sp) )
				break;
			ulr = *((unsigned long *)(ufp - 4));
			ulfp = *((unsigned long *)(ufp - 12));
			printk("\033[1;33mBT for lr = 0x%lx,function address:0x%lx\033[0m\n", ulr,ulr-current->mm->start_code - 4 );
			ufp = ulfp;
		}
	}
#endif	// CONFIG_ALPHA_BT_GPF
#ifdef CONFIG_ALPHA_PRINT_GPF_STACK
	printk(" current->mm->start_code=%08lx, current->mm->end_code=%08lx\n",current->mm->start_code,current->mm->end_code);
	stack = (unsigned int *) regs->ARM_sp;
	stack_top = (unsigned int *) (((unsigned int)stack+4095+4096*2)/4096*4096);
	while(stack <stack_top) {
		if (*stack >= current->mm->start_code && *stack <= current->mm->end_code)
			printk("[%8x](%8x)\n", *stack, *stack - current->mm->start_code );
		else
			printk("%8x\n", *stack);
		stack++;
	}
#endif	// CONFIG_ALPHA_PRINT_GPF_STACK
#endif

#if 0
    {
        int i;
        unsigned long addr = regs->ARM_pc - 0x20;

        // dump registers
        printk("Registers :\n");
        for (i = 0; i <= 17; ++i) {
            printk("%08x, ", regs->uregs[i]);
            if (i % 4 == 3)
                printk("\n\t");
        }
        printk("\n");

        // dump code
        for (i = 0; i < 0x10; ++i, addr += 4) {
            if (i % 4 == 0)
                printk("%08lx : ", addr);
            printk("%08x ", *(unsigned int *) addr);
            if (i % 4 == 3)
                printk("\n");
        }
        printk("\n");
    }
#endif

    // clears PCI bus fault flags
    __raw_writeb(0x01, REG_BASE_HOST + PCI_host_host_reg2 + 3);
    __raw_writeb(0x00, REG_BASE_HOST + PCI_host_host_reg2 + 3);

#ifndef	CONFIG_ALPHA_NO_SIGSEGV_SIGNAL
    kill_proc(current->pid, SIGSEGV, 0);
#endif // CONFIG_ALPHA_NO_SIGSEGV_SIGNAL
}

// There are 4 IRQs are defined for PCI interrupt in EM86XX (IRQ_PCIINTA - IRQ_PCIINTD) 
// Actually these interrupt lines are not wired to incoming PCI interrupt pins (INTA - INTD),
// rather incoming interrupts are routed to one of GPIO, and boot loader sets up
// the GPIO interrupt mapping register to issue a PCI interrupt. 
// Only one interrupt is assigned for PCI interrupt
//
// backplane board interrupt routing (710-01 board)
//
// Slot 0   Slot 1   Slot 2   Slot 3
// INTA#    INTB#    INTC#    INTD#
// INTB#    INTC#    INTD#    INTA#
// INTC#    INTD#    INTA#    INTB#
// INTD#    INTA#    INTB#    INTC#
//
// In EM86XX board, they are tied together and go to GPIO9
int em86xx_pci_map_irq(struct pci_dev *dev, u8 slot, u8 pin)
{
    int irq;
    int pcislot = PCI_SLOT(dev->devfn);

    if (pcislot >= PCIEM86XX_IDSEL_MAX)
        irq = 0xff;
    else
#ifdef	CONFIG_ALPHA_STAND_ALONE_PCI_INTERRUPTS
	switch(pcislot) {
		case 1:
			irq = IRQ_GPIO0;
			break;
		case 2:
			irq = IRQ_GPIO2;
			break;
		case 3:
			irq = IRQ_GPIO1;
			break;
		default:
			irq = PCI_IRQ_MAP(pcislot, pin);
			break;
	}
#else	// CONFIG_ALPHA_STAND_ALONE_PCI_INTERRUPTS
        irq = PCI_IRQ_MAP(pcislot, pin);
#endif	// CONFIG_ALPHA_STAND_ALONE_PCI_INTERRUPTS

    return irq;
}

#endif
