/*
 * arch/arm/mach-em86xx/pcifpga.c
 *
 * Copyright (C) 2003-2004 Sigma Designs, Inc
 *
 * by Ho Lee 07/19/2003
 */

#include <linux/config.h>

#ifdef CONFIG_PCI_EM86XX_HOST_FPGA

#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <asm/hardware.h>
#include <asm/io.h>

#include <linux/interrupt.h>
#include <asm/irq.h>
#include <asm/mach/irq.h>

#include <linux/pci.h>
#include <asm/mach/pci.h>
#include <asm/system.h>

#include <asm/arch/em86xxapi.h>

#define PCIFPGA_DMATRANSFER         0
#define PCIFPGA_ANALYZE_PCIMASTER   0

#define PCIFPGA_VERBOSE             0
#define PCIFPGA_VERBOSE_IRQ         0
#define PCIFPGA_VERBOSE_DATA        0

//
// Macro definitions
//

// absolute address
#define PCIFPGA_ADDR(x)                 (PCIFPGA_BASE + ((x) << 2))
#define PCIFPGA_ADDR_NEXT(x,n)          (x += (n << 2))

// relative to PCIFPGA_BASE
#define PCIFPGA_CONFIG_ADDR(x)          (PCIFPGA_CONFIG_BASE_0 + (x))
#define PCIFPGA_IO_ADDR(x)              ((x) - PCIFPGA_IOMEM_PREFIX)
#define PCIFPGA_MEMORY_ADDR(x)          ((x) - PCIFPGA_IOMEM_PREFIX)

//
// Function prototypes
//

static void pcifpga_init(void *sysdata);
static int pcifpga_map_irq(struct pci_dev *dev, u8 slot, u8 pin);

static int pcifpga_read_config_byte(struct pci_dev *dev, int where, u8 *value);
static int pcifpga_read_config_word(struct pci_dev *dev, int where, u16 *value);
static int pcifpga_read_config_dword(struct pci_dev *dev, int where, u32 *value);
static int pcifpga_write_config_byte(struct pci_dev *dev, int where, u8 value);
static int pcifpga_write_config_word(struct pci_dev *dev, int where, u16 value);
static int pcifpga_write_config_dword(struct pci_dev *dev, int where, u32 value);

//
// Global variables
//

static int g_pcifpga_inited = 0;

// MBUS DMA transfer
static int g_sbox_port_read, g_sbox_port_write;
static unsigned int g_regbase_read, g_regbase_write;
static int g_irq_read, g_irq_write;

//
// PCI operations data structure
//

struct pci_ops pcifpga_pci_ops = {
    pcifpga_read_config_byte,
    pcifpga_read_config_word,
    pcifpga_read_config_dword,
    pcifpga_write_config_byte,
    pcifpga_write_config_word,
    pcifpga_write_config_dword,
};

//
// PCI host controller data structure
//

struct hw_pci em86xx_pci = {
    init : pcifpga_init,
    setup_resources : NULL, 
    swizzle : no_swizzle, 
    map_irq : pcifpga_map_irq,
    mem_offset : 0,
    io_offset : 0,
};

//
// Primitive FPGA I/O
//

static __inline__ unsigned short pcifpga_readw(unsigned int addr)
{
    return __raw_readw(PCIFPGA_ADDR(addr));
}

static __inline__ unsigned int pcifpga_readl(unsigned int addr)
{
    unsigned int data;

    data = __raw_readw(PCIFPGA_ADDR(addr));
    data |= __raw_readw(PCIFPGA_ADDR(addr + 2)) << 16;

    return data;
}

// sequencial address
static __inline__ void pcifpga_readsw_seqaddr(unsigned int addr, unsigned short *pdata, int count)
{
    int i;

    addr = PCIFPGA_ADDR(addr);
    
    for (i = 0; i < count; ++i) {
        pdata[i] = __raw_readw(addr);
        PCIFPGA_ADDR_NEXT(addr, 2);
    }
}

// same address
static __inline__ void pcifpga_readsl(unsigned int addr, unsigned int *pdata, int count)
{
#if 0
    int i;
    unsigned short *pdata16 = (unsigned short *) pdata;
    addr = PCIFPGA_ADDR(addr);
    for (i = 0; i < count * 2; ++i)
        pdata16[i] = __raw_readw(addr);
#else
    int i, n;
    unsigned short *pdata16 = (unsigned short *) pdata;
    unsigned int addr_lo, addr_hi;

    addr_lo = PCIFPGA_ADDR(addr);
    addr_hi = PCIFPGA_ADDR(addr + 2);
    
    for (i = 0, n = 0; i < count; ++i) {
        pdata16[n++] = __raw_readw(addr_lo);
        pdata16[n++] = __raw_readw(addr_hi);
    }
#endif
}

static __inline__ void pcifpga_writew(unsigned int data, unsigned int addr)
{
    __raw_writew(data, PCIFPGA_ADDR(addr));
}

static __inline__ void pcifpga_writel(unsigned int data, unsigned int addr)
{
    __raw_writew(data & 0xffff, PCIFPGA_ADDR(addr));
    __raw_writew((data >> 16) & 0xffff, PCIFPGA_ADDR(addr + 2));
}

// same address
static __inline__ void pcifpga_writesl(unsigned int addr, unsigned int *pdata, int count)
{
    int i, n;
    unsigned short *pdata16 = (unsigned short *) pdata;
    unsigned int addr_lo, addr_hi;

    addr_lo = PCIFPGA_ADDR(addr);
    addr_hi = PCIFPGA_ADDR(addr + 2);
    
    for (i = 0, n = 0; i < count; ++i) {
        __raw_writew(pdata16[n++], addr_lo);
        __raw_writew(pdata16[n++], addr_hi);
    }
}

// 4 bytes writes : set PCI command (config/io/memory)
static __inline__ void pcifpga_writel_setcmd(int cmd)
{
    pcifpga_writew(cmd, PCIFPGA_WRITEDWORDCMD);
}

// 4 bytes write : start write transaction (io/memory)
// do not write configuration area using this function
static __inline__ void pcifpga_writel_cmd(unsigned int data, unsigned int addr)
{
    pcifpga_writel(PCIFPGA_IOMEM_PREFIX | addr, PCIFPGA_WRITEDWORDADDR);
    pcifpga_writel(data, PCIFPGA_WRITEDWORDDATA);
}

static int pcifpga_wait_isr(int wait)
{
    unsigned int isr;

    while (((isr = pcifpga_readw(PCIFPGA_ISR)) & wait) == 0)
        ;

    return isr & wait;
}

//
// primitive PCI FPGA I/O APIs
//

unsigned short __pcifpga_readw(unsigned int addr)
{
    return pcifpga_readw(addr);
}

unsigned int __pcifpga_readl(unsigned int addr)
{
    return pcifpga_readl(addr);
}

void __pcifpga_writew(unsigned int data, unsigned int addr)
{
    pcifpga_writew(data, addr);
}

void __pcifpga_writel(unsigned int data, unsigned int addr)
{
    pcifpga_writel(data, addr);
}

void pcifpga_irqmask_set(int mask)
{
    pcifpga_writew(mask, PCIFPGA_IRQMASK);
}

//
// Primitive I/O : Config / IO / Memory 
//   These functions return 0 if successes, non-zero if fails
//
//   pcifpga_read_raw_byte/word/dword
//   pcifpga_write_raw_byte/word/dword
// 

static unsigned int pcifpga_addr_config(unsigned int addr)
{
    return PCIFPGA_CONFIG_ADDR(addr);
}

static unsigned int pcifpga_addr_io(unsigned int addr)
{
    return PCIFPGA_IO_ADDR(addr);
}

static unsigned int pcifpga_addr_mem(unsigned int addr)
{
    return PCIFPGA_MEMORY_ADDR(addr);
}

#define PCIFPGA_READ_RAW_X(x, type, name)   \
    static int pcifpga_read_raw_##name(unsigned int addr, unsigned int (*convaddr)(unsigned int), unsigned type *data) \
    { \
        unsigned int isr, addr32; \
        if (!g_pcifpga_inited) { \
            if (PCIFPGA_VERBOSE) \
                printk("PCI FPGA is not initialized\n"); \
            *data = 0; \
            return 0; \
        } \
        addr32 = addr & ~0x03; \
        addr32 = convaddr(addr32); \
        if (PCIFPGA_VERBOSE) \
            printk("Read " #name " %08x (%08x) = ", addr, PCIFPGA_ADDR(addr32)); \
        /* setup byte enable bits */ \
        if (sizeof(type) != 4) { \
            int byteenable = 0x0f; \
            int bit = addr & ((sizeof(type) == 1) ? 0x03 : 0x02); \
            byteenable &= ~(((sizeof(type) == 1) ? 1 : 3) << bit); \
            pcifpga_writew(byteenable, PCIFPGA_BYTEENABLE_DMASTART); \
        } \
        /* issue PCI configuration read transaction. it always returns 0 */ \
        em86xx_mask_fiq(IRQ_PCIFPGA); \
        pcifpga_readw(addr32); \
        /* wait for interrupt */ \
        isr = pcifpga_wait_isr(PCIFPGA_IRQ_RDRESP | PCIFPGA_IRQ_MASTERABORT); \
        if (isr & PCIFPGA_IRQ_RDRESP) { \
            if (sizeof(type) == 1) { \
                unsigned short data16; \
                data16 = pcifpga_readw(PCIFPGA_IDEREADDATA + (addr & 0x02)); \
                if (addr & 0x01) \
                    *data = (data16 >> 8) & 0xff; \
                else \
                    *data = (data16 & 0xff); \
            } else if (sizeof(type) == 2) { \
                if (addr & 0x01) \
                    printk("Align fault (2) "); \
                *data = pcifpga_readw(PCIFPGA_IDEREADDATA + (addr & 0x02)); \
            } else { \
                if (addr & 0x03) \
                    printk("Align fault (4)\n"); \
                *data = pcifpga_readl(PCIFPGA_IDEREADDATA); \
            } \
            if (PCIFPGA_VERBOSE) \
                printk("%0*x\n", (int) sizeof(type) * 2, *data); \
        } else if (isr & PCIFPGA_IRQ_MASTERABORT) { \
            /*if (PCIFPGA_VERBOSE) */\
                printk("Master abort : Read %08x (%08x)\n", addr, PCIFPGA_ADDR(addr32)); \
            *data = 0; \
        } \
        em86xx_unmask_fiq(IRQ_PCIFPGA); \
        return 0; \
    }

#define PCIFPGA_WRITE_RAW_X(x, type, name)  \
    static int pcifpga_write_raw_##name(unsigned int addr, unsigned int (*convaddr)(unsigned int), unsigned int data) \
    { \
        unsigned int addr16; \
        if (!g_pcifpga_inited) { \
            if (PCIFPGA_VERBOSE) \
                printk("PCI FPGA is not initialized\n"); \
            return 0; \
        } \
        addr16 = addr & ~0x01; \
        addr16 = convaddr(addr16); \
        if (PCIFPGA_VERBOSE) \
            printk("Write " #name " %08x (%08x) = %0*x ", \
                addr, PCIFPGA_ADDR(addr16), (int) sizeof(type) * 2, data); \
        if (sizeof(type) == 1) { \
            unsigned int align = addr & 0x01; \
            unsigned short data16; \
            /* setup byte enable bits */ \
            int byteenable = 0x0f; \
            int bit = addr & 0x03; \
            byteenable &= ~(1 << bit); \
            pcifpga_writew(byteenable, PCIFPGA_BYTEENABLE_DMASTART); \
            /* write data */ \
            data16 = (data << (align * 8)); \
            pcifpga_writew(data16, addr16); \
        } else if (sizeof(type) == 2) { \
            if (addr & 0x01) \
                printk("Align fault (2)"); \
            pcifpga_writew(data, addr16); \
        } else { \
            unsigned int addr32; \
            if (addr & 0x03) \
                printk("Align fault (4)"); \
            addr32 = addr & ~0x03; \
            addr32 = convaddr(addr32); \
            if (addr < PCIFPGA_CONFIG_END) \
                pcifpga_writel(data, addr32); \
            else \
                pcifpga_writel_cmd(data, addr32); \
        } \
        if (PCIFPGA_VERBOSE) \
            printk("\n"); \
        return 0; \
    }

PCIFPGA_READ_RAW_X(b, char, byte)
PCIFPGA_READ_RAW_X(w, short, word)
PCIFPGA_READ_RAW_X(l, int, dword)

PCIFPGA_WRITE_RAW_X(b, char, byte)
PCIFPGA_WRITE_RAW_X(w, short, word)
PCIFPGA_WRITE_RAW_X(l, int, dword)

//
// primitive Configuration I/O
//   These functions return PCIBIOS_SUCCESSFUL if successes, non-zero if fails
//
//   pcifpga_read_config_byte/word/dword
//   pcifpga_write_config_byte/word/dword
//

#define PCIFPGA_READ_CONFIG_X(x, type, name)    \
    static int pcifpga_read_config_##name(struct pci_dev *dev, int addr, unsigned type *data) \
    { \
        int idsel = PCI_SLOT(dev->devfn), func = PCI_FUNC(dev->devfn); \
        if (idsel >= PCIFPGA_IDSEL_MAX) { \
            return PCIBIOS_DEVICE_NOT_FOUND; \
        } else { \
            if (PCIFPGA_VERBOSE) \
                printk("CONFIG "); \
            return pcifpga_read_raw_##name((idsel * 0x1000) | (func << 8) | addr, pcifpga_addr_config, data) == 0 ? PCIBIOS_SUCCESSFUL : PCIBIOS_DEVICE_NOT_FOUND; \
        } \
    }

#define PCIFPGA_WRITE_CONFIG_X(x, type, name)   \
    static int pcifpga_write_config_##name(struct pci_dev *dev, int addr, unsigned type data) \
    { \
        int idsel = PCI_SLOT(dev->devfn), func = PCI_FUNC(dev->devfn); \
        if (idsel >= PCIFPGA_IDSEL_MAX) { \
            return PCIBIOS_DEVICE_NOT_FOUND; \
        } else { \
            if (PCIFPGA_VERBOSE) \
                printk("CONFIG "); \
            if (sizeof(type) == 4) \
                pcifpga_writel_setcmd(PCI_CMD_CONFIGWRITE); \
            return pcifpga_write_raw_##name((idsel * 0x1000) | (func << 8) | addr, pcifpga_addr_config, data) == 0 ? PCIBIOS_SUCCESSFUL : PCIBIOS_DEVICE_NOT_FOUND; \
        } \
    }

PCIFPGA_READ_CONFIG_X(b, char, byte)
PCIFPGA_READ_CONFIG_X(w, short, word)
PCIFPGA_READ_CONFIG_X(l, int, dword)

PCIFPGA_WRITE_CONFIG_X(b, char, byte)
PCIFPGA_WRITE_CONFIG_X(w, short, word)
PCIFPGA_WRITE_CONFIG_X(l, int, dword)

//
// primitive PCI I/O Access
//
//   pcifpga_in_byte/word/dword
//   pcifpga_out_byte/word/dword
//

#define PCIFPGA_IN_X(x, type, name) \
    unsigned type pcifpga_in_##name(unsigned int addr) \
    { \
        unsigned type data; \
        if (PCIFPGA_VERBOSE) \
            printk("IO "); \
        pcifpga_read_raw_##name(addr, pcifpga_addr_io, &data); \
        return data; \
    } 

#define PCIFPGA_OUT_X(x, type, name) \
    void pcifpga_out_##name(unsigned int data, unsigned int addr) \
    { \
        if (PCIFPGA_VERBOSE) \
            printk("IO "); \
            if (sizeof(type) == 4) \
                pcifpga_writel_setcmd(PCI_CMD_IOWRITE); \
        pcifpga_write_raw_##name(addr, pcifpga_addr_io, data); \
    } 

PCIFPGA_IN_X(b, char, byte)
PCIFPGA_IN_X(w, short, word)
PCIFPGA_IN_X(l, int, dword)

PCIFPGA_OUT_X(b, char, byte)
PCIFPGA_OUT_X(w, short, word)
PCIFPGA_OUT_X(l, int, dword)

//
// primitive PCI Memory Access
//
//   pcifpga_read_byte/word/dword
//   pcifpga_write_byte/word/dword
//

#define PCIFPGA_READ_X(x, type, name) \
    unsigned type pcifpga_read_##name(unsigned int addr) \
    { \
        unsigned type data; \
        if (PCIFPGA_VERBOSE) \
            printk("MEMORY "); \
        pcifpga_read_raw_##name(addr, pcifpga_addr_mem, &data); \
        return data; \
    } 

#define PCIFPGA_WRITE_X(x, type, name) \
    void pcifpga_write_##name(unsigned int data, unsigned int addr) \
    { \
        if (PCIFPGA_VERBOSE) \
            printk("MEMORY "); \
            if (sizeof(type) == 4) \
                pcifpga_writel_setcmd(PCI_CMD_MEMORYWRITE); \
        pcifpga_write_raw_##name(addr, pcifpga_addr_mem, data); \
    } 

PCIFPGA_READ_X(b, char, byte)
PCIFPGA_READ_X(w, short, word)
PCIFPGA_READ_X(l, int, dword)

PCIFPGA_WRITE_X(b, char, byte)
PCIFPGA_WRITE_X(w, short, word)
PCIFPGA_WRITE_X(l, int, dword)

//
// Block operations
//

void pcifpga_memcpy_fromio(void *ptr, unsigned long addr, size_t len)
{
    unsigned char *pdata = (unsigned char *) ptr;

    while (len-- > 0) 
        *pdata = pcifpga_raw_readb(addr++);
}

void pcifpga_memcpy_toio(unsigned long addr, const void *ptr, size_t len)
{
    unsigned char *pdata = (unsigned char *) ptr;

    while (len-- > 0) 
        pcifpga_raw_writeb(*pdata++, addr++);
}

void pcifpga_memset_io(unsigned long addr, int data, size_t len)
{
    while (len-- > 0) 
        pcifpga_raw_writeb(data, addr++);
}

//
// String operations
//

static __inline unsigned char align_readb(unsigned char *pdata) 
{
    return *pdata;
}

static __inline__ unsigned short align_readw(unsigned short *pdata)
{
    unsigned char *pdata8 = (unsigned char *) pdata;
    unsigned short data = pdata8[0] | (pdata8[1] << 8);
    return data;
}

static __inline__ unsigned int align_readl(unsigned int *pdata)
{
    if ((unsigned int) pdata & 0x01) {
        unsigned char *pdata8 = (unsigned char *) pdata;
        unsigned int data = pdata8[0] | (pdata8[1] << 8) | (pdata8[2] << 16) | (pdata8[3] << 24);
        return data;
    } else {
        unsigned short *pdata16 = (unsigned short *) pdata;
        unsigned int data = pdata16[0] | (pdata16[1] << 16);
        return data;
    }
}

static __inline__ void align_writeb(unsigned char *pdata, unsigned char data)
{
    *pdata = data;
}

static __inline__ void align_writew(unsigned short *pdata, unsigned short data)
{
    unsigned char *pdata8 = (unsigned char *) pdata;
    pdata8[0] = (data & 0xff);
    pdata8[1] = ((data >> 8) & 0xff);
}

static __inline__ void align_writel(unsigned int *pdata, unsigned int data)
{
    if ((unsigned int) pdata & 0x01) {
        unsigned char *pdata8 = (unsigned char *) pdata;
        pdata8[0] = (data & 0xff);
        pdata8[1] = ((data >> 8) & 0xff);
        pdata8[2] = ((data >> 16) & 0xff);
        pdata8[3] = ((data >> 24) & 0xff);
    } else {
        unsigned short *pdata16 = (unsigned short *) pdata;
        pdata16[0] = (data & 0xffff);
        pdata16[1] = ((data >> 16) & 0xffff);
    }
}

#define PCIFPGA_RAW_READS_X(x, type, name) \
    void pcifpga_raw_reads##x(unsigned int addr, void *data, int len) \
    { \
        unsigned type *pdata = (unsigned type *) data; \
        /* alignment check */ \
        if (sizeof(type) != 1 && (((unsigned int) data) & (sizeof(type) - 1)) != 0) { \
            /* alignment mismatch */ \
            while (len -- > 0) { \
                align_write##x(pdata, pcifpga_raw_read##x(addr)); \
                ++pdata; \
            } \
        } else { \
            /* alignment is OK */ \
            while (len-- > 0) \
                *pdata++ = pcifpga_raw_read##x(addr); \
        } \
    }

#define PCIFPGA_RAW_WRITES_X(x, type, name) \
    void pcifpga_raw_writes##x(unsigned int addr, void *data, int len) \
    { \
        unsigned type *pdata = (unsigned type *) data; \
        /* alignment check */ \
        if (sizeof(type) != 1 && (((unsigned int) data) & (sizeof(type) - 1)) != 0) { \
            /* alignment mismatch */ \
            while (len-- > 0) { \
                pcifpga_raw_write##x(align_read##x(pdata), addr); \
                pdata++; \
            } \
        } else { \
            /* alignment is OK */ \
            while (len-- > 0) \
                pcifpga_raw_write##x(*pdata++, addr); \
        } \
    }

PCIFPGA_RAW_READS_X(b, char, byte)
PCIFPGA_RAW_READS_X(w, short, word)
PCIFPGA_RAW_READS_X(l, int, dword)

PCIFPGA_RAW_WRITES_X(b, char, byte)
PCIFPGA_RAW_WRITES_X(w, short, word)
PCIFPGA_RAW_WRITES_X(l, int, dword)
    
//
// Initialization & Core operations
//

static void pcifpga_fiq(int fiq, void *devinfo);
static void pcifpga_softirq(int irq, void *devinfo, struct pt_regs *regs);

void pcifpga_init(void *sysdata)
{
    // test the scratch pad register 
    static unsigned short s_test_data[] = {
        0x4d47, 0x4953, 0x424d, 0 };
    int i;

    for (i = 0; s_test_data[i] != 0; ++i) {
        pcifpga_writew(s_test_data[i], PCIFPGA_SCRATCH);
        if (pcifpga_readw(PCIFPGA_SCRATCH) != s_test_data[i])
            return;
    }

    printk("PCI FPGA version %d\n", pcifpga_readw(PCIFPGA_VERSION));

    // setup MBUS
    g_sbox_port_read = em86xx_mbus_alloc_dma(SBOX_IDEFLASH, 1, &g_regbase_read, &g_irq_read);
    g_sbox_port_write = em86xx_mbus_alloc_dma(SBOX_IDEFLASH, 0, &g_regbase_write, &g_irq_write);

    // set interrupt mask
    pcifpga_irqmask_set(PCIFPGA_IRQ_FIFOBUSY);

    // set I/O and Memory base & mask
    pcifpga_writel(PCIFPGA_IOBASE_VALID, PCIFPGA_IOBASE);
    pcifpga_writel(PCIFPGA_IOBASE_MASK, PCIFPGA_IOMASK);
    pcifpga_writel(PCIFPGA_MEMORYBASE_VALID, PCIFPGA_MEMORYBASE);
    pcifpga_writel(PCIFPGA_MEMORYBASE_MASK, PCIFPGA_MEMORYMASK);

    // basic initialization is done
    g_pcifpga_inited = 1;

    // register handler for FPGA interrupt
    request_fiq(IRQ_PCIFPGA, pcifpga_fiq, "PCI FPGA", NULL);
    request_irq(IRQ_SOFTINT, pcifpga_softirq, SA_SHIRQ, "PCI FPGA", (void *) pcifpga_softirq);

    // Scan PCI Bus
    pci_scan_bus(0, &pcifpga_pci_ops, sysdata);
}

int pcifpga_map_irq(struct pci_dev *dev, u8 slot, u8 pin)
{
    int irq;
    int idsel = PCI_SLOT(dev->devfn);

    if (idsel >= PCIFPGA_IDSEL_MAX) 
        irq = 0xff;
    else 
        irq = IRQ_SOFTINT;

    return irq;
}

#if PCIFPGA_DMATRANSFER

// read : direction of the transfer 
//   0 (write) : host => FPGA 
//   1 (read) : FPGA => host
// addr : address in host memory
// size : number of bytes
void pcifpga_do_dma(int read, unsigned int addr, unsigned int size)
{
    // clean data cache before transfer
    cpu_dcache_clean_range(addr, addr + size);

    // setup PB automode registers
    __raw_writel(0, REG_BASE_HOST + PB_automode_start_address);
    __raw_writel(0x03140000 | ((read ? 1 : 0) << 16) | (size >> 1), REG_BASE_HOST + PB_automode_control);

    // setup MBUS
    em86xx_mbus_setup_dma_linear(read ? g_regbase_read : g_regbase_write, addr, size);

    // start DMA transfer
    pcifpga_writew(read ? PCIFPGA_DMASTART_WRITE : PCIFPGA_DMASTART_READ, PCIFPGA_BYTEENABLE_DMASTART);

    // wait until MBUS transfer is done
    em86xx_mbus_wait(read ? g_regbase_read : g_regbase_write);

    // invalidate data cache after transfer
    if (read)
        cpu_dcache_invalidate_range(addr, addr + size);
}

#endif

// If the interrupt is detected by edge detector, the interrupt line should
// be cleared inside interrupt handler, because there is no edge change, 
// so it doesn't get any more interrupt
//
// If the interrupt is detected by level detector, it doesn't matter. Even
// the interrupt handler exit without processing every interrupt bits, 
// interrupt handler is executed again right after it return to normal mode.
//
// interrupt line is active low : the interrupt is detected by edge detector. 
// To process PCI interrupt : 
//   FIQ mode : pcifpga_fiq() is called
//              mask PCI interrupt
//              call pcifpga_do_pciirq()
//   IRQ mode : call pcifpga_call_irq_handler
//              user interrupt handler is called
//   FIQ mode : unmask PCI interrupt
//              return to FIQ mode and process remaing part
//
// interrupt line is active high : the interrupt is detected by level detector.
// Prior implementation has problem : FIQ handler may be call while the processor
//   is in IRQ mode. So while processing PCI interrupt, FIQ handler is called and 
//   it shares the same stack, and eventually system down. To avoid this problem, 
//   the FIQ vector should keep depth (the maximum depth is 2), and use assign 
//   alternate stack for 2nd depth FIQ handler. 
// To process PCI interrupt : 
//   FIQ mode : pcifpga_fiq() is called
//              mask PCI interrupt
//              mark soft interrupt
//              exit FIQ handler
//   IRQ mode : interrupt handler is called automatically
//   FIQ mode : if FIQ occurrs again, FIQ handler is called. We already block the
//              PCI interrupts, so the interrupt are about PCI bus mastering transfers.
//   IRQ mode : execute the remaining part of IRQ handler
//              unmask PCI interrupt
void pcifpga_fiq(int fiq, void *devinfo)
{
    unsigned int isr, mask;
    int nloop = 0;

    for (;;) {
        ++nloop;

        isr = pcifpga_readw(PCIFPGA_ISR);
        mask = pcifpga_readw(PCIFPGA_IRQMASK);

        if ((isr & ~mask) == 0) {
            if (nloop == 1)
                printk("Ambiguous PCI-FPGA interrupt : ISR = %04x, MASK = %04x\n", isr, mask);
            return;
        }

        if (PCIFPGA_VERBOSE_IRQ) {
            static char *s_isrstr[] = {
                "WRREQ", "RDREQ", "RDRESP", "INTA", "INTB", "INTC", "INTD", "ERROR", "ABORT", "FIFOBUSY",
            };
        
            int i;
    
            printk("%d : ISR = 0x%04x (", nloop, isr); 
            for (i = 0; i < 10; ++i)
                if (isr & (1 << i)) {
                    if (mask & (1 << i)) 
                        printk("<%s> ", s_isrstr[i]);
                    else
                        printk("%s ", s_isrstr[i]);
                }
            printk(")\n");
        }

        isr &= ~mask;
    
        while (pcifpga_readw(PCIFPGA_ISR) & PCIFPGA_IRQ_RDREQ) {
            // Read request from device by bus mastering
            unsigned int addr = pcifpga_readl(PCIFPGA_READADDR);
            unsigned int len = pcifpga_readw(PCIFPGA_READLEN);
            unsigned int *pdata = (unsigned int *) addr;

#if PCIFPGA_ANALYZE_PCIMASTER
            static int laststart = 0, lastend = 0;

            if (len > 0) {
                if (addr < lastend && (addr + len * 4) > laststart) {
                    printk("Read Request overwrap : [%08x - %08x] [%08x - %08x]\n",
                        laststart, lastend, addr, addr + len * 4);
                }
                laststart = addr;
                lastend = addr + len * 4;
            }
#endif
    
            if (len > 0) {
                if (PCIFPGA_VERBOSE_IRQ)
                    printk("[PCI Read Request] %08x - %04x\n", addr, len);
        
#if PCIFPGA_DMATRANSFER
                pcifpga_do_dma(0, addr, len << 2);
#else
                pcifpga_writesl(PCIFPGA_PCIREADRESPONSEDATA, pdata, len);
#endif
        
#if PCIFPGA_VERBOSE_DATA
                if (PCIFPGA_VERBOSE_IRQ) {
                    int i;
        
                    for (i = 0; i < len; ++i) {
                        printk("%08x ", pdata[i]);
                        if ((i & 0x03) == 0x03)
                            printk("\n");
                    }
                    if ((i & 0x03) != 0)
                        printk("\n");
                }
#endif
            }
        } 
    
        while (pcifpga_readw(PCIFPGA_ISR) & PCIFPGA_IRQ_WRREQ) {
            // Write request from device by bus mastering
            unsigned int addr = pcifpga_readl(PCIFPGA_WRITEADDR);
            unsigned int len = pcifpga_readw(PCIFPGA_WRITELEN);
            unsigned int *pdata = (unsigned int *) addr;

#if PCIFPGA_ANALYZE_PCIMASTER
            static int laststart = 0, lastend = 0;

            if (len > 0) {
                if (addr < lastend && (addr + len * 4) > laststart) {
                    printk("Write Request overwrap : [%08x - %08x] [%08x - %08x]\n",
                        laststart, lastend, addr, addr + len * 4);
                }
                laststart = addr;
                lastend = addr + len * 4;
            }
#endif
    
            if (len > 0) {
                if (PCIFPGA_VERBOSE_IRQ)
                    printk("[PCI Write Request] %08x - %04x\n", addr, len);
    
#if PCIFPGA_DMATRANSFER
                pcifpga_do_dma(1, addr, len << 2);
#else
                pcifpga_readsl(PCIFPGA_PCIWRITEDATA, pdata, len);
#endif
    
#if PCIFPGA_VERBOSE_DATA
                if (PCIFPGA_VERBOSE_IRQ) {
                    int i;
    
                    for (i = 0; i < len; ++i) {
                        printk("%08x ", pdata[i]);
                        if ((i & 0x03) == 0x03)
                            printk("\n");
                    }
                    if ((i & 0x03) != 0)
                        printk("\n");
                }
#endif
            }
        }
    
        if (isr & PCIFPGA_IRQ_RDRESP) {
            // Read data ready
            // this kind interrupt shouldn't occur because the read operation
            // is done with interrupt disabled
            if (PCIFPGA_VERBOSE_IRQ)
                printk("[### PCI Read Data Ready]\n");
        }

        if (pcifpga_readw(PCIFPGA_ISR) & PCIFPGA_IRQ_FIFOBUSY) {
            //while (pcifpga_readw(PCIFPGA_ISR) & PCIFPGA_IRQ_FIFOBUSY)
            //  ;
        }
    
        if (isr & PCIFPGA_IRQ_INTMASK) {
            // interrupt request
            if (PCIFPGA_VERBOSE_IRQ) {
                int i, irq;
                printk("[PCI Interrupt Request ");
                for (i = PCIFPGA_IRQ_INTA, irq = 0; i <= PCIFPGA_IRQ_INTD; i <<= 1, ++irq) 
                    if (i & isr)
                        printk("%c", 'A' + irq);
                printk("]\n");
            }
            pcifpga_irqmask_set(PCIFPGA_IRQMASK_PCIINT);
            em86xx_softirq_set(SOFTIRQ_PCIFPGA);
        }
    
        if (isr & PCIFPGA_IRQ_ERROR) {
            if (PCIFPGA_VERBOSE_IRQ)
                printk("[PCI Error]\n");
        } 
    
        if (isr & PCIFPGA_IRQ_MASTERABORT) {
            if (PCIFPGA_VERBOSE_IRQ)
                printk("[PCI Master Abort]\n");
        }
    } 
}

// interrupt handler
void pcifpga_softirq(int irq, void *devinfo, struct pt_regs *regs)
{
    em86xx_softirq_clr(SOFTIRQ_PCIFPGA);
}

#ifdef CONFIG_PCI_EM86XX_HOST_FPGA_OBSOLETE

// 
// old PCI FPGA interrupt processing archtecture
//

void pcifpga_do_pciirq(void);

// call registered interrupt handler directly
// I tried to call do_IRQ() function from pcifpga_do_pciirq(),
// but it ends in kernel panic (null pointer access)
// I guess the regs value shouldn't be NULL to call do_IRQ()
void pcifpga_call_irq_handler(int irq, struct pt_regs *regs)
{
    struct irqdesc *desc = irq_desc + irq;
    struct irqaction *action;

    desc = desc + irq;
    action = desc->action;

    desc->mask_ack(irq);

    if (action) {
        do {
            action->handler(irq, action->dev_id, regs);
            action = action->next;
        } while (action);
    }
}

#endif

//
// Miscellaneous
//

int pcifpga_info(void)
{
    static char *s_isrstr[] = {
        "WRREQ", "RDREQ", "RDRESP", "INTA", "INTB", "INTC", "INTD", "ERROR", "ABORT", "FIFOBUSY",
    };
    
    int i;
    unsigned int isr, irqmask;

    if (!g_pcifpga_inited)
        return 1;
    
    printk("FPGA-PB PCI Host Controller : \n");
    
    irqmask = pcifpga_readw(PCIFPGA_IRQMASK);
    printk("  ISR           : 0x%08x (", isr = pcifpga_readw(PCIFPGA_ISR));
    for (i = 0; i < 10; ++i)
        if (isr & (1 << i)) {
            if (irqmask & (1 << i)) 
                printk("<%s> ", s_isrstr[i]);
            else
                printk("%s ", s_isrstr[i]);
        } else if (irqmask & (1 << i))
            printk("[%s] ", s_isrstr[i]);
        
    printk(")\n");
    
    printk("  Read Address  : 0x%08x\n", pcifpga_readl(PCIFPGA_READADDR));
    printk("  Read Length   : 0x%04x\n", pcifpga_readl(PCIFPGA_READLEN));
    printk("  Write Address : 0x%08x\n", pcifpga_readl(PCIFPGA_WRITEADDR));
    printk("  Write Length  : 0x%04x\n", pcifpga_readl(PCIFPGA_WRITELEN));
    printk("  IO Base       : 0x%08x\n", pcifpga_readl(PCIFPGA_IOBASE));
    printk("  IO Mask       : 0x%08x\n", pcifpga_readl(PCIFPGA_IOMASK));
    printk("  MEMORY Base   : 0x%08x\n", pcifpga_readl(PCIFPGA_MEMORYBASE));
    printk("  MEMORY Mask   : 0x%08x\n", pcifpga_readl(PCIFPGA_MEMORYMASK));
    
    return 0;
}

#endif
