/*
 *  linux/arch/arm/mm/proc-pt110.S: MMU functions for PT110
 *
 *  Copyright (C) 2001 Sigma Designs, Inc
 *  Copyright (C) 2000 Steve Hill (sjhill@cotw.com)
 *                     Rob Scott (rscott@mtrob.fdns.net)
 *  Copyright (C) 2000 ARM Limited, Deep Blue Solutions Ltd.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *
 * These are the low level assembler for performing cache and TLB
 * functions on the PT110.  The ARM720T has a writethrough IDC
 * cache, so we don't need to clean it. The PT110 has writethrough or writeback
 * caches. PT110 can also clean/flush a single entry and separate I-cache and
 * D-cache flush. But for now we always clean/flush everything
 *
 *
 *  Changelog:
 *   05-09-2000 SJH Created by moving 720 specific functions
 *          out of 'proc-arm6,7.S' per RMK discussion
 *   07-25-2000 SJH Added idle function.
 *   08-25-2000 DBS Updated for integration of ARM Ltd version.
 *   12-03-2001 FG      Forked 740 function from 720
 *   12-12-2001 FG  Forked pt110 from arm740
 *   01-20-2004 Ho Lee, code cleanup & checkup
 */
#include <linux/linkage.h>
#include <asm/assembler.h>
#include <asm/constants.h>
#include <asm/procinfo.h>
#include <asm/hardware.h>

/*
 * comment on cache management
 * 
 * terms :
 *   clean : writing dirty lines out to memory
 *   invalidate/flush : invalidate cache lines
 *
 * single cache entry flushing : Flushing single entry instruction flushes
 * single line in the cache. The location is determined by the index address
 * passed to the instruction. Note that the register contains 'index' not 
 * 'address'. Even the instruction accepts 'address' and and it would flush 
 * even the tag is not met. That is the corresponding entry would be flushed 
 * regardless of 'cache hit' or 'cache miss' of the address. And this may
 * cause side effect flushing undesired cache entry. So before invalidating
 * cache entry, clean the cache entry first not to lose the write-back cache.
 *
 * interrupt disabling in the middle of cache operations : IRQ and FIQ should
 * be disabled during the cache operation. For example, if we clean and 
 * invalidate data cache, and interrupt handler intervens between these
 * two operations and makes the cache dirty, 'invalidate' operation after
 * finishing interrupt handler would wipe out the write-back cache contents. 
 * 
 * DMA write (read from host memory) : 
 *   Clean cache before starting DMA transfer.
 * DMA read (write to host memory) :
 *   Clean cache before starting DMA transfer. If the cache is not cleaned,
 *     later 'clean operation' would overwrite data already modifed by DMA 
 *     controller.
 *   Clean and invaldate after DMA transfer. This should be done with IRQ/FIQ
 *     diabled. Clean operation writes dirty lines made by interrupt handler
 *     out to memory, and invalidate operations will invalidate cache because
 *     the area is already overwritten by DMA controller.
 *
 * Cache instructions :
 *   clean data cache (all) :                mcr p15, 0, Rd, c7, c10, 0
 *   clean data cache (single) :             mcr p15, 0, Rd, c7, c10, 1
 *   invalidate instruction cache (all) :    mcr p15, 0, Rd, c7, c5, 0
 *   invalidate instruction cache (single) : mcr p15, 0, Rd, c7, c5, 1
 *   invalidate data cache (all) :           mcr p15, 0, Rd, c7, c6, 0
 *   invalidate data cache (single) :        mcr p15, 0, Rd, c7, c6, 1
 *
 * Cache manaement functions :
 *   cache_clean_invalidate : clean and invalidate I & D
 *   dcache_invalidate : clean and invalidate D
 *   dcache_clean : clean D
 *   icache_invalidate : clean D and invalidate I
 */

/*
 * Invalidating data cache region is more efficient than invalidating all
 * data cache. But cleaning all data cache would be more faster than 
 * flushing data cache region because it may avoid lengthy loop
 *
 * define or undefine this maro
*/
//#define FLUSH_DCACHE_ALL

/*
 * During invalidating/cleaning cache entries, the FIQ can happen and 
 * may corrupt the cache lines. With this option defined, the cache  
 * cleaning/invalidating on same cache will be atomic.
 *
 * define or undefine this maro
*/
#define DISABLE_FIQ

/*
 * void cpu_pt110_cache_clean_invalidate_all(void)
 *
 * This function flushes all cache lines
 */

ENTRY(cpu_pt110_cache_clean_invalidate_all)
        save_flags_clif r2, r3
        mcr p15, 0, r0, c7, c10, 0      @ clean data cache (all)
        mcr p15, 0, r0, c7, c5, 0       @ flush instruction cache (all)
        mcr p15, 0, r0, c7, c6, 0       @ flush data cache (all)
        restore_flags r2
        mov pc, lr

/* 
 * void cpu_pt110_cache_clean_invalidate_range(unsigned long address, unsigned long end, int flags)
 *   r0 = address 
 *   r1 = end
 *   r2 = flags 
 *     non zero : I cache as well
 */

ENTRY(cpu_pt110_cache_clean_invalidate_range)
        bic r0, r0, #0xF
        orr r1, r1, #0xF
#ifdef DISABLE_FIQ
	stmfd sp!, {r4}
#endif
        teq r2, #0
        movne r0, #0
        mcrne p15, 0, r0, c7, c5, 0     @ flush instruction cache (all)
        save_flags_cli r2, r3
//#ifdef FLUSH_DCACHE_ALL
        mcr p15, 0, r0, c7, c10, 0      @ clean data cache (all)
//#endif
1:
#ifdef DISABLE_FIQ
        save_flags_clf r4, r3
#endif
#ifndef FLUSH_DCACHE_ALL
        mcr p15, 0, r0, c7, c10, 1      @ clean data cache (single)
#endif
        mcr p15, 0, r0, c7, c6, 1       @ flush data cache (single)
        add r0, r0, #0x10
#ifdef DISABLE_FIQ
        restore_flags r4
#endif
        cmp r0, r1
        blo 1b
        restore_flags r2
#ifdef DISABLE_FIQ
	ldmfd sp!, {r4}
#endif
        mov pc, lr

/*
 * void cpu_pt110_icache_invalidate_page(void *virt_page);
 *   r0 = start address
 * void cpu_pt110_icache_invalidate_range(unsigned long start, unsigned long end);
 *   r0 = start address
 *   r1 = end address
 * PAGE_SIZE = 4KB = 0x1000
 */

ENTRY(cpu_pt110_icache_invalidate_page)
        bic r0, r0, #0xFF0
        orr r1, r0, #0xFF0
ENTRY(cpu_pt110_icache_invalidate_range)
        bic r0, r0, #0xF
        orr r1, r1, #0xF
#ifdef DISABLE_FIQ
	stmfd sp!, {r4}
#endif
        save_flags_cli r2, r3
//#ifdef FLUSH_DCACHE_ALL
        mcr p15, 0, r0, c7, c10, 0      @ clean data cache (all)
//#endif
1:      
#ifdef DISABLE_FIQ
        save_flags_clf r4, r3
#endif
#ifndef FLUSH_DCACHE_ALL
        mcr p15, 0, r0, c7, c10, 1      @ clean data cache (single)
#endif
        mcr p15, 0, r0, c7, c5, 1       @ flush instruction cache (single)
        add r0, r0, #0x10
#ifdef DISABLE_FIQ
        restore_flags r4
#endif
        cmp r0, r1
        blo 1b
        restore_flags r2
#ifdef DISABLE_FIQ
	ldmfd sp!, {r4}
#endif
        mov pc, lr

/* 
 * void cpu_dcache_invalidate_range(unsigned long start, unsigned long end);
 *   r0 = start address
 *   r1 = end address
 */

ENTRY(cpu_pt110_dcache_invalidate_range)
        bic r0, r0, #0xF
        orr r1, r1, #0xF
#ifdef DISABLE_FIQ
	stmfd sp!, {r4}
#endif
        save_flags_cli r2, r3
//#ifdef FLUSH_DCACHE_ALL
        mcr p15, 0, r0, c7, c10, 0      @ clean data cache (all)
//#endif
1:
#ifdef DISABLE_FIQ
        save_flags_clf r4, r3
#endif
#ifndef FLUSH_DCACHE_ALL
        mcr p15, 0, r0, c7, c10, 1      @ clean data cache (single)
#endif
        mcr p15, 0, r0, c7, c6, 1       @ flush data cache (single)
        add r0, r0, #0x10
#ifdef DISABLE_FIQ
        restore_flags r4
#endif
        cmp r0, r1
        blo 1b
        restore_flags r2
#ifdef DISABLE_FIQ
	ldmfd sp!, {r4}
#endif
        mov pc, lr

/*
 * void cpu_dcache_clean_range(unsigned long start, unsigned long end);
 * void cpu_flush_ram_page(void *virt_page);
 *   r0 = start address
 * void cpu_dcache_clean_page(void *virt_page);
 *   r0 = start address
 *   r1 = end address
 */

#if 1
ENTRY(cpu_pt110_dcache_clean_page)
ENTRY(cpu_pt110_flush_ram_page)
ENTRY(cpu_pt110_dcache_clean_range)
        mcr p15, 0, r0, c7, c10, 0      @ clean data cache (all)
        mov pc, lr
#else
#ifdef FLUSH_DCACHE_ALL
ENTRY(cpu_pt110_dcache_clean_page)
ENTRY(cpu_pt110_flush_ram_page)
ENTRY(cpu_pt110_dcache_clean_range)
        mcr p15, 0, r0, c7, c10, 0      @ clean data cache (all)
        mov pc, lr
#else
ENTRY(cpu_pt110_dcache_clean_page)
ENTRY(cpu_pt110_flush_ram_page)
        bic r0, r0, #0xFF0
        orr r1, r0, #0xFF0
ENTRY(cpu_pt110_dcache_clean_range)
        bic r0, r0, #0xF
        orr r1, r1, #0xF
#ifdef DISABLE_FIQ
	stmfd sp!, {r4}
#endif
        save_flags_cli r2, r3
1:      
#ifdef DISABLE_FIQ
        save_flags_clf r4, r3
#endif
        mcr p15, 0, r0, c7, c10, 1      @ clean data cache (single)
        add r0, r0, #0x10
#ifdef DISABLE_FIQ
        restore_flags r4
#endif
        cmp r0, r1
        blo 1b
        restore_flags r2
#ifdef DISABLE_FIQ
	ldmfd sp!, {r4}
#endif
        mov pc, lr
#endif
#endif

/* 
 * void cpu_dcache_clean_entry(unsigned long address);
 *   r0 = address
 */

ENTRY(cpu_pt110_dcache_clean_entry)
        mcr p15, 0, r0, c7, c10, 1      @ clean data cache (single)
        mov pc, lr

/*
 * Function: pt110_tlb_invalidate_all (void)
 *
 * Purpose : flush all TLB entries in all caches
 */
ENTRY(cpu_pt110_tlb_invalidate_all)
        mov pc, lr

/*
 * Function: pt110_tlb_invalidate_page (unsigned long address, int end, int flags)
 *
 * Params  : address    Area start address
 *     : end    Area end address
 *     : flags  b0 = I cache as well
 *
 * Purpose : flush a TLB entry
 */
ENTRY(cpu_pt110_tlb_invalidate_range)
        mov pc, lr

/*
 * Function: pt110_tlb_invalidate_page (unsigned long address, int flags)
 *
 * Params  : address    Address
 *     : flags  b0 = I-TLB as well
 *
 * Purpose : flush a TLB entry
 */
ENTRY(cpu_pt110_tlb_invalidate_page)
        mov pc, lr

/*
 * Function: pt110_data_abort ()
 *
 * Params  : r0 = address of aborted instruction
 *
 * Purpose : obtain information about current aborted instruction
 *
 * Returns : r0 = address of abort
 *     : r1 != 0 if writing
 *     : r3 = FSR
 *     : sp = pointer to registers
 */

Ldata_ldmstm:   tst r4, #1 << 21            @ check writeback bit
        beq Ldata_simple
        mov r7, #0x11
        orr r7, r7, r7, lsl #8
        and r0, r4, r7
        and r2, r4, r7, lsl #1
        add r0, r0, r2, lsr #1
        and r2, r4, r7, lsl #2
        add r0, r0, r2, lsr #2
        and r2, r4, r7, lsl #3
        add r0, r0, r2, lsr #3
        add r0, r0, r0, lsr #8
        add r0, r0, r0, lsr #4
        and r7, r0, #15         @ r7 = no. of registers to transfer.
        and r5, r4, #15 << 16       @ Get Rn
        ldr r0, [sp, r5, lsr #14]       @ Get register
        tst r4, #1 << 23            @ U bit
        subne   r7, r0, r7, lsl #2
        addeq   r7, r0, r7, lsl #2      @ Do correction (signed)
Ldata_saver7:   str r7, [sp, r5, lsr #14]       @ Put register
Ldata_simple:
#ifdef NO_MM
        orr r1, r2, #1
        mov r0, #0
#else
        mrc p15, 0, r0, c6, c0, 0       @ get FAR
        mrc p15, 0, r3, c5, c0, 0       @ get FSR
#endif
        and r3, r3, #255
#ifdef NO_MM
        mov r0, #1
#endif
        mov pc, lr

ENTRY(cpu_pt110_data_abort)
        ldr r4, [r0]            @ read instruction causing problem
        tst r4, r4, lsr #21         @ C = bit 20
        sbc r1, r1, r1          @ r1 = C - 1
        and r2, r4, #15 << 24
        add pc, pc, r2, lsr #22     @ Now branch to the relevent processing routine
        movs    pc, lr

        b   Ldata_lateldrhpost      @ ldrh  rd, [rn], #m/rm
        b   Ldata_lateldrhpre       @ ldrh  rd, [rn, #m/rm]
        b   Ldata_unknown
        b   Ldata_unknown
        b   Ldata_lateldrpostconst      @ ldr   rd, [rn], #m
        b   Ldata_lateldrpreconst       @ ldr   rd, [rn, #m] 
        b   Ldata_lateldrpostreg        @ ldr   rd, [rn], rm
        b   Ldata_lateldrprereg     @ ldr   rd, [rn, rm]
        b   Ldata_ldmstm            @ ldm*a rn, <rlist>
        b   Ldata_ldmstm            @ ldm*b rn, <rlist>
        b   Ldata_unknown
        b   Ldata_unknown
        b   Ldata_simple            @ ldc   rd, [rn], #m    @ Same as ldr   rd, [rn], #m
        b   Ldata_simple            @ ldc   rd, [rn, #m]
        b   Ldata_unknown

Ldata_unknown:  @ Part of jumptable
        mov r0, r2
        mov r1, r4
        mov r2, r3
        bl  baddataabort
        b   ret_from_exception

Ldata_lateldrhpre:
        tst r4, #1 << 21            @ check writeback bit
        beq Ldata_simple
Ldata_lateldrhpost:
        and r5, r4, #0x00f          @ get Rm / low nibble of immediate value
        tst r4, #1 << 22            @ if (immediate offset)
        andne   r2, r4, #0xf00          @ { immediate high nibble
        orrne   r2, r5, r2, lsr #4      @   combine nibbles } else
        ldreq   r2, [sp, r5, lsl #2]        @ { load Rm value }
        and r5, r4, #15 << 16       @ get Rn
        ldr r0, [sp, r5, lsr #14]       @ load Rn value
        tst r4, #1 << 23            @ U bit
        subne   r7, r0, r2
        addeq   r7, r0, r2
        b   Ldata_saver7

Ldata_lateldrpreconst:
        tst r4, #1 << 21            @ check writeback bit
        beq Ldata_simple
Ldata_lateldrpostconst:
        movs    r2, r4, lsl #20         @ Get offset
        beq Ldata_simple
        and r5, r4, #15 << 16       @ Get Rn
        ldr r0, [sp, r5, lsr #14]
        tst r4, #1 << 23            @ U bit
        subne   r7, r0, r2, lsr #20
        addeq   r7, r0, r2, lsr #20
        b   Ldata_saver7

Ldata_lateldrprereg:
        tst r4, #1 << 21            @ check writeback bit
        beq Ldata_simple
Ldata_lateldrpostreg:
        and r5, r4, #15
        ldr r2, [sp, r5, lsl #2]        @ Get Rm
        mov r3, r4, lsr #7
        ands    r3, r3, #31
        and r6, r4, #0x70
        orreq   r6, r6, #8
        add pc, pc, r6
        mov r0, r0

        mov r2, r2, lsl r3          @ 0: LSL #!0
        b   1f
        b   1f              @ 1: LSL #0
        mov r0, r0
        b   1f              @ 2: MUL?
        mov r0, r0
        b   1f              @ 3: MUL?
        mov r0, r0
        mov r2, r2, lsr r3          @ 4: LSR #!0
        b   1f
        mov r2, r2, lsr #32         @ 5: LSR #32
        b   1f
        b   1f              @ 6: MUL?
        mov r0, r0
        b   1f              @ 7: MUL?
        mov r0, r0
        mov r2, r2, asr r3          @ 8: ASR #!0
        b   1f
        mov r2, r2, asr #32         @ 9: ASR #32
        b   1f
        b   1f              @ A: MUL?
        mov r0, r0
        b   1f              @ B: MUL?
        mov r0, r0
        mov r2, r2, ror r3          @ C: ROR #!0
        b   1f
        mov r2, r2, rrx         @ D: RRX
        b   1f
        mov r0, r0              @ E: MUL?
        mov r0, r0
        mov r0, r0              @ F: MUL?


1:      and r5, r4, #15 << 16       @ Get Rn
        ldr r0, [sp, r5, lsr #14]
        tst r4, #1 << 23            @ U bit
        subne   r7, r0, r2
        addeq   r7, r0, r2
        b   Ldata_saver7

/*
 * Function: pt110_check_bugs (void)
 *     : pt110_proc_init (void)
 *     : pt110_proc_fin (void)
 *
 * Notes   : This processor does not require these
 */
ENTRY(cpu_pt110_check_bugs)
        mrs ip, cpsr
        bic ip, ip, #F_BIT
        msr cpsr, ip
        mov pc, lr

ENTRY(cpu_pt110_proc_init)
        mov pc, lr

ENTRY(cpu_pt110_proc_fin)
        stmfd   sp!, {lr}
        mov ip, #F_BIT | I_BIT | SVC_MODE
        msr cpsr_c, ip

        ldr r0, =0x80           @ 
        mcr p15, 0, r0, c1, c0, 0       @ disable caches
        mcr p15, 0, r1, c7, c10, 0      @ clean data cache (all)
        mcr p15, 0, r1, c7, c5, 0       @ flsuh instruction cache (all)
        mcr p15, 0, r1, c7, c6, 0       @ flush data cache (all)
        

        ldmfd   sp!, {pc}

/*
 * Function: pt110_proc_do_idle(void)
 * Params  : r0 = unused
 * Purpose : put the processer in proper idle mode
 */
ENTRY(cpu_pt110_do_idle)
        mov pc, lr

/*
 * Function: pt110_set_pgd(unsigned long pgd_phys)
 * Params  : pgd_phys   Physical address of page table
 * Purpose : Perform a task switch, saving the old process' state and restoring
 *       the new.
 */
ENTRY(cpu_pt110_set_pgd)
        mov pc, lr

/*
 * Function: pt110_set_pmd ()
 *
 * Params  : r0 = Address to set
 *     : r1 = value to set
 *
 * Purpose : Set a PMD and flush it out of any WB cache
 * FIXME: I have no idea what this is supposed to do or what a PMD is - FG
 */
ENTRY(cpu_pt110_set_pmd)
        tst r1, #3
        orrne   r1, r1, #16         @ Updatable bit is
        str r1, [r0]            @ always set on ARM720
        mov pc, lr

/*
 * Function: pt110_set_pte(pte_t *ptep, pte_t pte)
 * Params  : r0 = Address to set
 *     : r1 = value to set
 * Purpose : Set a PTE and flush it out of any WB cache
 */
        .align  5
ENTRY(cpu_pt110_set_pte)
        str r1, [r0], #-1024        @ linux version

        eor r1, r1, #LPTE_PRESENT | LPTE_YOUNG | LPTE_WRITE | LPTE_DIRTY

        bic r2, r1, #0xff0
        bic r2, r2, #3
        orr r2, r2, #HPTE_TYPE_SMALL

        tst r1, #LPTE_USER | LPTE_EXEC  @ User or Exec?
        orrne   r2, r2, #HPTE_AP_READ

        tst r1, #LPTE_WRITE | LPTE_DIRTY    @ Write and Dirty?
        orreq   r2, r2, #HPTE_AP_WRITE

        tst r1, #LPTE_PRESENT | LPTE_YOUNG  @ Present and Young
        movne   r2, #0

        str r2, [r0]            @ hardware version
        mov pc, lr

/*
 * Function: pt110_reset
 * Params  : r0 = address to jump to
 * Notes   : This sets up everything for a reset
 */
ENTRY(cpu_pt110_reset)
        mov ip, #0
    
        mcr p15, 0, ip, c7, c10, 0      @ clean data cache (all)
        mcr p15, 0, ip, c7, c5, 0       @ flush instruction cache (all)
        mcr p15, 0, ip, c7, c6, 0       @ flush data cache (all)

    
        mcr p15, 0, ip, c1, c0, 0       @ ctrl register little endian, no caches no

        mov pc, r0


cpu_armvlsi_name:
        .asciz  "ARM"
cpu_pt110_name:
        .asciz  "pt110"
        .align

        .section ".text.init", #alloc, #execinstr

__pt110_setup:  mov r0, #0
        mcr p15, 0, r0, c7, c10, 0      @ clean data cache (all)
        mcr p15, 0, r0, c7, c5, 0       @ flsuh instruction cache (all)
        mcr p15, 0, r0, c7, c6, 0       @ flush data cache (all)
#ifdef CONFIG_ARCH_EM86XX
#if 0
        mov r0, #0x1b                   @ 256 MB regions
        orr r0, r0, r0, lsl #8          @ Instruction and Data region size
        mcr p15, 0x0, r0, c6, c0, 0     @ Region Size register (CP15-6)
        ldr r0, =0xff1e                 @ Enable Data Cache, Region 1, 2, 3 (DRAM), and 4 (Host)
                                        @ Enable Instruction Cache
        mcr p15, 0x0, r0, c2, c0, 0     @ Cacheability Control register (CP15-2)

        mov r0, #0x0e                   @ Enable Write Buffering, Region 1, 2, 3 (DRAM)
        mcr p15, 0x0, r0, c3, c0, 0     @ Write Buffer Control register (CP15-3)

        ldr r0, =0xffff                 @ Read/write access in any mode
        mcr p15, 0x0, r0, c4, c0, 0     @ Instruction Space Protection register (CP15-4)

        ldr r0, =0xffff                 @ Read/write access in any mode
        mcr p15, 0x0, r0, c5, c0, 0     @ Data Space Protection register (CP15-5)

        mov r0, #0x00                   @ (little-endian)
        orr r0, r0, #0x1000             @ Instruction Cache Enable
        orr r0, r0, #0x0004             @ Data Cache Enable
        orr r0, r0, #0x0002             @ Write Buffering
        orr r0, r0, #0x0001             @ Protection Enable
#else
        mrc p15, 0, r0, c1, c0, 0       @ read current configuration
#endif
#else
        mov r0, #0x15
        orr r0, r0, r0, lsl #8          
        mcr p15, 0, r0, c6, c0, 0       @ set region size 0x15=4M (ICache and DCache)
        mov     r0, #0xF0           
        orr r0, r0, r0, lsl #8          
        mcr p15, 0, r0, c2, c0, 0       @ set Icache and Dcache for region 4,5,6,7 (SDRAM)
        mov     r0, #0xF0           
        mcr p15, 0, r0, c3, c0, 0       @ set Writeback for region 4,5,6,7 (SDRAM)
        mov     r0, #0xFF
        orr r0, r0, r0, lsl #8                          
        mcr p15, 0, r0, c4, c0, 0       @ set All Access for regions 0-7 (Icache)
        mcr p15, 0, r0, c5, c0, 0       @ set All Access for region 0-7 (Dcache)
        mov r0, #0x1000
        orr     r0, r0, #0x07           @ return control reg value in r0 - Little Endian, I&D caches, Write Buffer.
#endif

        mov pc, lr              @ __ret (head-armv.S)

/*
 * Purpose : Function pointers used to access above functions - all calls
 *       come through these
 */
        .type   pt110_processor_functions, #object
ENTRY(pt110_processor_functions)
        .word   cpu_pt110_data_abort
        .word   cpu_pt110_check_bugs
        .word   cpu_pt110_proc_init
        .word   cpu_pt110_proc_fin
        .word   cpu_pt110_reset
        .word   cpu_pt110_do_idle

        /* cache */
        .word   cpu_pt110_cache_clean_invalidate_all
        .word   cpu_pt110_cache_clean_invalidate_range
        .word   cpu_pt110_flush_ram_page

        /* dcache */
        .word   cpu_pt110_dcache_invalidate_range
        .word   cpu_pt110_dcache_clean_range
        .word   cpu_pt110_dcache_clean_page
        .word   cpu_pt110_dcache_clean_entry

        /* icache */
        .word   cpu_pt110_icache_invalidate_range
        .word   cpu_pt110_icache_invalidate_page

        /* tlb */
        .word   cpu_pt110_tlb_invalidate_all
        .word   cpu_pt110_tlb_invalidate_range
        .word   cpu_pt110_tlb_invalidate_page

        /* pgtable */
        .word   cpu_pt110_set_pgd
        .word   cpu_pt110_set_pmd
        .word   cpu_pt110_set_pte
        .size   pt110_processor_functions, . - pt110_processor_functions

        .type   cpu_pt110_info, #object
cpu_pt110_info:
        .long   cpu_armvlsi_name
        .long   cpu_pt110_name
        .size   cpu_pt110_info, . - cpu_pt110_info

        .type   cpu_arch_name, #object
cpu_arch_name:  .asciz  "armv4"
        .size   cpu_arch_name, . - cpu_arch_name

        .type   cpu_elf_name, #object
cpu_elf_name:   .asciz  "v4"
        .size   cpu_elf_name, . - cpu_elf_name
        .align

/*
 * See /include/asm-armnommu for a definition of this structure.
 */

        .section ".proc.info", #alloc, #execinstr

        .type   __pt110_proc_info, #object
__pt110_proc_info:
        .long   0x50021100              @ cpu_val
        .long   0xffffffff              @ cpu_mask
        .long   0x00000c1e              @ section_mmu_flags
        b   __pt110_setup               @ cpu_flush
        .long   cpu_arch_name               @ arch_name
        .long   cpu_elf_name                @ elf_name
        .long   HWCAP_SWP | HWCAP_HALF | HWCAP_26BIT    @ elf_hwcap
        .long   cpu_pt110_info              @ info
        .long   pt110_processor_functions
        .size   __pt110_proc_info, . - __pt110_proc_info
