/*****************************************************************************/

/*
 *	ht1381.c -- driver for Dallas Semiconductor HT1381 Real Time Clock.
 *
 * 	(C) Copyright 2001, Greg Ungerer (gerg@snapgear.com)
 */

/*****************************************************************************/

#include <linux/config.h>
#include <linux/types.h>
#include <linux/miscdevice.h>
#include <linux/init.h>
#include <linux/module.h> 
#include <linux/kernel.h>
#include <linux/fs.h>
#include <linux/mm.h>
#include <linux/wait.h>
#include <asm/param.h>
#include <linux/rtc.h>
#include <linux/proc_fs.h>
#include <asm/uaccess.h> 

#include <linux/mc146818rtc.h>
#include <linux/rtc.h>
#include <linux/types.h>

#include "ht1381.h"
#include "pca9555.h"

extern int (*set_rtc)(void);
extern unsigned long (*rtc_get_time)(void);
extern int (*rtc_set_time)(unsigned long);

#define RTC_MONTH_MASK          0x1f
#define RTC_SECONDS_MASK        0x7f

/*
 * CMOS Year Epoch
 */
#define EPOCH                   2000    

/*****************************************************************************/

/*
 *	Size of RTC region. 32 bytes of calender and 32 bytes RAM.
 *	Not all addresses are valid in this range...
 */
#define	HT1381_MSIZE	0x07

/*****************************************************************************/

/*
 *	HT1381 defines.
 */
#define	RTC_CMD_READ	0x81		/* Read command */
#define	RTC_CMD_WRITE	0x80		/* Write command */

#define RTC_ADDR_WP	0x07
#define	RTC_ADDR_YEAR	0x06		/* Address of year register */
#define	RTC_ADDR_DAY	0x05		/* Address of day of week register */
#define	RTC_ADDR_MON	0x04		/* Address of month register */
#define	RTC_ADDR_DATE	0x03		/* Address of day of month register */
#define	RTC_ADDR_HOUR	0x02		/* Address of hour register */
#define	RTC_ADDR_MIN	0x01		/* Address of minute register */
#define	RTC_ADDR_SEC	0x00		/* Address of second register */

/*****************************************************************************/

#define IOEXT_RTC_CLK           0x0100
#define IOEXT_RTC_RST           0x0200
#define IOEXT_RTC_DATA          0x0400
		
static inline void UDELAY(int usec){
        usec*=10;
        while(usec)
                usec--;

}

#if 1
#define FEBRUARY                2 
#define STARTOFTIME             1970
#define SECDAY                  86400L
#define SECYR                   (SECDAY * 365)
#define leapyear(y)             ((!((y) % 4) && ((y) % 100)) || !((y) % 400))
#define days_in_year(y)         (leapyear(y) ? 366 : 365)
#define days_in_month(m)        (month_days[(m) - 1])

static int month_days[12] = {
        31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31
};

void to_tm(unsigned long tim, struct rtc_time *tm)
{
        long hms, day, gday;
        int i;

        gday = day = tim / SECDAY;
        hms = tim % SECDAY;

        /* Hours, minutes, seconds are easy */
        tm->tm_hour = hms / 3600;
        tm->tm_min = (hms % 3600) / 60;
        tm->tm_sec = (hms % 3600) % 60;

        /* Number of years in days */
        for (i = STARTOFTIME; day >= days_in_year(i); i++)
                day -= days_in_year(i);
        tm->tm_year = i;
        /* Number of months in days left */
        if (leapyear(tm->tm_year))
                days_in_month(FEBRUARY) = 29;
        for (i = 1; day >= days_in_month(i); i++)
                day -= days_in_month(i);
        days_in_month(FEBRUARY) = 28;
        tm->tm_mon = i - 1;             /* tm_mon starts from 0 to 11 */

        /* Days are what is left over (+1) from all that. */
        tm->tm_mday = day + 1;

        /*
         * Determine the day of week
         */
        tm->tm_wday = (gday + 4) % 7;   /* 1970/1/1 was Thursday */
}
#endif

void ht1381_sendbits(unsigned int val)
{
	int	i;
	int	bit;
	//printk("sending\n");
	//printk("val = 0x%x\n", val);
	for (i = 8; (i); i--, val >>= 1) 
	{
		//*mcf5249_gpio_out = (*mcf5249_gpio_out & ~RTC_IODATA) |
		//	 ((val & 0x1) ? RTC_IODATA : 0);
		bit = ((val & 0x1) ? IOEXT_RTC_DATA : 0);
		//printk("bit = 0x%x\n", bit);
		pca9555_reg_get_set(IOEXT_RTC_DATA, bit, 1);
		//*mcf5249_gpio_out |= RTC_SCLK;		// clock high
		pca9555_reg_get_set(IOEXT_RTC_CLK, IOEXT_RTC_CLK, 1);
		UDELAY(100);
		//*mcf5249_gpio_out &= ~RTC_SCLK;	// clock low
		pca9555_reg_get_set(IOEXT_RTC_CLK, 0, 1);
	}
}

unsigned int ht1381_recvbits(void)
{
	unsigned int	val;
	int		i;
	unsigned short ret;

	for (i = 0, val = 0; (i < 8); i++) {
		//val |= (((*mcf5249_gpio_read & RTC_IODATA) ? 1 : 0) << i);
		ret = pca9555_reg_get_set(IOEXT_RTC_DATA, 0, 0);
		//printk("ret = 0x%x\n", ret);
                if (ret)
                        val |= (1<<i);

		//*mcf5249_gpio_out |= RTC_SCLK;		// clock high
		pca9555_reg_get_set(IOEXT_RTC_CLK, IOEXT_RTC_CLK, 1);
		UDELAY(100);
		//*mcf5249_gpio_out &= ~RTC_SCLK;	// clock low
		pca9555_reg_get_set(IOEXT_RTC_CLK, 0, 1);
	}
	//printk("val = 0x%x\n", val);
	return(val);
}

unsigned int ht1381_readbyte(unsigned int addr)
{
	unsigned int	val;
	unsigned long	flags;

#if 0
	printk("ht1381_readbyte(addr=%x)\n", addr);
#endif

	//save_flags(flags); cli();
	//*mcf5249_gpio_en |= RTC_RESET | RTC_IODATA | RTC_SCLK;
	//*mcf5249_gpio_out &= ~(RTC_RESET | RTC_IODATA | RTC_SCLK);
	pca9555_reg_get_set(IOEXT_RTC_CLK|IOEXT_RTC_RST|IOEXT_RTC_DATA, 0, 1);

	//*mcf5249_gpio_out |= RTC_RESET;
	pca9555_reg_get_set(IOEXT_RTC_RST, IOEXT_RTC_RST, 1);
	ht1381_sendbits(((addr & 0x3f) << 1) | RTC_CMD_READ);
	//*mcf5249_gpio_en &= ~RTC_IODATA;

	val = ht1381_recvbits();
//	printk("readbyte=%x\n",val);
	//*mcf5249_gpio_out &= ~RTC_RESET;
	pca9555_reg_get_set(IOEXT_RTC_RST, 0, 1);
	//restore_flags(flags);

	return(val);
}

void ht1381_writebyte(unsigned int addr, unsigned int val)
{
	unsigned long	flags;

#if 0
	printk("ht1381_writebyte(addr=%x)\n", addr);
#endif

	//save_flags(flags); cli();
	//*mcf5249_gpio_en |= RTC_RESET | RTC_IODATA | RTC_SCLK;
	//*mcf5249_gpio_out &= ~(RTC_RESET | RTC_IODATA | RTC_SCLK);
	pca9555_reg_get_set(IOEXT_RTC_CLK|IOEXT_RTC_RST|IOEXT_RTC_DATA, 0, 1);

	//*mcf5249_gpio_out |= RTC_RESET;
	pca9555_reg_get_set(IOEXT_RTC_RST, IOEXT_RTC_RST, 1);
	ht1381_sendbits(((addr & 0x3f) << 1) | RTC_CMD_WRITE);
	ht1381_sendbits(val);
	//*mcf5249_gpio_out &= ~RTC_RESET;
	pca9555_reg_get_set(IOEXT_RTC_RST, 0, 1);
	//restore_flags(flags);
}


/*****************************************************************************/

static inline int bcd2int(int val)
{
        return((((val & 0xf0) >> 4) * 10) + (val & 0xf));
}

static inline int int2bcd(int val)
{
        return(((val / 10) << 4) + (val % 10));
}


/*****************************************************************************/

static ssize_t ht1381_read(struct file *fp, char *buf, size_t count, loff_t *ptr)
{
	int	total;

#if 0
	printk("ht1381_read(buf=%x,count=%d)\n", (int) buf, count);
#endif

	if (fp->f_pos >= HT1381_MSIZE)
		return(0);
	if (count > (HT1381_MSIZE - fp->f_pos))
		count = HT1381_MSIZE - fp->f_pos;

	for (total = 0; (total < count); total++)
		put_user(ht1381_readbyte(fp->f_pos + total), buf++);

	fp->f_pos += total;
	return(total);
}

/*****************************************************************************/

static ssize_t ht1381_write(struct file *fp, const char *buf, size_t count, loff_t *ptr)
{
	int	total;
	char val;

#if 0
	printk("ht1381_write(buf=%x,count=%d)\n", (int) buf, count);
#endif

	if (fp->f_pos >= HT1381_MSIZE)
		return(0);
	if (count > (HT1381_MSIZE - fp->f_pos))
		count = HT1381_MSIZE - fp->f_pos;

	for (total = 0; (total < count); total++, buf++)
	{
		get_user(val,buf);
		ht1381_writebyte((fp->f_pos + total), val);
	}

	fp->f_pos += total;
	return(total);
}

static int ht1381_read_proc(char *page, char **start, off_t off, int count,
                                int *eof, void *data)
{
	int len;
        len = get_ht1381_status(page);
        if (len <= off + count)
                *eof = 1;
        *start = page + off;
        len -= off;
        if (len > count)
                len = count;
        if (len < 0)
                len = 0;
        return len;
}

unsigned long rtc_ht1381_get_time(void)
{
        unsigned int year, month, day, hour, minute, second;

        second = bcd2int(ht1381_readbyte(RTC_ADDR_SEC));
        minute = bcd2int(ht1381_readbyte(RTC_ADDR_MIN));
        hour = bcd2int(ht1381_readbyte(RTC_ADDR_HOUR));
        day = bcd2int(ht1381_readbyte(RTC_ADDR_DATE));
        month = bcd2int(ht1381_readbyte(RTC_ADDR_MON));
        year = bcd2int(ht1381_readbyte(RTC_ADDR_YEAR));

        return mktime(year+2000, month, day, hour, minute, second);

}

static int rtc_ht1381_set_time(unsigned long t)
{
        struct rtc_time tm;
        u8 year, month, day, hour, minute, second;
        u8 cmos_year, cmos_month, cmos_day, cmos_hour, cmos_minute, cmos_second;
        int cmos_century;

        cmos_second = (u8)ht1381_readbyte(RTC_ADDR_SEC);
        cmos_minute = (u8)ht1381_readbyte(RTC_ADDR_MIN);
        cmos_hour = (u8)ht1381_readbyte(RTC_ADDR_HOUR);
        cmos_day = (u8)ht1381_readbyte(RTC_ADDR_DATE);
        cmos_month = (u8)ht1381_readbyte(RTC_ADDR_MON);
        cmos_year = (u8)ht1381_readbyte(RTC_ADDR_YEAR);
	printk("cmos_year = %x\n",cmos_year);


        /* convert */
        to_tm(t, &tm);

        /* check each field one by one */
	printk("tm.tm_year = %d\n",tm.tm_year);
        year = int2bcd(tm.tm_year - EPOCH);
	printk("year = %x\n",year);
        if (year != cmos_year) {   
                ht1381_writebyte(RTC_ADDR_YEAR,year);
        }                          
                                   
        month = int2bcd(tm.tm_mon + 1);
        if (month != (cmos_month & 0x1f)) {
                ht1381_writebyte(RTC_ADDR_MON,(month & 0x1f) | (cmos_month & ~0x1f));
        }       

        day = int2bcd(tm.tm_mday);
        if (day != cmos_day)
                ht1381_writebyte(RTC_ADDR_DATE,day);

        if (cmos_hour & 0x80) {
                /* 12 hour format */
                hour = 0x80;
                if (tm.tm_hour > 12) {
                        hour |= 0x20 | (int2bcd(hour-12) & 0x1f);
                } else {
                        hour |= int2bcd(tm.tm_hour);
                }
        } else {
                /* 24 hour format */
                hour = int2bcd(tm.tm_hour) & 0x3f;
        }
        if (hour != cmos_hour) ht1381_writebyte(RTC_ADDR_HOUR,hour);

        minute = int2bcd(tm.tm_min);
        if (minute !=  cmos_minute) {
                ht1381_writebyte(RTC_ADDR_MIN,minute);
        }

        second = int2bcd(tm.tm_sec);
        if (second !=  cmos_second) {
                ht1381_writebyte(RTC_ADDR_SEC,second & RTC_SECONDS_MASK);
        }

        return 0;
}

static int get_ht1381_status(char *buf)
{
        char *p;
        struct rtc_time tm;
        unsigned long curr_time;

        curr_time = rtc_ht1381_get_time();
        to_tm(curr_time, &tm);

        p = buf;
        
        /*
         * There is no way to tell if the luser has the RTC set for local
         * time or for Universal Standard Time (GMT). Probably local though.
         */                     
        p += sprintf(p,
                     "rtc_time\t: %02d:%02d:%02d\n"
                     "rtc_date\t: %04d-%02d-%02d\n"
                     "rtc_epoch\t: %04lu\n",
                     tm.tm_hour, tm.tm_min, tm.tm_sec,
                     tm.tm_year, tm.tm_mon + 1, tm.tm_mday, 0L);
                
        return p - buf;
}       



static int ht1381_ioctl(struct inode *inode, struct file *file,
                                unsigned int cmd, unsigned long arg)
{       
        struct rtc_time rtc_tm;
        ulong curr_time;

        switch (cmd) {
        case RTC_RD_TIME:       /* Read the time/date from RTC  */
                curr_time = rtc_ht1381_get_time();
                to_tm(curr_time, &rtc_tm);
                rtc_tm.tm_year -= 1900;
                return copy_to_user((void *) arg, &rtc_tm, sizeof(rtc_tm)) ?
                        -EFAULT : 0;
        case RTC_SET_TIME:      /* Set the RTC */
                if (!capable(CAP_SYS_TIME))
                        return -EACCES;
                                   
                if (copy_from_user(&rtc_tm, 
                                   (struct rtc_time *) arg,
                                   sizeof(struct rtc_time)))
                        return -EFAULT;

                curr_time = mktime(rtc_tm.tm_year + 1900,
                                   rtc_tm.tm_mon + 1, /* tm_mon starts from 0 */
                                   rtc_tm.tm_mday,
                                   rtc_tm.tm_hour,
                                   rtc_tm.tm_min,
                                   rtc_tm.tm_sec);
                return rtc_ht1381_set_time(curr_time);
        default:
                return -EINVAL;
        }
}       

/*****************************************************************************/

/*
 *	Exported file operations structure for driver...
 */

static struct file_operations ht1381_fops =
{
	owner:	THIS_MODULE, 
	read:	ht1381_read,
	ioctl:	ht1381_ioctl,
	write:	ht1381_write,
};

static struct miscdevice ht1381_dev =
{
	RTC_MINOR,
	"rtc",
	&ht1381_fops
};

/*****************************************************************************/

void ht1381_reset(void)
{
	/* Hardware dependant reset/init */
	//*(volatile unsigned long *)(0x80000000+0x0bc)|=RTC_RESET | RTC_IODATA | RTC_SCLK;

	//*mcf5249_gpio_en |= RTC_RESET | RTC_IODATA | RTC_SCLK;
	//*mcf5249_gpio_out &= ~(RTC_RESET | RTC_IODATA | RTC_SCLK);
	pca9555_reg_get_set(IOEXT_RTC_CLK|IOEXT_RTC_RST|IOEXT_RTC_DATA, 0, 1);
#if 1
	/* Init */
	// disable write protect
	ht1381_writebyte(RTC_ADDR_WP,0x0);	// register7 set to zero 
	// enable the oscillator
	ht1381_writebyte(RTC_ADDR_SEC,0x0);	// register0 set to zero 
#endif	
#if 0
        /* set the function pointers */
        rtc_get_time = rtc_ht1381_get_time;
        rtc_set_time = rtc_ht1381_set_time;
#endif

}

int ht1381_init(void)
{
	misc_register(&ht1381_dev);

	printk ("HT1381: Copyright (C) 2001, Greg Ungerer "
		"(gerg@snapgear.com)\n");

	ht1381_reset();

        create_proc_read_entry("driver/rtc", 0, 0, ht1381_read_proc, NULL);

        xtime.tv_sec = rtc_ht1381_get_time();
#ifdef CONFIG_ALPHA_RTC	
        rtc_set_time = rtc_ht1381_set_time;
#endif	
	return 0;
}

void ht1381_exit(void)
{
        remove_proc_entry("driver/rtc", NULL);
        misc_deregister(&ht1381_dev);

}

