/*
 *  drivers/char/serial_em86xx.c
 *
 *  Driver for EM86XX serial ports
 *
 *  Based on drivers/char/serial.c, by Linus Torvalds, Theodore Ts'o.
 *
 *  Copyright 1999 ARM Limited
 *  Copyright (C) 2000 Deep Blue Solutions Ltd.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  $Id: serial_em86xx.c,v 1.1.2.1 2006/03/10 12:31:22 william Exp $
 */

/*
 * This is a generic driver for EM86XX and EM86XX serial ports.
 * Based on serial_amba.c
 */

#include <linux/config.h>
#include <linux/module.h>
#include <linux/errno.h>
#include <linux/signal.h>
#include <linux/sched.h>
#include <linux/interrupt.h>
#include <linux/tty.h>
#include <linux/tty_flip.h>
#include <linux/major.h>
#include <linux/string.h>
#include <linux/fcntl.h>
#include <linux/ptrace.h>
#include <linux/ioport.h>
#include <linux/mm.h>
#include <linux/slab.h>
#include <linux/init.h>
#include <linux/circ_buf.h>
#include <linux/serial.h>
#include <linux/console.h>
#include <linux/sysrq.h>
#include <linux/version.h>

#include <asm/system.h>
#include <asm/io.h>
#include <asm/irq.h>
#include <asm/uaccess.h>
#include <asm/bitops.h>
#include <asm/hardware.h>

#if defined(CONFIG_SERIAL_EM86XX_CONSOLE) && defined(CONFIG_MAGIC_SYSRQ)
#define SUPPORT_SYSRQ
#endif

#include <linux/serial_core.h>

#include <asm/hardware/serial_em86xx.h>

#define SERIAL_EM86XX_MAJOR	204
#define SERIAL_EM86XX_MINOR	16
#define SERIAL_EM86XX_NR	UART_NR

#define CALLOUT_EM86XX_NAME	"cuaam"
#define CALLOUT_EM86XX_MAJOR	205
#define CALLOUT_EM86XX_MINOR	16
#define CALLOUT_EM86XX_NR	UART_NR

static struct tty_driver normal, callout;
static struct tty_struct *em86xx_table[UART_NR];
static struct termios *em86xx_termios[UART_NR], *em86xx_termios_locked[UART_NR];
#ifdef SUPPORT_SYSRQ
static struct console em86xx_console;
#endif

#define EM86XX_ISR_PASS_LIMIT	256

/*
 * Access macros for the EM86XX UARTs
 */

#define UART_GET_IER(p)		inl((p)->membase + EM86XX_UART_IER)
#define UART_PUT_IER(p,c)	outl((c), (p)->membase + EM86XX_UART_IER)
#define UART_GET_MSR(p)		inl((p)->membase + EM86XX_UART_MSR)
#define UART_GET_LSR(p)		inl((p)->membase + EM86XX_UART_LSR)
#define UART_GET_IIR(p)		inl((p)->membase + EM86XX_UART_IIR)
#define UART_GET_RBR(p)		inl((p)->membase + EM86XX_UART_RBR)
#define UART_PUT_TBR(p,c)	outl((c), (p)->membase + EM86XX_UART_TBR)
#define UART_GET_LCR(p)		inl((p)->membase + EM86XX_UART_LCR)
#define UART_PUT_LCR(p,c)	outl((c), (p)->membase + EM86XX_UART_LCR)
#define UART_GET_FCR(p)		inl((p)->membase + EM86XX_UART_FCR)
#define UART_PUT_FCR(p,c)	outl((c), (p)->membase + EM86XX_UART_FCR)
#define UART_GET_MCR(p)		inl((p)->membase + EM86XX_UART_MCR)
#define UART_PUT_MCR(p,c)	outl((c), (p)->membase + EM86XX_UART_MCR)
#define UART_GET_CLKDIV(p)	inl((p)->membase + EM86XX_UART_CLKDIV)
#define UART_PUT_CLKDIV(p,c)	outl((c), (p)->membase + EM86XX_UART_CLKDIV)
#define UART_PUT_CLKSEL(p,c)    outl((c), (p)->membase + EM86XX_UART_CLKSEL)

#define UART_GET_INT_STATUS(p)	(UART_GET_IIR(p) & 0xF)
#define UART_GET_CHAR(p)	UART_GET_RBR(p)
#define UART_PUT_CHAR(p, c)	UART_PUT_TBR(p, c)

/*
#define UART_GET_LCRL(p)	readb((p)->membase + EM86XX_UARTLCR_L)
#define UART_PUT_LCRL(p,c)	writel((c), (p)->membase + EM86XX_UARTLCR_L)
#define UART_GET_LCRM(p)	readb((p)->membase + EM86XX_UARTLCR_M)
#define UART_PUT_LCRM(p,c)	writel((c), (p)->membase + EM86XX_UARTLCR_M)
#define UART_GET_LCRH(p)	readb((p)->membase + EM86XX_UARTLCR_H)
#define UART_PUT_LCRH(p,c)	writel((c), (p)->membase + EM86XX_UARTLCR_H)
#define UART_RX_DATA(s)		(((s) & EM86XX_UARTFR_RXFE) == 0)
#define UART_TX_READY(s)	(((s) & EM86XX_UARTFR_TXFF) == 0)
#define UART_TX_EMPTY(p)	((UART_GET_FR(p) & EM86XX_UARTFR_TMSK) == 0)
*/

#define UART_RX_READY(s)	((s)& EM86XX_UART_LSR_DR)
#define UART_TX_READY(s)	((s)& EM86XX_UART_LSR_THRE) // Tx holding reg is empty
#define UART_TX_EMPTY(s)	((s)& EM86XX_UART_LSR_TEMT) // Tx shift reg is empty

#define UART_DUMMY_RSR_RX	256
#define UART_PORT_SIZE		64

#define RX_FIFO_TRIG_DEFAULT    8
#define TX_FIFO_TRIG_DEFAULT	8
#define RX_FIFO_TRIG(port)		((port)->unused[0])
#define TX_FIFO_TRIG(port)		((port)->unused[1])


/*
 * Our private driver data mappings.
 */
#define drv_old_status	driver_priv

static void em86xx_uart_stop_tx(struct uart_port *port, u_int from_tty)
{
	unsigned int ier;

	ier = UART_GET_IER(port);
	ier &= ~EM86XX_UART_IER_ETBEI;
	UART_PUT_IER(port, ier);
}

static void em86xx_uart_start_tx(struct uart_port *port, u_int nonempty, u_int from_tty)
{
	if (nonempty) {
		unsigned int ier;

		ier = UART_GET_IER(port);
		UART_PUT_IER(port, 0);  // Need an edge to reenable transmit interrupt if it wasnt disabled
		ier |= EM86XX_UART_IER_ETBEI;
		UART_PUT_IER(port, ier);
	}
}

static void em86xx_uart_stop_rx(struct uart_port *port)
{
	unsigned int ier;

	ier = UART_GET_IER(port);
	ier &= ~(EM86XX_UART_IER_ERBFI);
	UART_PUT_IER(port, ier);
}

static void em86xx_uart_enable_ms(struct uart_port *port)
{
	unsigned int ier;

	ier = UART_GET_IER(port);
	ier |= EM86XX_UART_IER_EDSSI;
	UART_PUT_IER(port, ier);
}

static void
#ifdef SUPPORT_SYSRQ
em86xx_uart_rx_chars(struct uart_info *info, struct pt_regs *regs)
#else
em86xx_uart_rx_chars(struct uart_info *info)
#endif
{
	struct tty_struct *tty = info->tty;
	unsigned int status, ch, max_count = 256;
	struct uart_port *port = info->port;

	status = UART_GET_LSR(port);
	while (UART_RX_READY(status) && max_count--) {
		if (tty->flip.count >= TTY_FLIPBUF_SIZE) {
			tty->flip.tqueue.routine((void *)tty);
			if (tty->flip.count >= TTY_FLIPBUF_SIZE) {
				printk(KERN_WARNING "TTY_DONT_FLIP set\n");
				return;
			}
		}

		ch = UART_GET_CHAR(port);

		*tty->flip.char_buf_ptr = ch;
		*tty->flip.flag_buf_ptr = TTY_NORMAL;
		port->icount.rx++;

		/*
		 * Note that the error handling code is
		 * out of the main execution path
		 */
		if (status & EM86XX_UART_LSR_ANY) {
			if (status & EM86XX_UART_LSR_BI) {
				status &= ~(EM86XX_UART_LSR_FE | EM86XX_UART_LSR_PE);
				port->icount.brk++;
				if (uart_handle_break(info, &em86xx_console))
					goto ignore_char;
			} else if (status & EM86XX_UART_LSR_PE)
				port->icount.parity++;
			else if (status & EM86XX_UART_LSR_FE)
				port->icount.frame++;
			if (status & EM86XX_UART_LSR_OE)
				port->icount.overrun++;

			status &= port->read_status_mask;

			if (status & EM86XX_UART_LSR_BI)
				*tty->flip.flag_buf_ptr = TTY_BREAK;
			else if (status & EM86XX_UART_LSR_PE)
				*tty->flip.flag_buf_ptr = TTY_PARITY;
			else if (status & EM86XX_UART_LSR_FE)
				*tty->flip.flag_buf_ptr = TTY_FRAME;
		}

		if (uart_handle_sysrq_char(info, ch, regs))
			goto ignore_char;

		if ((status & port->ignore_status_mask) == 0) {
			tty->flip.flag_buf_ptr++;
			tty->flip.char_buf_ptr++;
			tty->flip.count++;
		}
		if ((status & EM86XX_UART_LSR_OE) &&
		    tty->flip.count < TTY_FLIPBUF_SIZE) {
			/*
			 * Overrun is special, since it's reported
			 * immediately, and doesn't affect the current
			 * character
			 */
			*tty->flip.char_buf_ptr++ = 0;
			*tty->flip.flag_buf_ptr++ = TTY_OVERRUN;
			tty->flip.count++;
		}
	ignore_char:
		status = UART_GET_LSR(port);
	}
	tty_flip_buffer_push(tty);
	return;
}

/*
static char debug_str[9];
static char debug_ref[8]="FEEDME_W";
static char debug_pos=0;
*/

static void em86xx_uart_tx_chars(struct uart_info *info)
{
	struct uart_port *port = info->port;
	int count;

	if (port->x_char) {
		UART_PUT_CHAR(port, port->x_char);
		port->icount.tx++;
		port->x_char = 0;
		return;
	}
	if (info->xmit.head == info->xmit.tail
	    || info->tty->stopped
	    || info->tty->hw_stopped) {
		em86xx_uart_stop_tx(port, 0);
		return;
	}

	count = port->fifosize >> 1;
	do {
		UART_PUT_CHAR(port, info->xmit.buf[info->xmit.tail]);
		info->xmit.tail = (info->xmit.tail + 1) & (UART_XMIT_SIZE - 1);
		port->icount.tx++;
		if (info->xmit.head == info->xmit.tail)
			break;
	} while (--count > 0);

	if (CIRC_CNT(info->xmit.head, info->xmit.tail, UART_XMIT_SIZE) <
			WAKEUP_CHARS)
		uart_event(info, EVT_WRITE_WAKEUP);

	if (info->xmit.head == info->xmit.tail)
		em86xx_uart_stop_tx(info->port, 0);
}

static void em86xx_uart_modem_status(struct uart_info *info)
{
	struct uart_port *port = info->port;
	unsigned int status, delta;

	status = UART_GET_MSR(port);

	delta = status ^ info->drv_old_status;
	info->drv_old_status = status;

	if (!delta)
		return;

	if (delta & EM86XX_UART_MSR_DCD)
		uart_handle_dcd_change(info, status & EM86XX_UART_MSR_DCD);

	if (delta & EM86XX_UART_MSR_DSR)
		port->icount.dsr++;

	if (delta & EM86XX_UART_MSR_CTS)
		uart_handle_cts_change(info, status & EM86XX_UART_MSR_CTS);

	wake_up_interruptible(&info->delta_msr_wait);
}

static void em86xx_uart_int(int irq, void *dev_id, struct pt_regs *regs)
{
	struct uart_info *info = dev_id;
	unsigned int status, pass_counter = EM86XX_ISR_PASS_LIMIT;

	status = UART_GET_INT_STATUS(info->port);
	do {
#if 1
            	if ((((status >> 1) & UART_INT_RX_DATA) == UART_INT_RX_DATA) ||
            		(((status >> 1) & UART_INT_RX_TIMEOUT) == UART_INT_RX_TIMEOUT)) {
#ifdef SUPPORT_SYSRQ
			em86xx_uart_rx_chars(info, regs);
#else
			em86xx_uart_rx_chars(info);
#endif
		}
            	if (((status >> 1) & UART_INT_TX_EMPTY) == UART_INT_TX_EMPTY) {
			em86xx_uart_tx_chars(info);
		}
            	if (((status >> 1) & UART_INT_MODEM) == UART_INT_MODEM) {
			em86xx_uart_modem_status(info);
		}
#else
		switch(status>>1)	{
		case UART_INT_RX_TIMEOUT:
		case UART_INT_RX_DATA:
#ifdef SUPPORT_SYSRQ
			em86xx_uart_rx_chars(info, regs);
#else
			em86xx_uart_rx_chars(info);
#endif
			break;
		case UART_INT_TX_EMPTY:
			em86xx_uart_tx_chars(info);
			break;
		case UART_INT_MODEM:
			em86xx_uart_modem_status(info);
			break;
		default:
            		if (((status >> 1) & UART_INT_RX_DATA) == UART_INT_RX_DATA) {
#ifdef SUPPORT_SYSRQ
				em86xx_uart_rx_chars(info, regs);
#else
				em86xx_uart_rx_chars(info);
#endif
			}
            		if (((status >> 1) & UART_INT_TX_EMPTY) == UART_INT_TX_EMPTY) 
				em86xx_uart_tx_chars(info);
			break;

		}
#endif
		if (pass_counter-- == 0)
			break;
		status = UART_GET_INT_STATUS(info->port);
	} while (!status & UART_INT_PENDING);
}

static u_int em86xx_uart_tx_empty(struct uart_port *port)
{
	return UART_TX_EMPTY(UART_GET_LSR(port)) ? TIOCSER_TEMT : 0;
}

static int em86xx_uart_get_mctrl(struct uart_port *port)
{
	unsigned int result = 0;
	unsigned int status;

	status = UART_GET_MSR(port);
	if (status & EM86XX_UART_MSR_DCD)
		result |= TIOCM_CAR;
	if (status & EM86XX_UART_MSR_DSR)
		result |= TIOCM_DSR;
	if (status & EM86XX_UART_MSR_CTS)
		result |= TIOCM_CTS;

	/* Other modem lines not handled (RI,LOOP, OUT1 OUT2)*/

	return result;
}

static void em86xx_uart_set_mctrl(struct uart_port *port, u_int mctrl)
{
	u_int mcr = UART_GET_MCR(port);

	if (mctrl & TIOCM_RTS)
		mcr |= EM86XX_UART_MCR_RTS;

	if (mctrl & TIOCM_DTR)
		mcr |= EM86XX_UART_MCR_DTR;

	UART_PUT_MCR(port, mcr);
}

static void em86xx_uart_break_ctl(struct uart_port *port, int break_state)
{
	unsigned int lcr;

	lcr = UART_GET_LCR(port);
	if (break_state == -1)
		lcr |= EM86XX_UART_LCR_SETBRK;
	else
		lcr &= ~EM86XX_UART_LCR_SETBRK;
	UART_PUT_LCR(port, lcr);
}

static int em86xx_uart_startup(struct uart_port *port, struct uart_info *info)
{
	int retval;

	/*
	 * Allocate the IRQ
	 */
	retval = request_irq(port->irq, em86xx_uart_int, 0, "EM86XX UART", info);
	if (retval)
		return retval;

	/*
	 * initialise the old status of the modem signals
	 */
	info->drv_old_status = UART_GET_MSR(port);

	/*
	 * Finally, enable interrupts
	 * FIXME : Modem interrupt (EDSSI) 
	 */
	UART_PUT_IER(port, EM86XX_UART_IER_ERBFI | EM86XX_UART_IER_ETBEI | EM86XX_UART_IER_ELSI );

	return 0;
}

static void em86xx_uart_shutdown(struct uart_port *port, struct uart_info *info)
{
	/*
	 * Free the interrupt
	 */
	free_irq(port->irq, info);

	/*
	 * disable all interrupts, disable the port
	 */
	UART_PUT_IER(port, 0);

	/* disable break condition and fifos */
	UART_PUT_LCR(port, UART_GET_LCR(port) &	~(EM86XX_UART_LCR_SETBRK));
	UART_PUT_FCR(port, UART_GET_FCR(port) &	~(EM86XX_UART_FCR_FIFOENA));
}

static void em86xx_uart_change_speed(struct uart_port *port, u_int cflag, u_int iflag, u_int quot)
{
	u_int lcr, fcr, old_ier;
	unsigned long flags;

	/* byte size and parity */
	switch (cflag & CSIZE) {
	case CS5: lcr = EM86XX_UART_LCR_WLEN_5; break;
	case CS6: lcr = EM86XX_UART_LCR_WLEN_6; break;
	case CS7: lcr = EM86XX_UART_LCR_WLEN_7; break;
	default:  lcr = EM86XX_UART_LCR_WLEN_8; break; // CS8
	}
	if (cflag & CSTOPB)
		lcr |= EM86XX_UART_LCR_STP2;
	if (cflag & PARENB) {
		lcr |= EM86XX_UART_LCR_PEN;
		if (!(cflag & PARODD))
			lcr |= EM86XX_UART_LCR_EPS;
	}
	fcr = EM86XX_UART_FCR_TXRST | EM86XX_UART_FCR_RXRST;
	if (port->fifosize > 1)
		fcr |= EM86XX_UART_FCR_FIFOENA; // XXX: TX irq trigger

	UART_PUT_LCR(port, lcr);
	UART_PUT_FCR(port, fcr);
	
	port->read_status_mask = EM86XX_UART_LSR_OE;
	if (iflag & INPCK)
		port->read_status_mask |= EM86XX_UART_LSR_FE | EM86XX_UART_LSR_PE;
	if (iflag & (BRKINT | PARMRK))
		port->read_status_mask |= EM86XX_UART_LSR_BI;

	/*
	 * Characters to ignore
	 */
	port->ignore_status_mask = 0;
	if (iflag & IGNPAR)
		port->ignore_status_mask |= EM86XX_UART_LSR_FE | EM86XX_UART_LSR_PE;
	if (iflag & IGNBRK) {
		port->ignore_status_mask |= EM86XX_UART_LSR_BI;
		/*
		 * If we're ignoring parity and break indicators,
		 * ignore overruns too (for real raw support).
		 */
		if (iflag & IGNPAR)
			port->ignore_status_mask |= EM86XX_UART_LSR_OE;
	}

	/*
	 * Ignore all characters if CREAD is not set.
	 */
	if ((cflag & CREAD) == 0)
		port->ignore_status_mask |= UART_DUMMY_RSR_RX;

	/* first, disable everything */
	save_flags(flags); cli();
	old_ier = UART_GET_IER(port) & ~EM86XX_UART_IER_EDSSI;

	if ((port->flags & ASYNC_HARDPPS_CD) ||
	    (cflag & CRTSCTS) || !(cflag & CLOCAL))
		old_ier |= EM86XX_UART_IER_EDSSI;

	UART_PUT_IER(port, 0);

	/* Set baud rate */
	UART_PUT_CLKDIV(port, quot);
#ifdef CONFIG_USE_SYSCLK
	UART_PUT_CLKSEL(port, 0);
#else
	UART_PUT_CLKSEL(port, 1);
#endif
	UART_PUT_IER(port, old_ier);

	restore_flags(flags);
}

/*
 * Release the memory region(s) being used by 'port'
 */
static void em86xx_uart_release_port(struct uart_port *port)
{
	release_mem_region(port->mapbase, UART_PORT_SIZE);
}

/*
 * Request the memory region(s) being used by 'port'
 */
static int em86xx_uart_request_port(struct uart_port *port)
{
	return request_mem_region(port->mapbase, UART_PORT_SIZE, "serial_em86xx")
			!= NULL ? 0 : -EBUSY;
}

/*
 * Configure/autoconfigure the port.
 */
static void em86xx_uart_config_port(struct uart_port *port, int flags)
{
	if (flags & UART_CONFIG_TYPE) {
		port->type = PORT_EM86XX;
		em86xx_uart_request_port(port);
	}
}

/*
 * verify the new serial_struct (for TIOCSSERIAL).
 */
static int em86xx_uart_verify_port(struct uart_port *port, struct serial_struct *ser)
{
	int ret = 0;
	if (ser->type != PORT_UNKNOWN && ser->type != PORT_EM86XX)
		ret = -EINVAL;
	if (ser->irq < 0 || ser->irq >= NR_IRQS)
		ret = -EINVAL;
	if (ser->baud_base < 9600)
		ret = -EINVAL;
	return ret;
}

static struct uart_ops em86xx_pops = {
	tx_empty:	em86xx_uart_tx_empty,
	set_mctrl:	em86xx_uart_set_mctrl,
	get_mctrl:	em86xx_uart_get_mctrl,
	stop_tx:	em86xx_uart_stop_tx,
	start_tx:	em86xx_uart_start_tx,
	stop_rx:	em86xx_uart_stop_rx,
	enable_ms:	em86xx_uart_enable_ms,
	break_ctl:	em86xx_uart_break_ctl,
	startup:	em86xx_uart_startup,
	shutdown:	em86xx_uart_shutdown,
	change_speed:	em86xx_uart_change_speed,
	release_port:	em86xx_uart_release_port,
	request_port:	em86xx_uart_request_port,
	config_port:	em86xx_uart_config_port,
	verify_port:	em86xx_uart_verify_port,
};

#define EM86XX_UART_PORT0 \
	{ \
		membase:	(void *)IO_ADDRESS(CPU_UART0_BASE), \
		mapbase:	CPU_UART0_BASE, \
		iotype:		SERIAL_IO_MEM, \
		irq:		IRQ_UART0, \
		uartclk:	14745600, \
		fifosize:	16, \
		unused:		{ RX_FIFO_TRIG_DEFAULT, TX_FIFO_TRIG_DEFAULT}, \
		ops:		&em86xx_pops, \
		flags:		ASYNC_BOOT_AUTOCONF, \
	} \

#define EM86XX_UART_PORT1 \
	{ \
		membase:	(void *)IO_ADDRESS(CPU_UART1_BASE), \
		mapbase:	CPU_UART1_BASE, \
		iotype:		SERIAL_IO_MEM, \
		irq:		IRQ_UART1, \
		uartclk:	14745600, \
		fifosize:	16, \
		unused:		{ RX_FIFO_TRIG_DEFAULT, TX_FIFO_TRIG_DEFAULT}, \
		ops:		&em86xx_pops, \
		flags:		ASYNC_BOOT_AUTOCONF, \
	} \

static struct uart_port em86xx_ports[UART_NR] = {
#ifdef CONFIG_SERIAL_EM86XX_SWAPPORT
	EM86XX_UART_PORT1, EM86XX_UART_PORT0
#else
	EM86XX_UART_PORT0, EM86XX_UART_PORT1
#endif
};

#ifdef CONFIG_SERIAL_EM86XX_CONSOLE
#ifdef used_and_not_const_char_pointer
static int em86xx_uart_console_read(struct uart_port *port, char *s, u_int count)
{
	unsigned int status;
	int c;
#if DEBUG
	printk("em86xx_uart_console_read() called\n");
#endif

	c = 0;
	while (c < count) {
		status = UART_GET_LSR(port);
		if (UART_RX_READY(status)) {
			*s++ = UART_GET_CHAR(port);
			c++;
		} else {
			// nothing more to get, return
			return c;
		}
	}
	// return the count
	return c;
}
#endif

static void em86xx_uart_console_write(struct console *co, const char *s, u_int count)
{
	struct uart_port *port = em86xx_ports + co->index;
	unsigned int status, old_ier;
	int i;

	/*
	 *	First save the IER then disable the interrupts
	 */
	old_ier = UART_GET_IER(port);
	UART_PUT_IER(port, 0);

	/*
	 *	Now, do each character
	 */
	for (i = 0; i < count; i++) {
		do {
			status = UART_GET_LSR(port);
		} while (!UART_TX_READY(status));
		UART_PUT_CHAR(port, s[i]);
		if (s[i] == '\n') {
			do {
				status = UART_GET_LSR(port);
			} while (!UART_TX_READY(status));
			UART_PUT_CHAR(port, '\r');
		}
	}

	/*
	 *	Finally, wait for transmitter to become empty
	 *	and restore the TCR
	 */
	do {
		status = UART_GET_LSR(port);
	} while (!UART_TX_EMPTY(status));
	UART_PUT_IER(port, old_ier);
}

static kdev_t em86xx_uart_console_device(struct console *co)
{
	return MKDEV(SERIAL_EM86XX_MAJOR, SERIAL_EM86XX_MINOR + co->index);
}

static int em86xx_uart_console_wait_key(struct console *co)
{
	struct uart_port *port = em86xx_ports + co->index;
	unsigned int status;

	do {
		status = UART_GET_LSR(port);
	} while (!UART_RX_READY(status));
	return UART_GET_CHAR(port);
}

/*
static void __init
em86xx_uart_console_get_options(struct uart_port *port, int *baud, int *parity, int *bits)
{
	u_int lcr, quot;
	lcr = UART_GET_LCR(port);

	*parity = 'n';
	if (lcr & EM86XX_UART_LCR_PEN) {
		if (lcr & EM86XX_UART_LCR_EPS)
			*parity = 'e';
		else
			*parity = 'o';
	}
	*bits = (lcr & EM86XX_UART_LCR_WLS) + 5;

	quot = UART_GET_CLKDIV(port);
	//FIXME : "+1" is here to avoid division by 0 errors,
	// need to check what REALLY happens when CLKDIV is set to 0, (disabled the UART or ?)
	*baud = port->uartclk / (16 * quot + 1);
}
*/

static int __init em86xx_uart_console_setup(struct console *co, char *options)
{
	struct uart_port *port;
	int baud = CONFIG_SERIAL_EM86XX_SERIAL_BAUDRATE;
	int bits = 8;
	int parity = 'n';
	int flow = 'n';

	/*
	 * Check whether an invalid uart number has been specified, and
	 * if so, search for the first available port that does have
	 * console support.
	 */
	printk("serial_em86xx: setup_console @ %d\n",baud);

	port = uart_get_console(em86xx_ports, CONFIG_SERIAL_EM86XX_UARTNR, co);

	if (options)
		uart_parse_options(options, &baud, &parity, &bits, &flow);
//	else 
//		em86xx_uart_console_get_options(port, &baud, &parity, &bits);


	return uart_set_options(port, co, baud, parity, bits, flow);
}

static struct console em86xx_console = {
	write:		em86xx_uart_console_write,
#ifdef used_and_not_const_char_pointer
	read:		em86xx_uart_console_read,
#endif
	device:		em86xx_uart_console_device,
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,19)	// removed from 2.4.19
	wait_key:	em86xx_uart_console_wait_key,
#endif
	setup:		em86xx_uart_console_setup,
	flags:		CON_PRINTBUFFER,
	index:		-1,
};

void __init em86xx_uart_clock_init(void)
{
	int i;
	
	for(i=0; i< UART_NR; i++) 
#ifdef CONFIG_USE_SYSCLK
		em86xx_ports[i].uartclk = __get_clock(2);
#else
		em86xx_ports[i].uartclk = EM86XX_EXT_CLOCK;
#endif
}

void __init em86xx_uart_console_init(void)
{
	em86xx_uart_clock_init();

	register_console(&em86xx_console);
}


#define EM86XX_CONSOLE	&em86xx_console
#else
#define EM86XX_CONSOLE	NULL
#endif

static struct uart_driver em86xx_reg = {
	owner:			THIS_MODULE,
	normal_major:		SERIAL_EM86XX_MAJOR,
#ifdef CONFIG_DEVFS_FS
	normal_name:		"ttyAM%d",
	callout_name:		"cuaam%d",
#else
	normal_name:		"ttyAM",
	callout_name:		"cuaam",
#endif
	normal_driver:		&normal,
	callout_major:		CALLOUT_EM86XX_MAJOR,
	callout_driver:		&callout,
	table:			em86xx_table,
	termios:		em86xx_termios,
	termios_locked:		em86xx_termios_locked,
	minor:			SERIAL_EM86XX_MINOR,
	nr:			CONFIG_SERIAL_EM86XX_UARTNR,
	port:			em86xx_ports,
	cons:			EM86XX_CONSOLE,
};

static int __init em86xx_uart_init(void)
{
	int ret;
	unsigned int cflag;

	em86xx_uart_clock_init();
	ret = uart_register_driver(&em86xx_reg);

	// 7/24/2003
	// uart_register_driver() sets the cflag as default 
	// setup default baud rate properly
	cflag = normal.init_termios.c_cflag & ~CBAUD;

	switch (CONFIG_SERIAL_EM86XX_SERIAL_BAUDRATE) {
	case 1200: cflag |= B1200; break;
	case 2400: cflag |= B2400; break;
	case 4800: cflag |= B4800; break;
	case 9600: cflag |= B9600; break;
	case 19200: cflag |= B19200; break;
	default: cflag |= B38400; break;
	case 57600: cflag |= B57600; break;
	case 115200: cflag |= B115200; break;
	case 230400: cflag |= B230400; break;
	case 460800: cflag |= B460800; break;
	}

	normal.init_termios.c_cflag = cflag;

	return ret;
}

static void __exit em86xx_uart_exit(void)
{
	uart_unregister_driver(&em86xx_reg);
}

module_init(em86xx_uart_init);
module_exit(em86xx_uart_exit);
