/*
 * driver/ide/em86xx/em86xx_bm.c
 * 
 * EM86XX Bus Master IDE driver
 *
 * PIO mode support by Ho Lee 01/15/2004
 * Multi-word DMA mode support by Ho Lee 01/29/2004
 */

/*
 * Tango Bus Master IDE interface is different from that of Japser
 * Japser's IDE controller is compatible wtih PCI-based IDE controller,
 * and Tango's IDE controller is based on Japser's, but it doesn't make 
 * use of PRD table for bus mastering and DMA transfer must go through
 * MBUS. So device driver for Tango bus master IDE controller with DMA
 * support looks similar to that of Mambo's ISA IDE controller. 
 *
 * So this implementation has many duplicated code with em86xx_isa.c
 * But I would like to have separate file to keep clean code for that,
 * because em86xx_isa.c contains some experimental code which is not used,
 */

#include <linux/kernel.h>
#include <linux/init.h>
#include <asm/io.h>
#include <asm/hardware.h>
#include <linux/ide.h>
#include <asm/arch/em86xxapi.h>
#include "em86xx_ide.h"

#ifdef MODULE
#include <linux/module.h>
MODULE_AUTHOR("Ho Lee");
MODULE_DESCRIPTION(" EM86XX BM IDE driver");
MODULE_LICENSE("GPL");
#endif

#ifdef CONFIG_BLK_DEV_EM86XX_BMIDEDMA
// handlers
static int em86xx_bmide_dma_read(ide_drive_t *drive);
static int em86xx_bmide_dma_write(ide_drive_t *drive);
static int em86xx_bmide_dma_begin(ide_drive_t *drive);
static int em86xx_bmide_dma_end(ide_drive_t *drive);
static int em86xx_bmide_dma_check(ide_drive_t *drive);
static int em86xx_bmide_dma_host_on(ide_drive_t *drive);
static int em86xx_bmide_dma_host_off(ide_drive_t *drive);
static int em86xx_bmide_tune_chipset(ide_drive_t *drive, u8 speed);

// helper functions
struct buffer_head *em86xx_bmide_dma_setup(int read, struct buffer_head *bh, int stop_if_fragment);
static int em86xx_bmide_rwdma(ide_drive_t *drive, int read, struct request *rq);
static int em86xx_bmide_dma_end_io(ide_drive_t *drive);
static ide_startstop_t em86xx_bmide_dma_intr(ide_drive_t *drive);

// switchbox and MBUS
static int g_sbox_port_read, g_sbox_port_write;
static unsigned int g_regbase_read, g_regbase_write;
static int g_irq_read, g_irq_write;

/* Merge BH as much as possible */
#define EM86XX_BMIDE_DMA_MERGE_BH

// bounce buffer
#define DMA_BOUNCE_BUF_ORDER		(16 - PAGE_SHIFT)	// maximum = 128 sectors = 2^7 sectors = 2^7 * 2^9 bytes = 2^16 bytes
static unsigned char *g_bounce_buf = NULL;
#endif

// helper functions
static void em86xx_bmide_tune_chipset_drive(int drive, int speed, int verbose);

//
// data structure initialization
//

void em86xx_bmide_init_hwif_ports(hw_regs_t *hw, ide_ioreg_t data_port, ide_ioreg_t ctrl_port, int *irq)
{
	int i;
	ide_ioreg_t reg;

	memset(hw, 0, sizeof(*hw));

	// setup io_ports
	for (i = IDE_DATA_OFFSET, reg = data_port; i <= IDE_STATUS_OFFSET; i++) {
		hw->io_ports[i] = reg;
		reg += 4;
	}

	hw->io_ports[IDE_CONTROL_OFFSET] = ctrl_port;
	hw->dma = NO_DMA;

	// Tango has two interrupts for IDE controller : IRQ_IDECTRL_IDE, IRQ_IDECTRL_IDEDMA
	// IRQ_IDECTRL_IDEDMA is to be used for DMA transfer but it is almost same with 
	// IRQ_IDECTRL_IDE interrupt.  Tango triggers IDEDMA interrupt after DMA FIFO is
	// flushed, and it stays high until BMIS register bit 2 is cleared. 
	// Use IRQ_IDECTRL_IDE interrupt for both of PIO and DMA.
	if (irq)
		*irq = IRQ_IDECTRL_IDE;
}

//
// initialize IDE driver
//

int __init em86xx_bmide_init(void)
{
	int index, i;
	ide_hwif_t *hwif;
#ifdef MODULE
        ide_drive_t *drive = NULL;
#endif
	hw_regs_t hw;

	// Enable bus master IDE interface
	__raw_writel(0x08400, REG_BASE_HOST + IDECTRL_pri_idectl);
	// no skew : normal IORDY, normal data
	__raw_writel(0x00000000, REG_BASE_HOST + IDECTRL_idesrc);
	// Enable non-PRD mode
	__raw_writel(0x04, REG_BASE_HOST + IDECTRL_bmic);

	em86xx_bmide_init_hwif_ports(&hw, REG_BASE_HOST_BMIDE, /* REG_BASE_HOST + IDECTRL_pri_ctrlblock */ 0, &hw.irq);
       index = ide_register_hw(&hw, NULL);
	printk("EM86XX Bus Mastering IDE activated as ide%d.\n", index);

	// initialize ide_hwifs data structure
	// supports only one interface
	for (i = index; i < index + 1; ++i) {
		// register DMA handler
		hwif = &ide_hwifs[i];
		hwif->OUTB = em86xx_ide_OUTB;
		hwif->OUTBSYNC = em86xx_ide_OUTBSYNC;
		hwif->INB = em86xx_ide_INB;
	    if (!noautodma) {
#ifdef CONFIG_BLK_DEV_EM86XX_BMIDEDMA
		hwif->autodma = 1;
		hwif->atapi_dma = 1;
#ifdef CONFIG_BLK_DEV_EM86XX_BMIDEUDMA
		hwif->ultra_mask = 0x7f;        // enable all Ultra DMA	
#else
		hwif->ultra_mask = 0x00;		// disable all Ultra DMA
#endif
		hwif->mwdma_mask = 0x07;		// enable all Multi-word DMA (0/1/2)
		hwif->swdma_mask = 0x80;		// disable all Single-word DMA

		// registers
		hwif->dma_command = REG_BASE_HOST + IDECTRL_bmic;
		hwif->dma_status = REG_BASE_HOST + IDECTRL_bmis;
		hwif->dma_prdtable = REG_BASE_HOST + IDECTRL_bmidtp;

		// handlers
		hwif->ide_dma_read = em86xx_bmide_dma_read;
		hwif->ide_dma_write = em86xx_bmide_dma_write;
		hwif->ide_dma_begin = em86xx_bmide_dma_begin;
		hwif->ide_dma_end = em86xx_bmide_dma_end;
		hwif->ide_dma_check = em86xx_bmide_dma_check;
		hwif->ide_dma_on = em86xx_ide_dma_on;
		hwif->ide_dma_off = em86xx_ide_dma_off;
		hwif->ide_dma_off_quietly = em86xx_ide_dma_off_quietly;
		hwif->ide_dma_test_irq = em86xx_ide_dma_test_irq;
		hwif->ide_dma_host_on = em86xx_bmide_dma_host_on;
		hwif->ide_dma_host_off = em86xx_bmide_dma_host_off;
		hwif->ide_dma_bad_drive = em86xx_ide_dma_bad_drive;
		hwif->ide_dma_good_drive = em86xx_ide_dma_good_drive;
		hwif->ide_dma_count = em86xx_ide_dma_count;
		hwif->ide_dma_verbose = em86xx_ide_dma_verbose;
		hwif->ide_dma_retune = em86xx_ide_dma_retune;
		hwif->ide_dma_lostirq = em86xx_ide_dma_lostirq;
		hwif->ide_dma_timeout = em86xx_ide_dma_timeout;
		hwif->speedproc = em86xx_bmide_tune_chipset;
#endif
	    }
	}

	// set to maximum PIO mode
	for (i = 0; i < MAX_DRIVES; ++i)
		em86xx_bmide_tune_chipset_drive(i, XFER_PIO_4, 0);

	if (!noautodma) {
#ifdef CONFIG_BLK_DEV_EM86XX_BMIDEDMA
	// setup switchbox
	g_sbox_port_read = em86xx_mbus_alloc_dma(SBOX_IDEDVD, 1, &g_regbase_read, &g_irq_read);
	g_sbox_port_write = em86xx_mbus_alloc_dma(SBOX_IDEDVD, 0, &g_regbase_write, &g_irq_write);

	if ((g_bounce_buf = (unsigned char *)__get_free_pages(GFP_KERNEL, DMA_BOUNCE_BUF_ORDER)) == NULL)
		printk("IDE: Can not allocate buffer for IDE transfer\n");

#ifdef EM86XX_BOUNCE_BUFFER_NOCACHE
	g_bounce_buf = (unsigned char *)em86xx_to_ncaddr((unsigned long)g_bounce_buf);
	printk("Bounce buffer starts at 0x%08lx\n", (unsigned long)g_bounce_buf);
#endif

#endif	// EM86XX_BMIDEDMA
	}

	return 0;
}

#ifdef CONFIG_BLK_DEV_EM86XX_BMIDEDMA

//
// IDE DMA handlers
//

int em86xx_bmide_dma_read(ide_drive_t *drive)
{
	struct request *rq;

	drive->waiting_for_dma = 1;
	rq = HWGROUP(drive)->rq;
	// printk("em86xx_bmide_dma_read : nsect = %ld\n", rq->nr_sectors);
	return em86xx_bmide_rwdma(drive, 1, rq);
}

int em86xx_bmide_dma_write(ide_drive_t *drive)
{
	struct request *rq;

	drive->waiting_for_dma = 1;
	rq = HWGROUP(drive)->rq;
	// printk("em86xx_bmide_dma_write : sector = %ld, nsect = %ld\n", rq->sector, rq->nr_sectors);
	return em86xx_bmide_rwdma(drive, 0, rq);
}

int em86xx_bmide_dma_begin(ide_drive_t *drive)
{
	return 0;
}

int em86xx_bmide_dma_end(ide_drive_t *drive)
{
	drive->waiting_for_dma = 0;
	return em86xx_bmide_dma_end_io(drive);
}

int em86xx_bmide_dma_check(ide_drive_t *drive)
{
	int mode;

#ifdef CONFIG_BLK_DEV_EM86XX_BMIDEUDMA
	if ((__raw_readl(REG_BASE_HOST + IDECTRL_idestatus) & 0x02) == 0x02)
        mode = 1;   // 40 donductor cable (UDMA33) : Mode 0/1/2
	else 
		mode = 4;   // 80 conductor cable : Mode 0/1/2/3/4/5/6
#else
	mode = 0;       // MW DMA mode 2
#endif

	return em86xx_ide_config_drive_for_dma(drive, mode);
}

int em86xx_bmide_dma_host_on(ide_drive_t *drive)
{
	if (drive->using_dma) {
		ide_hwif_t *hwif = HWIF(drive);
		u8 unit = (drive->select.b.unit & 0x01);
		u8 dma_stat = hwif->INB(hwif->dma_status);

		hwif->OUTB((dma_stat | (1 << (5 + unit))), hwif->dma_status);
		return 0;
	}
	return 1;
}

int em86xx_bmide_dma_host_off(ide_drive_t *drive)
{
	ide_hwif_t *hwif = HWIF(drive);
	u8 unit = (drive->select.b.unit & 0x01);
	u8 dma_stat = hwif->INB(hwif->dma_status);

	hwif->OUTB((dma_stat & ~(1 << (5 + unit))), hwif->dma_status);

	return 0;
}

int em86xx_bmide_tune_chipset(ide_drive_t *drive, u8 speed)
{
	em86xx_bmide_tune_chipset_drive(drive->select.b.unit, speed, 1);

	return ide_config_drive_speed(drive, speed);
}

//
// helper functions
//

#ifdef EM86XX_BMIDE_DMA_MERGE_BH
#define MBUS_LINEAR_MAX 	(0x2000 - 0x200)
#endif

// return the number of fragmented blocks
static __inline__ int em86xx_bmide_request_fragment(struct request *rq)
{
	int ntransfer = 0;
	struct buffer_head *bh = rq->bh;

#ifdef EM86XX_BMIDE_DMA_MERGE_BH
	unsigned long addr;
	unsigned int size;
	unsigned int sector;

	/* Try to merge multiple BH if possible */
	for (ntransfer = 0;;) {
		if (bh == NULL)
			break;
		else
			++ntransfer;
		addr = (unsigned long)bh->b_data;
		size = bh->b_size;
		sector = bh->b_rsector;
		for (bh = bh->b_reqnext; bh != NULL; bh = bh->b_reqnext) {
			if ((addr + size) != (unsigned long)bh->b_data)
				break;	/* data is not contiguous */
			else if ((sector + (size / SECTOR_SIZE)) != bh->b_rsector)
				break;	/* sector is not contiguous */
			else if ((size + bh->b_size) > MBUS_LINEAR_MAX)
				break;	/* Size too large */
			else {	/* Merge the next BH */
				size += bh->b_size;
			}
		}	
	}
#else
	do {
		++ntransfer;
	} while ((bh = bh->b_reqnext) != NULL);
#endif

	return ntransfer;
}

// prepare MBUS for DMA transfer
// stop_if_fragment must be non-zero
struct buffer_head *em86xx_bmide_dma_setup(int read, struct buffer_head *bh, int stop_if_fragment)
{
#ifdef EM86XX_BMIDE_DMA_MERGE_BH
	struct {
		unsigned long addr;
		unsigned int size;
		unsigned int sector;
	} transfer[2], *ptransfer;
	unsigned int ntransfer;

	/* Try to merge multiple BH if possible */
	for (ntransfer = 0, ptransfer = &transfer[0];; ptransfer++) {
		if (bh == NULL)
			break;
		else if (++ntransfer > 2)
			break;

		ptransfer->addr = (unsigned long)bh->b_data;
		ptransfer->size = bh->b_size;
		ptransfer->sector = bh->b_rsector;
		for (bh = bh->b_reqnext; bh != NULL; bh = bh->b_reqnext) {
			if ((ptransfer->addr + ptransfer->size) != (unsigned long)bh->b_data)
				break;	/* data is not contiguous */
			else if ((ptransfer->sector + (ptransfer->size / SECTOR_SIZE)) != bh->b_rsector)
				break;	/* sector is not contiguous */
			else if ((ptransfer->size + bh->b_size) > MBUS_LINEAR_MAX)
				break;	/* Size too large */
			else {	/* Merge the next BH */
				ptransfer->size += bh->b_size;
			}
		}	
	}

	if ((ntransfer > 2) && (stop_if_fragment != 0))
		return bh;
#else
	struct buffer_head *prevbh;
	struct {
		unsigned long addr;
		unsigned int size;
	} transfer[2], *ptransfer;
	unsigned int ntransfer, size;

	// scans buffer head and calculate the requested transfer size
	// MBUS can't transfer to more than two separate regions. 
	// So if the buffer head is too fragmented, return non-NULL value
	for (prevbh = NULL, ntransfer = 0, ptransfer = transfer, size = 0; bh != NULL; prevbh = bh, bh = bh->b_reqnext) {
		if (++ntransfer > 2) {
			if (stop_if_fragment)
				return bh;
			break;
		}
		ptransfer->addr = (unsigned long) bh->b_data;
		ptransfer->size = bh->b_size;
		// printk("MBUS : bh = %p, bh->next = %p, %08x, %08x\n", bh, bh->b_reqnext, ptransfer->addr, ptransfer->size);
		++ptransfer;
		size += bh->b_size;
	}
#endif

	// setup MBUS
	if (ntransfer == 1)
		em86xx_mbus_setup_dma_linear(read ? g_regbase_read : g_regbase_write, 
			em86xx_to_ncaddr(transfer[0].addr), transfer[0].size);
	else
		em86xx_mbus_setup_dma_double(read ? g_regbase_read : g_regbase_write, 
			em86xx_to_ncaddr(transfer[0].addr), transfer[0].size,
			em86xx_to_ncaddr(transfer[1].addr), transfer[1].size);

	return bh;
}

// core of DMA read and write transfer
int em86xx_bmide_rwdma(ide_drive_t *drive, int read, struct request *rq)
{
	unsigned int buflen = SECTOR_SIZE * rq->nr_sectors;

	// setup interrupt handler
	// only required for hard disk. CD-ROM driver registers a handler for itself
	if (drive->media == ide_disk)
		ide_set_handler(drive, &em86xx_bmide_dma_intr, 10 * WAIT_CMD, NULL);

	// setup MBUS
	if (em86xx_bmide_dma_setup(read, rq->bh, 1)) {
		// if the request is too fragmented (over two non-continuous regions)
		// use the bounce buffer. For read transfers, transfer to bounce buffer
		// and copy to original buffer head after transfer. For write transfers,
		// copy the data to be written to bounce buffer first.
		unsigned long buf = (unsigned long) g_bounce_buf;

		if (!read) {
			struct buffer_head *bh = rq->bh;
			do {
				memcpy((void *) buf, bh->b_data, bh->b_size);
				buf += bh->b_size;
			} while ((bh = bh->b_reqnext) != NULL);
#ifdef CONFIG_SD_CHECKERROR
			if (buf > ((unsigned long) g_bounce_buf + (1 << (DMA_BOUNCE_BUF_ORDER + PAGE_SHIFT))))
				printk("IDE: Bounce buffer is too small to fit requested transfers\n");
#endif
		}

#ifndef EM86XX_BOUNCE_BUFFER_NOCACHE
		// clean cache before starting transfer
		cpu_dcache_clean_range((unsigned long) g_bounce_buf, (unsigned long) g_bounce_buf + buflen);
#endif

		// try to setup MBUS again. This always succeeds
		em86xx_mbus_setup_dma(read ? g_regbase_read : g_regbase_write, (unsigned long) g_bounce_buf, buflen, NULL, NULL);
	} else
		em86xx_ide_dma_cache(rq, 1, 0);

	// setup IDE DMA pointer and length
	// IDE DMA pointer is not important since IDE controller is just a tunnel
	// to MBUS. 
	// __raw_writel((unsigned long) buf, REG_BASE_HOST + IDECTRL_ide_dmaptr);
	__raw_writel(((unsigned long) buflen), REG_BASE_HOST + IDECTRL_ide_dmalen);

	// send IDE command
	// 28 bits : READDMA or WRITEDMA
	// 48 bits : READDMA_EXT or WRITEDMA_EXT
	// DMARQ is asserted to low
	if (drive->media == ide_disk) {
		HWIF(drive)->OUTB(read ? (drive->addressing ? WIN_READDMA_EXT : WIN_READDMA) : (drive->addressing ? WIN_WRITEDMA_EXT : WIN_WRITEDMA), IDE_COMMAND_REG);
	}

	// start bus master transfer
	// non-prd mode & bus master start
	// DMAACK is asserted to high
	__raw_writel(0x05 | (read ? 0x08 : 0x00), REG_BASE_HOST + IDECTRL_bmic);

	return 0;
}

// cleanup after DMA transfer is done
int em86xx_bmide_dma_end_io(ide_drive_t *drive)
{
	int stat;
	struct request *rq = HWGROUP(drive)->rq;

	// wait for DMA interrupt
	// em86xx_wait_irq(IRQ_IDECTRL_IDEDMA);
	// clear DMA interrupt
	__raw_writel(__raw_readl(REG_BASE_HOST + IDECTRL_bmis), REG_BASE_HOST + IDECTRL_bmis);

	// confirm whether MBUS transfer is done
	// due to the memory arbitration, IDE device thinks the DMA transfer
	// is done, but the data might be held in MBUS FIFO
	stat = em86xx_mbus_wait(rq->cmd == READ ? g_regbase_read : g_regbase_write, SBOX_IDEDVD);

    // clear interrupt
    HWIF(drive)->INB(IDE_STATUS_REG);
 
    // Stop bus mastering
    __raw_writel(0x04, REG_BASE_HOST + IDECTRL_bmic);
	
	if (stat != 0)
		return 1;

	if (rq->cmd == READ) {
		if (em86xx_bmide_request_fragment(rq) > 2) {
			// copy buffer after transfer
			unsigned long buf = (unsigned long) g_bounce_buf;
			struct buffer_head *bh = rq->bh;
#ifndef EM86XX_BOUNCE_BUFFER_NOCACHE
			unsigned long buflen = SECTOR_SIZE * rq->nr_sectors;
	
			cpu_dcache_invalidate_range(buf, buf + buflen);
#endif
	
			do {
				memcpy(bh->b_data, (void *) buf, bh->b_size);
				buf += bh->b_size;
			} while ((bh = bh->b_reqnext) != NULL);
		} else {
			em86xx_ide_dma_cache(rq, 0, 1);
		}
	}

	return 0;
}

// DMA interrupt handler after transfer is done
ide_startstop_t em86xx_bmide_dma_intr(ide_drive_t *drive)
{
	int i;
	ide_hwif_t *hwif = HWIF(drive);
	struct request *rq = HWGROUP(drive)->rq;

	hwif->ide_dma_end(drive);

	for (i = rq->nr_sectors; i > 0;) {
		i -= rq->current_nr_sectors;
		ide_end_request(drive, 1);
	}

	return ide_stopped;
}

#endif

void em86xx_bmide_tune_chipset_drive(int drive, int speed, int verbose)
{
	int mode = speed & 0x0f;
	unsigned int ctrlreg = __raw_readl(REG_BASE_HOST + IDECTRL_pri_idectl);
	unsigned int newflag = 0;
	
	if (speed >= XFER_PIO_0 && speed <= XFER_PIO_4) {
		// setup timing for PIO mode
		static unsigned int s_pio_tim[] = {
#if 0
			0xf73900d5, 0xcb2f0093, 0xaf280052, 0xa30f0d51, 0x970d0441
#endif
			0xf7200dd5, 0xcc180d93, 0xaf130d52, 0xa30f0d51, 0x970d0441
		};

		mode -= 8;
		if (verbose)
			printk("IDE: Set drive %d to PIO mode %d\n",
				drive, mode);

		// fast timing for PIO
		newflag = 0x01;
		
		__raw_writel(s_pio_tim[mode], REG_BASE_HOST + ((drive == 0) ? IDECTRL_pri_drv0tim : IDECTRL_pri_drv1tim));
	} 
#ifdef CONFIG_BLK_DEV_EM86XX_BMIDEDMA
	else if (speed >= XFER_MW_DMA_0 && speed <= XFER_MW_DMA_2) {
		// setup timing for Multi-word DMA
		static unsigned int s_dma_tim[] = {
#if 0
			0xdf2a2a93, 0x9d0f0952, 0x970d0441, 0x82020211 
#endif
			0xdf2a2a93, 0x9d0f0952, 0x970d0441
		};
		static unsigned int s_udma_tim1[] = {
#if 0
			0x35440b08, 0x35440a06, 0x35440804, 0x11110202
#endif
			0x35440b08, 0x35440a06, 0x35440804
		};
		static unsigned int s_udma_tim2[] = {
#if 0
			0x00000208, 0x00000206, 0x00000204, 0x00000202
#endif
			0x00000208, 0x00000206, 0x00000204
		};

		if (verbose)
			printk("IDE: Set drive %d to Multi-word DMA mode %d\n",
				drive, mode);

		// fast timing for PIO, prefetch enable
		newflag = 0x05;

		__raw_writel(s_dma_tim[mode], REG_BASE_HOST + ((drive == 0) ? IDECTRL_pri_drv0tim : IDECTRL_pri_drv1tim));
		__raw_writel(s_udma_tim1[mode], REG_BASE_HOST + ((drive == 0) ? IDECTRL_pri_drv0udmatim1 : IDECTRL_pri_drv1udmatim1));
		__raw_writel(s_udma_tim2[mode], REG_BASE_HOST + ((drive == 0) ? IDECTRL_pri_drv0udmatim2 : IDECTRL_pri_drv1udmatim2));
	} 
#endif
#ifdef CONFIG_BLK_DEV_EM86XX_BMIDEUDMA
	else if (speed >= XFER_UDMA_0 && speed <= XFER_UDMA_7) {
		// setup timing for Ultra DMA
		static unsigned int s_udma_tim1[] = {
#if 0
			0x33331f17, 0x3333180f, 0x3333130b, 0x33331308,
			0x33331305, 0x33331003, 0x11110202,
#endif
			0x33332016, 0x3333190e, 0x3333140a, 0x33331407,
			0x33331404, 0x33331103, 0x33331102, 0x11110202
		};
		static unsigned int s_udma_tim2[] = {
#if 0
			0x0000020d, 0x00000209, 0x00000206, 0x00000203, 
			0x00000201, 0x00000201, 0x00000201,
#endif
			0x0000010d, 0x00000109, 0x00000106, 0x00000103,
			0x00000101, 0x00000101, 0x00000101, 0x00000101
		};

		if (verbose)
			printk("IDE: Set drive %d to Ultra DMA mode %d\n",
				drive, mode);

		newflag = 0x00;

		// enable Ultra DMA
		__raw_writel(__raw_readl(REG_BASE_HOST + IDECTRL_udmactl) | ((drive == 0) ? 0x01 : 0x02),
			REG_BASE_HOST + IDECTRL_udmactl);
		
		// set timing register
		__raw_writel(s_udma_tim1[mode], REG_BASE_HOST + ((drive == 0) ? IDECTRL_pri_drv0udmatim1 : IDECTRL_pri_drv1udmatim1));
		__raw_writel(s_udma_tim2[mode], REG_BASE_HOST + ((drive == 0) ? IDECTRL_pri_drv0udmatim2 : IDECTRL_pri_drv1udmatim2));
	}
#endif

	ctrlreg &= ~(drive == 0 ? 0x0f : 0xf0);
	ctrlreg |= newflag << (drive * 4);
	__raw_writel(ctrlreg, REG_BASE_HOST + IDECTRL_pri_idectl);
}



void __exit em86xx_bmide_exit(void)
{

#ifdef CONFIG_BLK_DEV_EM86XX_BMIDEDMA
#ifdef EM86XX_BOUNCE_BUFFER_NOCACHE
        g_bounce_buf = (unsigned char *)em86xx_to_caddr((unsigned long)g_bounce_buf);
#endif
        free_pages((unsigned long)g_bounce_buf, DMA_BOUNCE_BUF_ORDER);
#endif
        return;
}

#ifdef MODULE
module_init(em86xx_bmide_init);
module_exit(em86xx_bmide_exit);
#endif


