/*
 * driver/ide/em86xx/em86xx_ide.c
 * 
 * EM86XX IDE driver common code
 *
 * by Ho Lee March 4, 2003
 */

#include <linux/kernel.h>
#include <linux/init.h>
#include <asm/io.h>
#include <asm/hardware.h>
#include <linux/ide.h>
#include <linux/module.h> 
#include <linux/delay.h> 
#include <asm/arch/em86xxapi.h>

#if defined(CONFIG_SD_IDE_FASTPROBE) || defined(CONFIG_BLK_DEV_EM86XX_ISAIDENOIRQ)

#define IDE_WAIT_TIMEOUT_SECS		32

int ide_wait_stat_sd(struct hwif_s *hwif, int reg, int mask, int waitmask, int usetimeout)
{
	int status;
	int timeout = jiffies + (IDE_WAIT_TIMEOUT_SECS * HZ);

	for(;;) {
		status = hwif->INB(reg);
        if((status & mask) == waitmask) break;
		if (usetimeout) {
			if (jiffies > timeout)
				return 1;
		}
		udelay(50);
	}

	return 0;
}

#endif

//
// IDE HWIF interface introduced by Linux 2.4.22
//

// EM86XX IDE interface uses 16 bits wide bus
// and byte write doesn't start write transaction

void em86xx_ide_OUTB(u8 data, unsigned long port)
{
	__raw_writew((unsigned short) data, port);
}


void em86xx_ide_OUTBSYNC(ide_drive_t *drive, u8 data, unsigned long port)
{
	__raw_writew((unsigned short) data, port);
}

u8 em86xx_ide_INB(unsigned long port)
{
	return (unsigned char) __raw_readw(port);
}

//
// DMA handlers
// 

int em86xx_ide_dma_on(ide_drive_t *drive)
{
	ide_hwif_t *hwif = HWIF(drive);

	if (hwif->ide_dma_host_on)
		hwif->ide_dma_host_on(drive);

	printk("IDE: DMA enabled for %s%s\n",
		drive->media == ide_disk ? "ATA DISK " : 
		(drive->media == ide_cdrom ? "ATAPI CDROM " : ""),
		drive->name);
	drive->using_dma = 1;
	return 0;
}

int em86xx_ide_dma_off(ide_drive_t *drive)
{
	ide_hwif_t *hwif = HWIF(drive);

	if (hwif->ide_dma_host_off)
		hwif->ide_dma_host_off(drive);

	printk("IDE: DMA disabled for %s\n", drive->name);
	drive->using_dma = 0;
	return 0;
}

int em86xx_ide_dma_off_quietly(ide_drive_t *drive)
{
	drive->using_dma = 0;
	return 0;
}

int em86xx_ide_dma_test_irq(ide_drive_t *drive)
{
	return 1;
}

int em86xx_ide_dma_bad_drive(ide_drive_t *drive)
{
	return 0;
}

int em86xx_ide_dma_good_drive(ide_drive_t *drive)
{
	return 0;
}

int em86xx_ide_dma_count(ide_drive_t *drive)
{
	return 0;
}

int em86xx_ide_dma_verbose(ide_drive_t *drive)
{
	return 0;
}

int em86xx_ide_dma_retune(ide_drive_t *drive)
{
	return 0;
}

int em86xx_ide_dma_lostirq(ide_drive_t *drive)
{
	return 0;
}

int em86xx_ide_dma_timeout(ide_drive_t *drive)
{
	return 0;
}

//
// helper functions
//

// clean or invalidate every buffer head in request queue
void em86xx_ide_dma_cache(struct request *rq, int do_clean, int do_invalidate)
{
	struct buffer_head *bh = rq->bh;

#if 1
    // by Ho Lee 03/22/2004
    // When IDE transfers a lot of data, it tends to make whole cache dirty,
    // and in this case, managing cache on regions has no meaning, and even it
    // makes whole transfer slower, and managing whole cache shows better 
    // performance. This logic manages whole cache when more than one request
    // are in the queue, and manages affected region when one request is in
    // the queue. 
    if (bh->b_reqnext) {    // more than one request
    	if (do_invalidate)
            em86xx_flush_cache_data();
    	else if (do_clean)
            em86xx_clean_cache_data();
    } else {                // only one request
		if (do_clean && do_invalidate)
			cpu_cache_clean_invalidate_range((unsigned int) bh->b_data, (unsigned int) bh->b_data + bh->b_size, 0);
		else if (do_clean)
			cpu_dcache_clean_range((unsigned int) bh->b_data, (unsigned int) bh->b_data + bh->b_size);
		else if (do_invalidate)
			cpu_dcache_invalidate_range((unsigned int) bh->b_data, (unsigned int) bh->b_data + bh->b_size);
    }
#else
	struct buffer_head *bh = rq->bh;

	do {
		if (do_clean && do_invalidate)
			cpu_cache_clean_invalidate_range((unsigned int) bh->b_data, (unsigned int) bh->b_data + bh->b_size, 0);
		else if (do_clean)
			cpu_dcache_clean_range((unsigned int) bh->b_data, (unsigned int) bh->b_data + bh->b_size);
		else if (do_invalidate)
			cpu_dcache_invalidate_range((unsigned int) bh->b_data, (unsigned int) bh->b_data + bh->b_size);
	} while ((bh = bh->b_reqnext) != NULL);
#endif
}

int em86xx_ide_config_drive_for_dma(ide_drive_t *drive, int mode)
{
	int config_allows_dma = 0;
	struct hd_driveid *id = drive->id;
	ide_hwif_t *hwif = HWIF(drive);
	int speed;

	// disable DMA for CDROM by default
	if (drive->media == ide_disk || drive->media == ide_cdrom)
		config_allows_dma = 1;

	if (id && (id->capability & 1) && hwif->autodma && config_allows_dma) {
		speed = ide_dma_speed(drive, mode);

		/* For CDROM, we do the max. MDMA mode 2 */
		if ((speed > XFER_MW_DMA_2) && (drive->media == ide_cdrom))
			speed = XFER_MW_DMA_2;

		if (speed >= XFER_MW_DMA_0) {
			if (hwif->speedproc)
				hwif->speedproc(drive, speed);
			ide_config_drive_speed(drive, speed);
			return hwif->ide_dma_on(drive);
		}
	}

	return hwif->ide_dma_off_quietly(drive);
}

