/*
 * driver/ide/em86xx/em86xx_isa.c
 * 
 * EM86XX ISA IDE driver
 *
 * by Ho Lee March 4, 2003
 * CD-ROM DMA implemented by Ho Lee 2003/09/18
 */

/*
 	CDROM DMA transfer
		For ATA DMA transfer, DMA handler register "handler" and sends 
		read or write command to IDE device. But for CDROM, DMA handler
		doesn't send any command to the device, and it can not register
		handler to finish the transfer. DMA handler is called in the 
		middle of transfer.
*/

#include <linux/kernel.h>
#include <linux/init.h>
#include <asm/io.h>
#include <asm/hardware.h>
#include <linux/ide.h>
#include <asm/arch/em86xxapi.h>
#include "em86xx_ide.h"

#ifdef MODULE
#include <linux/module.h>
MODULE_AUTHOR("Ho Lee");
MODULE_DESCRIPTION(" EM86XX ISA IDE driver");
MODULE_LICENSE("GPL");
#endif

// in ide.c
void ide_intr(int irq, void *dev_id, struct pt_regs *regs);

#ifdef CONFIG_BLK_DEV_EM86XX_ISAIDEDMA
#define EM86XX_ISAIDE_DMA_MERGE_BH
#define EM86XX_ISAIDE_DMA_BOUNCE_BUF

static int g_sbox_port_read, g_sbox_port_write;
static unsigned int g_regbase_read, g_regbase_write;
static int g_irq_read, g_irq_write;

// function prototypes
static int em86xx_isaide_dma_read(ide_drive_t *drive);
static int em86xx_isaide_dma_write(ide_drive_t *drive);
static int em86xx_isaide_dma_begin(ide_drive_t *drive);
static int em86xx_isaide_dma_end(ide_drive_t *drive);
static int em86xx_isaide_dma_check(ide_drive_t *drive);
static int em86xx_isaide_dma_host_on(ide_drive_t *drive);
static int em86xx_isaide_dma_host_off(ide_drive_t *drive);
static void em86xx_isaide_tune_chipset_drive(int drive, int speed, int verbose);
#ifdef CONFIG_BLK_DEV_EM86XX_ISAIDETIMING
static int em86xx_isaide_tune_chipset(ide_drive_t *drive, u8 speed);
static unsigned long get_pb_ide_timing(unsigned int sysclk, int mode);
#endif

static int em86xx_isaide_rwdma(ide_drive_t *drive, int read, struct request *rq);
static void em86xx_isaide_mbus_intr(int irq, void *arg);
static int em86xx_isaide_dma_end_io(ide_drive_t *drive);
static ide_startstop_t em86xx_isaide_dma_intr(ide_drive_t *drive);

#ifdef EM86XX_ISAIDE_DMA_BOUNCE_BUF
#define EM86XX_ISAIDE_DMA_BOUNCE_BUF_ORDER		(16 - PAGE_SHIFT)	// maximum = 128 sectors = 2^7 sectors = 2^7 * 2^9 = 2^16
static unsigned char *g_bounce_buf = NULL;
#endif
#endif

//
// data structure initialization
//

void em86xx_isaide_init_hwif_ports(hw_regs_t *hw, ide_ioreg_t data_port, ide_ioreg_t ctrl_port, int *irq)
{
	int i;
	ide_ioreg_t reg;

	memset(hw, 0, sizeof(*hw));

	// setup io_ports
	for (i = IDE_DATA_OFFSET, reg = data_port; i <= IDE_STATUS_OFFSET; i++) {
		hw->io_ports[i] = reg;
		reg += 4;
	}

	hw->io_ports[IDE_CONTROL_OFFSET] = ctrl_port;
	hw->dma = NO_DMA;

	if (irq)
		*irq = IRQ_IDE;
}

//
// initialize IDE driver
//

int __init em86xx_isaide_init(void)
{
	int index, i;
	ide_hwif_t *hwif = NULL;
#ifdef MODULE
	ide_drive_t *drive = NULL;
#endif
	hw_regs_t hw;


#ifdef CONFIG_ARCH_TANGO
	// IDE configuration compatible with Mambo
	// IDE pins are tristated
	// __raw_writel(0x10800, REG_BASE_HOST + IDECTRL_pri_idectl);
#endif

	em86xx_isaide_init_hwif_ports(&hw, REG_BASE_HOST_ISAIDE, 0, &hw.irq);
    	index = ide_register_hw(&hw, NULL);
	printk("EM86XX ISA IDE activated as ide%d.\n", index);

	// initialize ide_hwifs data structure
	// supports only one interface
	for (i = index; i < index + 1; ++i) {
		// register DMA handler
		hwif = &ide_hwifs[i];
		hwif->OUTB = em86xx_ide_OUTB;
		hwif->OUTBSYNC = em86xx_ide_OUTBSYNC;
		hwif->INB = em86xx_ide_INB;
	    if (!noautodma) {
#ifdef CONFIG_BLK_DEV_EM86XX_ISAIDEDMA
		hwif->autodma = 1;
		hwif->atapi_dma = 1;
		hwif->ultra_mask = 0x00;	// disable all Ultra DMA
		hwif->mwdma_mask = 0x07;	// enable all Multi-word DMA
		hwif->swdma_mask = 0x80;	// disable all Single-word DMA

		hwif->ide_dma_read = em86xx_isaide_dma_read;
		hwif->ide_dma_write = em86xx_isaide_dma_write;
		hwif->ide_dma_begin = em86xx_isaide_dma_begin;
		hwif->ide_dma_end = em86xx_isaide_dma_end;
		hwif->ide_dma_check = em86xx_isaide_dma_check;
		hwif->ide_dma_on = em86xx_ide_dma_on;
		hwif->ide_dma_off = em86xx_ide_dma_off;
		hwif->ide_dma_off_quietly = em86xx_ide_dma_off_quietly;
		hwif->ide_dma_test_irq = em86xx_ide_dma_test_irq;
		hwif->ide_dma_host_on = em86xx_isaide_dma_host_on;
		hwif->ide_dma_host_off = em86xx_isaide_dma_host_off;
		hwif->ide_dma_bad_drive = em86xx_ide_dma_bad_drive;
		hwif->ide_dma_good_drive = em86xx_ide_dma_good_drive;
		hwif->ide_dma_count = em86xx_ide_dma_count;
		hwif->ide_dma_verbose = em86xx_ide_dma_verbose;
		hwif->ide_dma_retune = em86xx_ide_dma_retune;
		hwif->ide_dma_lostirq = em86xx_ide_dma_lostirq;
		hwif->ide_dma_timeout = em86xx_ide_dma_timeout;
#ifdef CONFIG_BLK_DEV_EM86XX_ISAIDETIMING
		hwif->speedproc = em86xx_isaide_tune_chipset;
#endif
#endif
	    }
	}

	if (!noautodma) {
#ifdef CONFIG_BLK_DEV_EM86XX_ISAIDEDMA
	// setup switchbox
	g_sbox_port_read = em86xx_mbus_alloc_dma(SBOX_IDEFLASH, 1, &g_regbase_read, &g_irq_read);
	g_sbox_port_write = em86xx_mbus_alloc_dma(SBOX_IDEFLASH, 0, &g_regbase_write, &g_irq_write);

#ifdef EM86XX_ISAIDE_DMA_BOUNCE_BUF
	if ((g_bounce_buf = (unsigned char *) __get_free_pages(GFP_KERNEL, EM86XX_ISAIDE_DMA_BOUNCE_BUF_ORDER)) == NULL)
		printk("EM86XX IDE : Can not allocate buffer for IDE transfer\n");
#ifdef EM86XX_BOUNCE_BUFFER_NOCACHE
	g_bounce_buf = (unsigned char *)em86xx_to_ncaddr((unsigned long)g_bounce_buf);
printk("Bounce buffer starts at 0x%08lx\n", (unsigned long)g_bounce_buf);
#endif
#endif	// BOUNCE_BUF
#endif	// EM86XX_ISAIDEDMA
	}

#if defined(MODULE) && defined(CONFIG_BLK_DEV_IDE)
#ifdef HWIF_PROBE_CLASSIC_METHOD
        probe_hwif(hwif);
        hwif_init(hwif);
#else 
	probe_hwif_init(hwif);
#endif /* HWIF_PROBE_CLASSIC_METHOD */

	for (i = 0; i < MAX_DRIVES; ++i) {
                drive = &hwif->drives[i];
                if (drive->present && !drive->dead)
                        ide_attach_drive(drive);
        }
#endif

	 return 0;
}

//
// IDE DMA processing
//

#ifdef CONFIG_BLK_DEV_EM86XX_ISAIDEDMA

int em86xx_isaide_dma_read(ide_drive_t *drive)
{
	struct request *rq;

	drive->waiting_for_dma = 1;
	rq = HWGROUP(drive)->rq;
	// printk("em86xx_isaide_dma_read : nsect = %ld\n", rq->nr_sectors);
	return em86xx_isaide_rwdma(drive, 1, rq);
}

int em86xx_isaide_dma_write(ide_drive_t *drive)
{
	struct request *rq;

	drive->waiting_for_dma = 1;
	rq = HWGROUP(drive)->rq;
	// printk("em86xx_isaide_dma_write : nsect = %ld\n", rq->nr_sectors);
	return em86xx_isaide_rwdma(drive, 0, rq);
}

int em86xx_isaide_dma_begin(ide_drive_t *drive)
{
	return 0;
}

int em86xx_isaide_dma_end(ide_drive_t *drive)
{
	drive->waiting_for_dma = 0;
	return em86xx_isaide_dma_end_io(drive);
}

int em86xx_isaide_dma_check(ide_drive_t *drive)
{
	return em86xx_ide_config_drive_for_dma(drive, 0);
}

int em86xx_isaide_dma_host_on(ide_drive_t *drive)
{
	return 0;
}

int em86xx_isaide_dma_host_off(ide_drive_t *drive)
{
	return 0;
}

#ifdef CONFIG_BLK_DEV_EM86XX_ISAIDETIMING

void em86xx_isaide_tune_chipset_drive(int drive, int speed, int verbose)
{
	int mode = speed & 0x0f;
	unsigned int sysclk, timing;
	
	if (speed >= XFER_PIO_0 && speed <= XFER_PIO_4)
		mode -= 8;
	else if (speed >= XFER_MW_DMA_0 && speed <= XFER_MW_DMA_2)
		mode += 5;
	else
		return;	

	// get_pb_ide_timing() receives mode in reversed order
	//   PIO : 7-3
	//   Multi-word DMA : 2-0
	mode = 7 - mode;
	sysclk = __get_clock_mhz();
	timing = get_pb_ide_timing(sysclk, mode);

	__raw_writel(timing, REG_BASE_HOST + PB_timing_slot(PB_TIMINGSLOT_IDE));
}

int em86xx_isaide_tune_chipset(ide_drive_t *drive, u8 speed)
{
	em86xx_isaide_tune_chipset_drive(drive->select.b.unit, speed, 1);

	return ide_config_drive_speed(drive, speed);
}

#endif

#ifdef EM86XX_ISAIDE_DMA_MERGE_BH
#define MBUS_LINEAR_MAX 	(0x2000 - 0x200)
#endif

int em86xx_isaide_request_fragment(struct request *rq)
{
	int ntransfer = 0;
	struct buffer_head *bh = rq->bh;

#ifdef EM86XX_ISAIDE_DMA_MERGE_BH
	unsigned long addr;
	unsigned int size;
	unsigned int sector;

	/* Try to merge multiple BH if possible */
	for (ntransfer = 0;;) {
		if (bh == NULL)
			break;
		else
			++ntransfer;
		addr = (unsigned long)bh->b_data;
		size = bh->b_size;
		sector = bh->b_rsector;
		for (bh = bh->b_reqnext; bh != NULL; bh = bh->b_reqnext) {
			if ((addr + size) != (unsigned long)bh->b_data)
				break;	/* data is not contiguous */
			else if ((sector + (size / SECTOR_SIZE)) != bh->b_rsector)
				break;	/* sector is not contiguous */
			else if ((size + bh->b_size) > MBUS_LINEAR_MAX)
				break;	/* Size too large */
			else {	/* Merge the next BH */
				size += bh->b_size;
			}
		}	
	}
#else
	do {
		++ntransfer;
	} while ((bh = bh->b_reqnext) != NULL);
#endif

	return ntransfer;
}

struct buffer_head *em86xx_isaide_dma_setup(int read, struct buffer_head *bh, int stop_if_fragment)
{
#ifdef EM86XX_ISAIDE_DMA_MERGE_BH
	struct {
		unsigned long addr;
		unsigned int size;
		unsigned int sector;
	} transfer[2], *ptransfer;
	unsigned int ntransfer, size;

	/* Try to merge multiple BH if possible */
	for (ntransfer = size = 0, ptransfer = &transfer[0];; ptransfer++) {
		if (bh == NULL)
			break;
		else if (++ntransfer > 2)
			break;

		ptransfer->addr = (unsigned long)bh->b_data;
		ptransfer->size = bh->b_size;
		ptransfer->sector = bh->b_rsector;
		for (bh = bh->b_reqnext; bh != NULL; bh = bh->b_reqnext) {
			if ((ptransfer->addr + ptransfer->size) != (unsigned long)bh->b_data)
				break;	/* data is not contiguous */
			else if ((ptransfer->sector + (ptransfer->size / SECTOR_SIZE)) != bh->b_rsector)
				break;	/* sector is not contiguous */
			else if ((ptransfer->size + bh->b_size) > MBUS_LINEAR_MAX)
				break;	/* Size too large */
			else {	/* Merge the next BH */
				ptransfer->size += bh->b_size;
			}
		}	
		size += ptransfer->size;
	}

	if ((ntransfer > 2) && (stop_if_fragment != 0))
		return bh;
#else
	struct buffer_head *prevbh;

	struct {
		unsigned int addr;
		unsigned int size;
	} transfer[2], *ptransfer;
	unsigned int ntransfer, size;

	for (prevbh = NULL, ntransfer = 0, ptransfer = transfer, size = 0; bh != NULL; prevbh = bh, bh = bh->b_reqnext) {
		if (++ntransfer > 2) {
			if (stop_if_fragment)
				return bh;
			break;
		}
		ptransfer->addr = (unsigned int) bh->b_data;
		ptransfer->size = bh->b_size;
		// printk("MBUS : bh = %p, bh->next = %p, %08x, %08x\n", bh, bh->b_reqnext, ptransfer->addr, ptransfer->size);
		++ptransfer;
		size += bh->b_size;
	}
#endif

	// Use DMA mode transfer
	// Transfer starts when the IDE device is ready and asserts DMARQ signal high

	// setup PB automode registers
	__raw_writel(0, REG_BASE_HOST + PB_automode_start_address);
	__raw_writel(0x00140000 | ((read ? 1 : 0) << 16) | (size >> 1), REG_BASE_HOST + PB_automode_control);

	// setup MBUS
	if (bh)
		em86xx_mbus_setup_dma_common(read ? g_regbase_read : g_regbase_write, 
			em86xx_to_ncaddr(transfer[0].addr), size, em86xx_isaide_mbus_intr, bh);
	
	if (ntransfer == 1)
		em86xx_mbus_setup_dma_linear(read ? g_regbase_read : g_regbase_write, 
			em86xx_to_ncaddr(transfer[0].addr), transfer[0].size);
	else
		em86xx_mbus_setup_dma_double(read ? g_regbase_read : g_regbase_write, 
			em86xx_to_ncaddr(transfer[0].addr), transfer[0].size,
			em86xx_to_ncaddr(transfer[1].addr), transfer[1].size);

	return bh;
}

int em86xx_isaide_rwdma(ide_drive_t *drive, int read, struct request *rq)
{
#if 0
	// address lines are 24 bits long excluding MSB 4 bits
	if (((unsigned int) buf & 0x0f000000) != 0) {
		printk("EM86XX IDE : Buffer must reside under 16MB\n");
		return 1;
	}
	// size is 13 bits long
	if (((unsigned int) buflen & 0xffffe000) != 0) {
		printk("EM86XX IDE : Length must be smaller than 8KB(0x2000) (nsector = %d)\n", nsector);
		return 1;
	}
#endif

	// setup handler
	if (drive->media == ide_disk)
		ide_set_handler(drive, &em86xx_isaide_dma_intr, 2 * WAIT_CMD, NULL);

#ifdef EM86XX_ISAIDE_DMA_BOUNCE_BUF
	if (em86xx_isaide_dma_setup(read, rq->bh, 1)) {
		// copy buffer before transfer
		unsigned int buflen = SECTOR_SIZE * rq->nr_sectors;
		unsigned int buf = (unsigned int) g_bounce_buf;

		if (!read) {
			struct buffer_head *bh = rq->bh;

			do {
				memcpy((void *) buf, bh->b_data, bh->b_size);
				buf += bh->b_size;
			} while ((bh = bh->b_reqnext) != NULL);
		}
 
#ifndef EM86XX_BOUNCE_BUFFER_NOCACHE
		cpu_dcache_clean_range((unsigned long) g_bounce_buf, (unsigned long) (g_bounce_buf + buflen));
#endif

		// setup PB automode registers
		__raw_writel(0, REG_BASE_HOST + PB_automode_start_address);
		__raw_writel(0x00140000 | ((read ? 1 : 0) << 16) | (buflen >> 1), REG_BASE_HOST + PB_automode_control);

		// setup MBUS
		em86xx_mbus_setup_dma(read ? g_regbase_read : g_regbase_write, (unsigned long) g_bounce_buf, buflen, NULL, NULL);
	} else
		em86xx_ide_dma_cache(rq, 1, 0);
#else
	// cache control
	em86xx_ide_dma_cache(rq, 1, 0);

	// setup MBUS
	em86xx_isaide_dma_setup(read, rq->bh, 0);
#endif

	em86xx_isaide_dma_begin(drive);

	if (drive->media == ide_disk) {
		// send IDE command
		HWIF(drive)->OUTB(read ? (drive->addressing ? WIN_READDMA_EXT : WIN_READDMA) : (drive->addressing ? WIN_WRITEDMA_EXT : WIN_WRITEDMA), IDE_COMMAND_REG);
	}

	return 0;
}

void em86xx_isaide_mbus_intr(int irq, void *arg)
{
	int index = irq - IRQ_MBUS_W0;
	struct buffer_head *bh = (struct buffer_head *) arg;
	
	// printk("[MBUS] ");

	if (bh != NULL) 
		em86xx_isaide_dma_setup((index < 2) ? 1 : 0, bh, 0);
}

int em86xx_isaide_dma_end_io(ide_drive_t *drive)
{
	struct request *rq = HWGROUP(drive)->rq;

	int stat = em86xx_mbus_wait(rq->cmd == READ ? g_regbase_read : g_regbase_write, SBOX_IDEFLASH);

	if (stat != 0)
		return 1;

#ifdef EM86XX_ISAIDE_DMA_BOUNCE_BUF
	if (rq->cmd == READ) {
		if (em86xx_isaide_request_fragment(rq) > 2) {
			// copy buffer after transfer
			unsigned int buf = (unsigned int) g_bounce_buf;
			struct buffer_head *bh = rq->bh;
#ifndef EM86XX_BOUNCE_BUFFER_NOCACHE
			unsigned int buflen = SECTOR_SIZE * rq->nr_sectors;
	
			cpu_dcache_invalidate_range(buf, buf + buflen);
#endif
	
			do {
				memcpy(bh->b_data, (void *) buf, bh->b_size);
				buf += bh->b_size;
			} while ((bh = bh->b_reqnext) != NULL);
		} else {
			em86xx_ide_dma_cache(rq, 0, 1);
//printk("[I]");
//cpu_cache_clean_invalidate_all();
		}
	}
#else
	if (rq->cmd == READ) 
		em86xx_ide_dma_cache(rq, 0, 1);
#endif

	return 0;
}

ide_startstop_t em86xx_isaide_dma_intr(ide_drive_t *drive)
{
	int i;
	ide_hwif_t *hwif = HWIF(drive);
	struct request *rq = HWGROUP(drive)->rq;

	hwif->ide_dma_end(drive);

	for (i = rq->nr_sectors; i > 0;) {
		i -= rq->current_nr_sectors;
		ide_end_request(drive, 1);
	}

	return ide_stopped;
}

#endif

#ifdef CONFIG_BLK_DEV_EM86XX_ISAIDETIMING
/*
 * Calculate the peripheral bus timing needed for the register used by IDE
 * devices.
 */

/* 
 * The tables bellows use indices as Multi-word DMA 2, Multi-word DMA 1,
 * Multi-word DMA 0, PIO 4, PIO 3, PIO 2, PIO 1, and PIO 0 (From 0 to 7, 
 * respectively).
 */

/* For storing value pair */
typedef struct {
	unsigned char x;
	unsigned char y;
} value_pair;

/* Table used for calculate Ta and Tb */
static const value_pair tab_values[] = {
	{3,0}, {5,0}, {7,2}, {3,0}, {5,0}, {5,0}, {7,2}, {10,4}
};

/* Table used for calculate Tc */
static const value_pair tc_values[] = {
	{13,8}, {19,11}, {60,41}, {10,5}, {12,6}, {14,8}, {17,11}, {23,15}
};

/* Table used for calculate Td */
static const value_pair td_values[] = {
	{1,100}, {1,100}, {3,200}, {1,100}, {1,100}, {1,100}, {3,200}, {5,250}
};

/* 
 * Calculate the required timing for setting up peripheral bus timing register.
 *
 * Input: sysclk = system clock rate (in MHz)
 *        mode = IDE mode (0-2: Multiword DMA mode 2-0, 3-7: PIO mode 4-0)
 * Return: 0 = Error
 *         else = value for setting up the timing register
 */
unsigned long get_pb_ide_timing(unsigned int sysclk, int mode)
{
	unsigned char ta = 0;
	unsigned char tb = 0;
	unsigned char tc = 0;
	unsigned char td = 0;
	long temp = 0;
	const value_pair *val_ptr = (const value_pair *)0;

	/* The formula is only effective between 100-240MHz */
	if ((sysclk < 100) && (sysclk > 240))
		return(0);
	else if ((mode < 0) && (mode > 7)) /* Mode index is 0-7 only */
		return(0);

	/* (x, y) = tab_values[mode] */
	/* (Ta + Tb) = Temp = (ceiling((sysclk - 100) * x / 140) + y) */
	/* Ta = ceiling(Temp / 2) */
	/* Tb = (Temp / 2) */
	val_ptr = &tab_values[mode];
	temp = ((sysclk - 100) * val_ptr->x) / 140;
	if ((temp * 140) < ((sysclk - 100) * val_ptr->x))
		temp++;
	temp += val_ptr->y;
	tb = ta = (unsigned char)((temp >> 1) & 0xff);
	if (temp & 0x1)
		ta++;

	/* (x, y) = tc_values[mode] */
	/* Tc = (ceiling((sysclk - 100) * x / 140) + y) */
	val_ptr = &tc_values[mode];
	tc = (unsigned char)(((sysclk - 100) * val_ptr->x) / 140) & 0xff;
	if ((tc * 140) < ((sysclk - 100) * val_ptr->x))
		tc++;
	tc += val_ptr->y;

	/* (x, y) = td_values[mode] */
	/* Td = ceiling(sysclk + y - 400) * x / y) */
	val_ptr = &td_values[mode];
	if ((temp = (sysclk + val_ptr->y) - 400) < 0)
		td = 0; /* It's possible that Td < 0, so we set it to 0 */
	else {
		td = (unsigned char)(((temp * val_ptr->x) / val_ptr->y) & 0xff);
		if ((td * val_ptr->y) < (temp * val_ptr->x))
			td++;
	}
	return((td << 24) | (tc << 16) | (tb << 8) | ta);
}



#endif
void __exit em86xx_isaide_exit(void)
{
#ifdef EM86XX_ISAIDE_DMA_BOUNCE_BUF
#ifdef EM86XX_BOUNCE_BUFFER_NOCACHE
        g_bounce_buf = (unsigned char *)em86xx_to_caddr((unsigned long)g_bounce_buf);
#endif
        free_pages((unsigned long)g_bounce_buf, EM86XX_ISAIDE_DMA_BOUNCE_BUF_ORDER);
#endif
        return;
}


#ifdef MODULE
module_init(em86xx_isaide_init);
module_exit(em86xx_isaide_exit);
#endif

