/*
 * drivers/mtd/maps/em86xx_map.c
 *
 * Copyright (C) 2003-2004, Sigma Designs Inc.
 *
 * Mapping of flash in EM86XX
 *
 * by Ho Lee
 */

#include <linux/module.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <asm/io.h>
#include <asm/hardware.h>
#include <linux/mtd/mtd.h>
#include <linux/mtd/map.h>
#include <linux/config.h>

#ifdef CONFIG_MTD_PARTITIONS
#include <linux/mtd/partitions.h>
#include <asm/arch/board/mtdpartitions.h>
#endif

#define EM86XX_MTD_MAXFLASHS    2

#ifdef CONFIG_MTD_EM86XX_16BIT_BUSWIDTH
#define EM86XX_MTD_BUSWIDTH     2
#else
#define EM86XX_MTD_BUSWIDTH     1
#endif

#define EM86XX_MTD_WINDOWSIZE   0x01000000

// partition table : refer to physmap driver
#ifdef CONFIG_MTD_PARTITIONS
#ifdef CONFIG_MTD_CMDLINE_PARTS
static struct mtd_partition *mtd_parts = 0;
static int mtd_parts_nb = 0;
#else
static struct mtd_partition em86xxmap_partitions[] = {
    EM86XX_MTD_PARTITIONS
};

#define NUM_PARTITIONS  (sizeof(em86xxmap_partitions) / sizeof(struct mtd_partition))
#endif // CONFIG_MTD_CMDLINE_PARTS
#endif // CONFIG_MTD_PARTITIONS

// FTL : Flash TransLation 
// input: flash address - ouput: cpu address
#define FTL(x)                  ((x) + map->map_priv_1)

static struct mtd_info *mymtd[EM86XX_MTD_MAXFLASHS];

__u8 em86xx_mtdmap_read8(struct map_info *map, unsigned long ofs)
{
    return __raw_readb(FTL(ofs));
}

__u16 em86xx_mtdmap_read16(struct map_info *map, unsigned long ofs)
{
    return __raw_readw(FTL(ofs));
}

__u32 em86xx_mtdmap_read32(struct map_info *map, unsigned long ofs)
{
    return __raw_readl(FTL(ofs));
}

void em86xx_mtdmap_copy_from(struct map_info *map, void *to, unsigned long from, ssize_t len)
{       
    memcpy(to, (void *) FTL(from), len);
}

void em86xx_mtdmap_write8(struct map_info *map, __u8 d, unsigned long adr)
{
    __raw_writeb(d, FTL(adr));
    mb();
}

void em86xx_mtdmap_write16(struct map_info *map, __u16 d, unsigned long adr)
{
    __raw_writew(d, FTL(adr));
    mb();
}

void em86xx_mtdmap_write32(struct map_info *map, __u32 d, unsigned long adr)
{
    __raw_writel(d, FTL(adr));
    mb();
}

void em86xx_mtdmap_copy_to(struct map_info *map, unsigned long to, const void *from, ssize_t len)
{
    memcpy((void *) FTL(to), from, len);
}

struct map_info em86xx_mtdmap_map[EM86XX_MTD_MAXFLASHS] = {
    {
        name: "EM86XX mapped flash",
        size: EM86XX_MTD_WINDOWSIZE, 
        buswidth: EM86XX_MTD_BUSWIDTH,
        read8: em86xx_mtdmap_read8,
        read16: em86xx_mtdmap_read16,
        read32: em86xx_mtdmap_read32,
        copy_from: em86xx_mtdmap_copy_from,
        write8: em86xx_mtdmap_write8,
        write16: em86xx_mtdmap_write16,
        write32: em86xx_mtdmap_write32,
        copy_to: em86xx_mtdmap_copy_to,
        map_priv_1 : MEMORY_BASE_HOST_PFLASH,
    },
#ifdef CONFIG_MTD_EM86XX_FLASH2
    {
        name: "EM86XX mapped flash 2",
        size: EM86XX_MTD_WINDOWSIZE, 
        buswidth: EM86XX_MTD_BUSWIDTH,
        read8: em86xx_mtdmap_read8,
        read16: em86xx_mtdmap_read16,
        read32: em86xx_mtdmap_read32,
        copy_from: em86xx_mtdmap_copy_from,
        write8: em86xx_mtdmap_write8,
        write16: em86xx_mtdmap_write16,
        write32: em86xx_mtdmap_write32,
        copy_to: em86xx_mtdmap_copy_to,
        map_priv_1 : MEMORY_BASE_HOST_PFLASH2,
    },
#endif
};

int __init init_em86xx_mtdmap(void)
{
    int i, nfound = 0;
#ifdef CONFIG_MTD_CMDLINE_PARTS
    int nb_parts = 0, parsed_nr_parts = 0;
    struct mtd_partition *parts;
    const char *part_type;
    static struct mtd_partition *parsed_parts;
#endif

    printk(KERN_INFO "Probing EM86XX Flash Memory\n");

    for (i = 0; i < EM86XX_MTD_MAXFLASHS && em86xx_mtdmap_map[i].size; ++i) {
        if (((mymtd[i] = do_map_probe("cfi_probe", &em86xx_mtdmap_map[i])) != NULL) ||
            ((mymtd[i] = do_map_probe("jedec_probe", &em86xx_mtdmap_map[i])) != NULL)) {
            mymtd[i]->module = THIS_MODULE;
            add_mtd_device(mymtd[i]);
            ++nfound;
            continue;
        } 
    }

    if (nfound == 0) {
        printk("No Flash found\n");
        return -ENXIO;
    }

    // Add partition definitions
#ifdef CONFIG_MTD_PARTITIONS
#ifdef CONFIG_MTD_CMDLINE_PARTS
    mtd_parts_nb = parse_cmdline_partitions(mymtd, &mtd_parts, "em86xx");
    if (mtd_parts_nb > 0) {
        printk(KERN_NOTICE 
               "Using command line partition definition\n");
        add_mtd_partitions(mymtd[0], mtd_parts, mtd_parts_nb);
    }
/*
    if (parsed_nr_parts == 0) {
        int ret = parse_cmdline_partitions(mymtd, &parsed_parts, "physmap");
        if (ret > 0) {
            part_type = "Command Line";
            parsed_nr_parts = ret;
        }
    }

    if (parsed_nr_parts > 0) {
        parts = parsed_parts;
        nb_parts = parsed_nr_parts;
    }

    if (nb_parts == 0) {
        printk(KERN_NOTICE "physmap: no partition info available, "
                "registering whole flash at once\n");
        add_mtd_device(mymtd);
    } else {
        printk(KERN_NOTICE "physmap: Using %s partition definition\n",
                part_type);
        add_mtd_partitions(mymtd, parts, nb_parts);
    }
*/
#else
    add_mtd_partitions(mymtd[0], em86xxmap_partitions, NUM_PARTITIONS);
#endif // CONFIG_MTD_CMDLINE_PARTS
#endif // CONFIG_MTD_PARTITIONS

    return 0;
}

static void __exit cleanup_em86xx_mtdmap(void)
{
    int i;

    for (i = 0; i < EM86XX_MTD_MAXFLASHS; ++i) {
        if (mymtd[i]) {
            del_mtd_device(mymtd[i]);
            map_destroy(mymtd[i]);
            mymtd[i] = NULL;
        }
    }
}

module_init(init_em86xx_mtdmap);
module_exit(cleanup_em86xx_mtdmap);

MODULE_LICENSE("GPL");
MODULE_DESCRIPTION("EM86XX MTD map driver");
