/*
 *
 */
#ifdef BOOTLOADER
#include "config.h"
#include "version.h"
#include "util.h"
#include "net.h"
#include "net_ipv4.h"
#include "em86xxapi.h"
#include "irqs.h"
#include "errno.h"
#include "uart.h"
#include "memcfg.h"
#else
#include <linux/config.h>
#include <linux/kernel.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/spinlock.h>
#include <linux/interrupt.h>
#include <linux/signal.h>
#include <linux/sched.h>
#include <linux/module.h>
#include <linux/compatmac.h>
#include <linux/mm.h>
#include <linux/init.h>
#include <linux/timer.h>
#include <linux/mii.h>
#include <asm/virtconvert.h>
#include <asm/arch/irqs.h>
#include <asm/arch/memcfg.h>
#endif //BOOTLOADER

#include "em86xx_eth.h"

/* Uncomment this for different messages */
//#define ETH_DEBUG 
#define ETH_ERRMSG 
#define ETH_MSG

#ifdef BOOTLOADER
#define DRIVER "em86xx_eth"

/* For static buffer allocation */
// #define STATIC_BUF_ALLOC
#define printk          uart_printf

#define PAGE_SHIFT	12
#define PAGE_SIZE	(1UL << PAGE_SHIFT)
#define PAGE_MASK	(~(PAGE_SIZE-1))
#define MAX_ORDER	10

#endif /* BOOTLOADER */

/* For software filtering*/
//#define USE_SW_FILTERING
#define USE_HW_FILTERING

#ifdef ETH_DEBUG
#define DBG_PRINT	printk
#else
static void inline DBG_PRINT(const char *x, ...) { ; }
#endif /* ETH_DEBUG */

#ifdef ETH_ERRMSG
#define ERR_PRINT	printk
#else
static void inline ERR_PRINT(const char *x, ...) { ; }
#endif /* ETH_ERRMSG */

#ifdef ETH_MSG
#define MSG_PRINT	printk
#else
static void inline MSG_PRINT(const char *x, ...) { ; }
#endif /* ETH_ERRMSG */

/* Hack: this is the mac address by default */
#define DEF_MAC_HI	0x00000011
#ifdef BOOTLOADER
#define DEF_MAC_LO	0xdeadbeef
#else
#define DEF_MAC_LO	0xaabbccdd
#endif

/* Default maximum link wait */
#define DEF_LINK_LOOP		1000
#define DEF_MAX_LINK_LOOP	0x1000000

#define ETH_IRQ_FLAGS	(DmaIntNormal	  | DmaIntAbnormal    | DmaIntRxStopped		| \
			DmaIntRxNoBuffer  | DmaIntRxCompleted)

#define ETH_RXTX_FLAGS	(DmaTxStart       | DmaRxStart)

#ifndef USE_HW_FILTERING 
#define ETH_MAC_FLAGS   (MacTxEnable      |  MacRxEnable       | MacPromiscuousModeOn	| \
                        MacPadStripEnable |  MacHashFilterOn   | MacPerfectFilterOff 	| \
                        MacRetryDisable   | MacFullDuplex      | MacHeartBeatOff)
#else 
#define ETH_MAC_FLAGS	(MacTxEnable   	  | MacRxEnable        | MacPromiscuousModeOff  | \
			MacPadStripEnable | MacHashFilterOn    | MacPerfectFilterOn     | \
                        MacRetryDisable   | MacFullDuplex      | MacMulticastFilterOn   | \
			MacHeartBeatOff)
#define DEFAULT_BROADCAST_ADDRESS { 0xFF,0xFF,0xFF,0xFF,0xFF,0xFF }
#endif /*USE_HW_FILTERING*/

#define MP_FRAME_FLAGS  (DescRxLast        | DescRxFirst)
#define TX_DESC1_FLAGS 	(DescTxLast        | DescTxFirst        | DescChain)
#define RX_ERROR_FLAGS  (DescFilteringFail | DescError		| DescRxTruncated	| \
			DescLengthError    | DescRxDamaged 	| DescRxLongFrame	| \
			DescRxLongFrame	   | DescRxCollision	| DescRxMiiError	| \
			DescRxCrc)
#define TX_ERROR_FLAGS  (DescTxLostCarrier | DescTxNoCarrier	| DescTxLateCollision 	| \
			DescTxExcCollisions| DescTxHeartbeatFail| DescTxUnderflow)

/* Number of descriptors and buffer size */
#define MIN_NUM_RDESC	8
#define MIN_NUM_TDESC	8
#define NUM_RDESC 	32	
#define NUM_TDESC	16	
#define R_BUF_SIZE	0x700	/* needs to be < 2KB */
#define T_BUF_SIZE	0x700	/* needs to be < 2KB */

/* Private data for this ethernet device */
typedef struct {
	struct net_device_stats stats; /* stats */
	unsigned long last_rxidx;      /* last rx idx to descriptor */
	unsigned long next_txidx;      /* next tx idx to descriptor */
} EM86XX_ETH_PRIV;

/* Descriptors and buffers: notice the alignments */
#ifdef STATIC_BUF_ALLOC 
static struct em86xx_desc eth_rxdsc[NUM_RDESC] __attribute__ ((__aligned__(16)));
static struct em86xx_desc eth_txdsc[NUM_TDESC] __attribute__ ((__aligned__(16)));
static unsigned char eth_rxbuf[NUM_RDESC * R_BUF_SIZE] __attribute__ ((__aligned__(16)));
static unsigned char eth_txbuf[NUM_TDESC * T_BUF_SIZE] __attribute__ ((__aligned__(16)));
#else
static struct em86xx_desc *eth_rxdsc 	= NULL;
static struct em86xx_desc *eth_txdsc 	= NULL;
static unsigned char *eth_rxbuf 	= NULL;
static unsigned char *eth_txbuf		= NULL;
static unsigned long desc_page  	= 0L;
static unsigned long rxbuf_pages 	= 0L;
static unsigned long txbuf_pages 	= 0L;
#ifndef BOOTLOADER
static unsigned long rxbuf_order 	= 0L;
static unsigned long txbuf_order 	= 0L;
#endif
#endif

/* Number of descriptors */
static unsigned long num_rxdesc = NUM_RDESC;
static unsigned long num_txdesc = NUM_TDESC;

/* The real data pointer to be accessed in non-cache region */
static volatile struct em86xx_desc *rxdsc = NULL;
static volatile struct em86xx_desc *txdsc = NULL;
static volatile unsigned char *rxbuf	  = NULL;
static volatile unsigned char *txbuf	  = NULL;

/* Reference count for device */
static int dev_count = 0;

/* For storing MAC address */
static unsigned long def_mac_hi = DEF_MAC_HI;
static unsigned long def_mac_lo = DEF_MAC_LO;

/* Link status flag */
static volatile int link_up 		 = 0;
static volatile int link_configured	 = 0;
static unsigned long max_link_loop	 = 0;
static unsigned long ld_count		 = 0;
static int reset_flag			 = 0;

static int em86xx_eth_reset_desc(struct net_device *dev, int *reset);

#ifdef BOOTLOADER
static void em86xx_eth_intr_handler(int irq, void *dev_id);
#else
static struct timer_list eth_timer;
static struct net_device em86xx_eth_dev;
static void em86xx_eth_intr_handler(int irq, void *dev_id, struct pt_regs *regs);

MODULE_AUTHOR("Craig Qu");
MODULE_DESCRIPTION("Sigma Designs ethernet driver");
MODULE_LICENSE("GPL");

MODULE_PARM(def_mac_hi, "i");
MODULE_PARM(def_mac_lo, "i");

#ifndef STATIC_BUF_ALLOC 
MODULE_PARM(num_rxdesc, "i");
MODULE_PARM(num_txdesc, "i");
#endif
#endif /* BOOTLOADER */

/* Get the device mac address */
static int em86xx_get_macaddr(unsigned char *addr)
{
#ifdef BOOTLOADER
	if(	mac_read(addr) == -1) {
		*addr++ = (def_mac_hi & 0x0000ff00) >> 8;
		*addr++ = (def_mac_hi & 0x000000ff);
		*addr++ = (def_mac_lo & 0xff000000) >> 24;
		*addr++ = (def_mac_lo & 0x00ff0000) >> 16;
		*addr++ = (def_mac_lo & 0x0000ff00) >> 8;
		*addr   = (def_mac_lo & 0x000000ff);
	}
#else
#ifdef CONFIG_SD_BOOTLOADER_INTEGRATION
	{
		memcfg_t *memcfg_ptr = (memcfg_t *)MEMCFG_START;
		if ((is_valid_memcfg(memcfg_ptr) != 0) && ((memcfg_ptr->mac_hi != 0) || (memcfg_ptr->mac_lo != 0))) {
			def_mac_hi = memcfg_ptr->mac_hi;
			def_mac_lo = memcfg_ptr->mac_lo;
		}
	}
#endif
	*addr++ = (def_mac_hi & 0x0000ff00) >> 8;
	*addr++ = (def_mac_hi & 0x000000ff);
	*addr++ = (def_mac_lo & 0xff000000) >> 24;
	*addr++ = (def_mac_lo & 0x00ff0000) >> 16;
	*addr++ = (def_mac_lo & 0x0000ff00) >> 8;
	*addr   = (def_mac_lo & 0x000000ff);
#endif	
	return 0;

}

/* Setting up MAC address of ethernet PHY and device data */
static int em86xx_set_mac(struct net_device *dev)
{
	unsigned long hi_mac, low_mac;

	/* Set up device mac address */
	if (em86xx_get_macaddr(dev->dev_addr))
		return(-EIO);

	if (!is_valid_ether_addr(dev->dev_addr))
		MSG_PRINT("%s: bogus mac address detected.\n", dev->name);

	hi_mac =  (dev->dev_addr[5] << 8)  | dev->dev_addr[4];
	low_mac = (dev->dev_addr[3] << 24) | (dev->dev_addr[2] << 16) | 
		  (dev->dev_addr[1] << 8)  | dev->dev_addr[0];

	/* Set up MAC address */
	em86xx_write_reg(EM86XX_MACAHR_REG, hi_mac);
	em86xx_write_reg(EM86XX_MACALR_REG, low_mac);

	return 0;
}

#ifndef BOOTLOADER
/* Setting customized mac address */
static int em86xx_eth_set_macaddr(struct net_device *dev, void *addr)
{
	unsigned long hi_mac, low_mac;

	/* Check if given address is valid ethernet MAC address */
	if (!is_valid_ether_addr(addr))
		return(-EIO);

	/* Turn off IRQ and stop receive/transmit */
	em86xx_write_reg(EM86XX_CR_REG, 0);
	em86xx_write_reg(EM86XX_IER_REG, 0);

	netif_carrier_off(dev); /* Shutdown interface */

	/* Save the customize mac address */
	memcpy(dev->dev_addr, addr, 6);
	hi_mac  = (dev->dev_addr[5] << 8) | dev->dev_addr[4];
	low_mac = (dev->dev_addr[3] << 24)| (dev->dev_addr[2] << 16) | 
		  (dev->dev_addr[1] << 8) | dev->dev_addr[0];

	def_mac_hi = hi_mac;
	def_mac_lo = low_mac;

	/* Set up MAC address */
	em86xx_write_reg(EM86XX_MACAHR_REG, hi_mac );
	em86xx_write_reg(EM86XX_MACALR_REG, low_mac );

	/* Restart everything */
	link_up = link_configured = 0;

	return 0;
}
#endif /* BOOTLOADER */

/*serach for phy address */
static int em86xx_phy_probe(void)
{
        int phy_addr 	= 0;
	u16 phy_id 	= 0;
        u16 phy_status 	= 0;

        /* search for total of 32 possible mii phy addresses */
        for (phy_addr = 0; phy_addr < 32; phy_addr++) {
                register int i;
                for(i = 0; i < 2; i++)
                        phy_status = em86xx_mii_read(phy_addr,GEN_sts);

                if (phy_status != 0xffff && phy_status != 0x0000)
			break;
	}

	phy_id =  em86xx_mii_read(phy_addr,GEN_id_lo);
	DBG_PRINT("Found PHY RTL%04x at Addr = %d Status=0x%x\n", phy_id, phy_addr, phy_status);
 
	return phy_addr;
}

/* Initialize the ethernet link status, mac, and flow control */
static int em86xx_link_config(struct net_device *dev, int timeout)
{
	unsigned long word = 0;
 	int i = timeout, fdx = 0, speed = 0, phy_addr= 0;

	/* If link is up and configured, don't do anything */
	if ((link_up != 0) && (link_configured != 0))
		return 0;
	else
		DBG_PRINT("%s: re-configure link (%ld).\n", dev->name, i);

	/* Turn off IRQ and stop receive/transmit */
	em86xx_write_reg(EM86XX_CR_REG, 0);
	em86xx_write_reg(EM86XX_IER_REG, 0);

	/*get PHY addr*/	
	phy_addr = em86xx_phy_probe();

        /* wait for autonegotiation complete */
        while( i-- > 0 ) {
                if( (em86xx_mii_read(phy_addr, GEN_sts) & AUTOCMPLT) != 0 )
			break;
        }
	if (i == 0)
		MSG_PRINT("%s: auto-negotiation failed.\n", dev->name);

        if( (em86xx_mii_read(phy_addr, GEN_sts) & LINK) == 0 ) {
		printk( "\nem86xx_eth::open(%s) - no link\n", dev->name );
		return -ENODEV;
        } else
  		link_up = 1;

        fdx = (em86xx_mii_read(phy_addr, GEN_ctl) & DUPLEX) != 0;
        printk(  "\nem86xx_eth::open(%s) - %s-duplex mode\n",
                       dev->name, fdx ? "Full" : "Half" );

        speed = (em86xx_mii_read(phy_addr, GEN_ctl) & SPEED) != 0;
        printk(  "em86xx_eth::open(%s) - 10%s Mbit/s\n",
                           dev->name, speed ? "0" : "" );

	/* set the descriptor base address */
	em86xx_write_reg(EM86XX_RLBAR_REG, (u32)rxdsc);
	em86xx_write_reg(EM86XX_TLBAR_REG, (u32)txdsc);

	
	/* configure PHY if, MII with 100Mbps or 10Mbps */
	word = MiiIfExtPhyMii | ((speed) ? MiiIfSpeed100M : MiiIfSpeed10M);
	em86xx_write_reg(EM86XX_IFR_REG, word); 

	/* set bus mode: burst length = 32 | BLE | ~DBO */
	word = (32 << 8) /*| (1 << 7) | (1 << 20)*/ ;
	em86xx_write_reg(EM86XX_BMR_REG, word); 

	/* enable MAC flow ctrl */
	word = (1 << 1);
	em86xx_write_reg(EM86XX_FCR_REG, word); 

	/* configure MAC ctrller */
	word = ETH_MAC_FLAGS;
	word |= MacLoopbackOff; 
	if (fdx)
  		word |= MacFullDuplex;    
	em86xx_write_reg(EM86XX_MACCR_REG, word);

	/* Turn on the IRQ and start receive/transmit */
	em86xx_write_reg(EM86XX_IER_REG, ETH_IRQ_FLAGS);
	em86xx_write_reg(EM86XX_CR_REG, ETH_RXTX_FLAGS); 

	link_configured = 1;

	return 0;
}

#ifdef USE_HW_FILTERING
static int em86xx_hash( u8 addr[6] )
{
        int i;
        u32 crc = 0xFFFFFFFF;
        u32 poly = 0xEDB88320;
        //u32 poly = 0x04C11DB7;
        for( i=0; i<6; i++ ){
                int bit;
                u8 data = addr[i];

                for( bit=0; bit<8; bit++ ){
                        int p = (crc^data) & 1;

                        crc >>= 1;
                        if( p != 0 ) crc ^= poly;
                        data >>= 1;
                }
        }

        /* return upper 6 bits */
        return (crc>>26) & 0x3F;
}
#endif




/* Ethernet hardware initialization */
static int em86xx_eth_hw_init(struct net_device *dev)
{
#ifdef USE_HW_FILTERING
	int n = 0;
 	u8  broadcast_addr[6]  = DEFAULT_BROADCAST_ADDRESS;
#endif

	/* reset dma engine*/
	em86xx_write_reg(EM86XX_BMR_REG, DmaResetOn);
	/*em86xx_write_reg(EM86XX_BMR_REG, DmaResetOff);*/

	if (em86xx_set_mac(dev))
		return(-EIO);

	/* stop MAC engine */
	em86xx_write_reg(EM86XX_CR_REG, 0x0); 
#ifndef USE_HW_FILTERING
	/* set up multicast hash table to MCHTHR MCHTLR */
	/* set multicast hash table to accept all */
	em86xx_write_reg(EM86XX_MCHTHR_REG, 0xffffffff); 
	em86xx_write_reg(EM86XX_MCHTLR_REG, 0xffffffff); 
#else
	/* clear hash table */
        em86xx_write_reg( EM86XX_MCHTHR_REG, 0 );
        em86xx_write_reg( EM86XX_MCHTLR_REG, 0 );

        /* set the broadcast address */
        n = em86xx_hash(broadcast_addr);
        if( n >= 32 )
        	em86xx_write_reg(EM86XX_MCHTHR_REG, 1 << (n-32) );
        else
                em86xx_write_reg(EM86XX_MCHTLR_REG, 1 << n );
#endif

	/* resetting descriptors */
#ifdef BOOTLOADER
        if (em86xx_eth_reset_desc(dev, &reset_flag))
#else
        if (em86xx_eth_reset_desc(&em86xx_eth_dev, &reset_flag))
#endif
		return(-EIO);

	/* configure IFR with PHY and speed, as well as PHY */
	em86xx_link_config(dev, max_link_loop);

	return 0;
}

#ifndef BOOTLOADER
/* Monitor the status of link, re-do link initialization if necessary. */
static void em86xx_eth_link_monitor(unsigned long dummy)
{
	unsigned long link_status;
	struct net_device *dev = (struct net_device *)dummy;
	int res;

	/* Setup the link/configuration is not done so. */
	if ((link_up == 0) || (link_configured == 0)) {
		res = em86xx_eth_hw_init(dev);
	}

	/* Get link status and adjust the delay loop accordingly */
	link_status = (em86xx_mii_read(1, GEN_sts) & LINK) ;
	if (link_status) {
  		if ((link_configured != 0) && (link_up != 0)) {
    			if (!netif_carrier_ok(dev)) {
	      			netif_carrier_on(dev); /* Up and running, turn netif on */
	      	        	max_link_loop = DEF_LINK_LOOP;
	      	        	ld_count >>= 2;

        	      		if (ld_count == 0)
      					ld_count = 1;
	    		}
	        } else {
	      		ld_count <<= 1;
	      		max_link_loop = ld_count * DEF_LINK_LOOP;
	
	      		if (max_link_loop > DEF_MAX_LINK_LOOP)
	      			max_link_loop = DEF_MAX_LINK_LOOP;
	
	              	netif_carrier_off(dev);
	              	link_configured = 0;
	     	}
	} else {
		MSG_PRINT("%s: detected link down.\n", dev->name);
	        ld_count <<= 1;
	        max_link_loop = ld_count * DEF_LINK_LOOP;

	  	if (max_link_loop > DEF_MAX_LINK_LOOP)
    			max_link_loop = DEF_MAX_LINK_LOOP;

	  	netif_carrier_off(dev);
	  	link_up = link_configured = 0;
	}

	/* Schedule for the next time */
	mod_timer(&eth_timer, jiffies + HZ*5); 
}
#endif

/* Setting rx/tx descriptors */
static void em86xx_eth_setup_desc(void)
{
	register int i;
	struct em86xx_desc *desc_ptr = NULL;
	unsigned long base_addr = 0;

	/* Setup rx descriptor */
	desc_ptr = (struct em86xx_desc *)(rxdsc);
	for (i = 0; i < (num_rxdesc - 1); i++, desc_ptr++) {
		desc_ptr->desc3 = (unsigned long)(desc_ptr + 1);
		desc_ptr->desc1 = (DescChain | R_BUF_SIZE);
  		desc_ptr->desc0 = DescOwnByDma; 
	}
	desc_ptr->desc3 = (unsigned long)(rxdsc);
	desc_ptr->desc1 = (DescChain | DescEndOfRing | R_BUF_SIZE);
	desc_ptr->desc0 = DescOwnByDma; 

	/* Setup tx descriptor */
	desc_ptr = (struct em86xx_desc *)(txdsc); 
	for (i =0;  i < (num_txdesc - 1); i++, desc_ptr++) {
		desc_ptr->desc3 = (unsigned long)(desc_ptr + 1);
		desc_ptr->desc1 = (TX_DESC1_FLAGS | T_BUF_SIZE);
  		desc_ptr->desc0 = DescOwnByCPU;
	}
	desc_ptr->desc3 = (unsigned long)(txdsc);
	desc_ptr->desc1 = (TX_DESC1_FLAGS | DescEndOfRing | T_BUF_SIZE);
	desc_ptr->desc0 = DescOwnByCPU;

	/* Point rx descriptor to buffer */
	desc_ptr = (struct em86xx_desc *)(rxdsc); 
      	base_addr = (unsigned long)(rxbuf); 
	for (i = 0; i < num_rxdesc; i++, desc_ptr++, base_addr += R_BUF_SIZE) 
		desc_ptr->desc2 = base_addr;

	/* Point tx descriptor to buffer */
	desc_ptr = (struct em86xx_desc *)(txdsc);
        base_addr = (unsigned long)(txbuf);
	for (i = 0; i < num_txdesc; i++, desc_ptr++, base_addr += T_BUF_SIZE) 
		desc_ptr->desc2 = base_addr;

}

/* Starting up the ethernet device */
static int em86xx_eth_open(struct net_device *dev)
{
	EM86XX_ETH_PRIV *private = (EM86XX_ETH_PRIV *)dev->priv;

	if ((dev == NULL) || (private == NULL))
		return(-EIO);
	DBG_PRINT("%s: starting interface.\n", dev->name);

#ifdef BOOTLOADER
	em86xx_eth_hw_init(dev);
        dev->state = NETDEV_UP;
#else
	/* Schedule timer for monitoring link status */
	init_timer(&eth_timer);
	eth_timer.function = em86xx_eth_link_monitor;
	eth_timer.data = (unsigned long)dev;
	mod_timer(&eth_timer, jiffies + HZ);

	netif_start_queue(dev);
#endif /*BOOTLOADER*/

	return 0;
}

/* Stopping the ethernet device */
static int em86xx_eth_close(struct net_device *dev)
{
	if (dev == NULL)
  		return(-EIO);

	/* Turn off IRQ and stop receive/transmit */
	em86xx_write_reg(EM86XX_CR_REG, 0);
	em86xx_write_reg(EM86XX_IER_REG, 0);

#ifdef BOOTLOADER
	dev->state = NETDEV_DOWN;
#else
	/* Kill timer */
	del_timer_sync(&eth_timer);

	/* Stop the transmit queue */
	netif_stop_queue(dev);
	link_up = link_configured = 0;

#endif /*BOOTLOADER*/
	return 0;
}

#ifndef BOOTLOADER
/* Setup multicast list */
static void em86xx_eth_set_multicast_list(struct net_device *dev)
{
	/* Multicast hash filter */
        u32 mc_filter[2];
        int i, rx_mode;

	rx_mode =  em86xx_read_reg(EM86XX_MACCR_REG);  
        DBG_PRINT(KERN_DEBUG"%s:   set_rx_mode(0x%x) done -- rx_mode=0x%x.\n",
                           dev->name, dev->flags, rx_mode);

        if (dev->flags & IFF_PROMISC) {
                printk("%s: Promiscuous mode enabled.\n", dev->name);
                rx_mode |= MacPromiscuousModeOn;
                mc_filter[1] = mc_filter[0] = 0xffffffff;
        } else {
                struct dev_mc_list *mclist;
                rx_mode &=  ~MacPromiscuousModeOn;
                mc_filter[1] = mc_filter[0] = 0;
                for (i = 0, mclist = dev->mc_list; mclist && i < dev->mc_count;
                         i++, mclist = mclist->next){
			int n = 0;
			n = em86xx_hash(mclist->dmi_addr);
			if( n >= 32 )
	        	        mc_filter[1] |= 1 << (n-32) ;
        		else
	        	        mc_filter[0] |= 1 << n ;
		}
        }
	em86xx_write_reg(EM86XX_MACCR_REG, rx_mode );
        em86xx_write_reg(EM86XX_MCHTLR_REG, mc_filter[0]);
        em86xx_write_reg(EM86XX_MCHTHR_REG, mc_filter[1]);

        return;
}
#endif

/* Transmit a packet */
#ifdef BOOTLOADER
static int em86xx_eth_tx(struct sk_buff *skb, struct net_device *dev, int async)
#else
static int em86xx_eth_tx(struct sk_buff *skb, struct net_device *dev)
#endif
{
	EM86XX_ETH_PRIV *private = NULL;
	volatile struct em86xx_desc *desc_ptr = NULL;
	struct net_device_stats *stats = NULL;
	unsigned long length 	= 0;
	//int j;

	if ((skb == NULL) || (dev == NULL)) {
		DBG_PRINT("em86xx_eth_tx: null dev\n");
		return(-EIO);
	} else if ((link_up == 0) || (link_configured == 0)) {
		DBG_PRINT("em86xx_eth_tx: link is not up yet\n");
#ifndef BOOTLOADER
		netif_carrier_off(dev);
#endif
		return(-EIO); 
	} else if ((length = ((ETH_ZLEN < skb->len) ? skb->len : ETH_ZLEN)) > T_BUF_SIZE) {
      		DBG_PRINT("em86xx_eth_tx: too big\n");
		return(-ENOMEM); 
	}

	private = (EM86XX_ETH_PRIV *)dev->priv;
	stats = &(private->stats);

	/* Send packet to device */
	desc_ptr = (volatile struct em86xx_desc *)(&txdsc[private->next_txidx]);
	if (desc_ptr->desc0 & DescOwnByDma) {
		stats->tx_dropped++;

        	ERR_PRINT("%s desc_ptr=0x%x: tx error (descriptor not owned by CPU).\n",
				 dev->name, (u32)desc_ptr);

		return(-EIO);
	} 

	/* Check if previous transmission has error */
	if (desc_ptr->desc0 & TX_ERROR_FLAGS) {
		ERR_PRINT("%s: tx error 0x%08lx\n", dev->name, desc_ptr->desc0);

		/* Update stats */
		stats->tx_errors++;
		stats->tx_packets--;
		stats->tx_bytes -= (desc_ptr->desc1 & DescSize1Mask);
	}

	/* Copy packet data to tx buffer */
	DBG_PRINT("tx copy data from skb->data=0x%x to desc2=0x%x length=0x%lx\n",
		(u32)skb->data, desc_ptr->desc2, length );

	memcpy((unsigned char *)(desc_ptr->desc2), skb->data, length);

	/* Setup tx descriptor */

	desc_ptr->desc1 = ( TX_DESC1_FLAGS | length );
	if (private->next_txidx == (num_txdesc - 1))
		desc_ptr->desc1 |=  DescEndOfRing ;
	desc_ptr->desc0 |= DescOwnByDma;

#ifdef BOOTLOADER
        private->next_txidx = imodulus((private->next_txidx + 1), num_txdesc);
#else
	dev->trans_start = jiffies;
	private->next_txidx = ((private->next_txidx + 1) % num_txdesc);
#endif

	/* Start transmission (should be in suspend mode already) */
	em86xx_write_reg(EM86XX_TPDR_REG, 0x0);

	/* Update stats */
	stats->tx_packets++;
	stats->tx_bytes += length;

	/* Free up socket buffer */
#ifdef BOOTLOADER
	async = 0;
	skb_free(skb);
#else
	dev_kfree_skb(skb);
#endif
	return 0;
}


static inline int is_sw_filtered_packet(struct net_device *dev, unsigned char *data)
{
#ifdef USE_SW_FILTERING
        if (((data[0] & 0x80) == 0) && (dev->dev_addr[5] != data[5]))
                return 1;
        else
                return 0;
#else
        return 0;
#endif
}


/* Receiving packet(s) */
static int em86xx_eth_rx(struct net_device *dev)
{
	EM86XX_ETH_PRIV *private 	= NULL;
	struct sk_buff *skb 		= NULL;
	struct net_device_stats *stats 	= NULL;
	unsigned char *data 		= NULL;
	volatile struct em86xx_desc *desc_ptr = NULL;
	unsigned long length 	= 0;
	unsigned long dsc	= 0;
	unsigned long count 	= 0, cnt = 0;

	if (dev == NULL)
  		return(-EIO);
	else {
  		private = (EM86XX_ETH_PRIV *)dev->priv;
	  	stats = &(private->stats);
	}


	/* Go thru descriptors list until we ran out or encounterred the one
	   being processed by DMA */
#ifdef BOOTLOADER 
	for (cnt = 0, count = private->last_rxidx; cnt < num_rxdesc; cnt++, count = imodulus(count + 1, num_rxdesc)) {
#else
	for (cnt = 0, count = private->last_rxidx; cnt < num_rxdesc; cnt++, count = ((count + 1) % num_rxdesc)) {
#endif
		desc_ptr = (volatile struct em86xx_desc *)(&rxdsc[count]);
	 	dsc = desc_ptr->desc0;

	  	if (dsc & DescOwnByDma) {
    			/* DMA is processing this one, break out the loop */
			break;
    		}
 
		/*filter out those are not for me */
      		data = (unsigned char *)(desc_ptr->desc2);
		if (is_sw_filtered_packet(dev, data)) {
			ERR_PRINT("%s: mac mismatched, dropped.\n", dev->name);
		} else	if ( dsc & RX_ERROR_FLAGS ) {
      			ERR_PRINT("%s: rx error 0x%08lx:%ld desc1=0x%08x\n", 
				dev->name, dsc, count, (u32)desc_ptr->desc1);

			/* We dropped any error packet */
		        stats->rx_errors++;
		 } else if ((dsc & MP_FRAME_FLAGS) != MP_FRAME_FLAGS) {
	        	 ERR_PRINT("%s: multi-packet frame detected 0x%08lx\n", dev->name, dsc);
		        /* We don't handle multi-packet frame, for now */
		        stats->rx_dropped++;
		 } else if ((length = ((dsc & DescFrameLengthMask) >> DescFrameLengthShift)) > R_BUF_SIZE) {
      			ERR_PRINT("%s: rx dropped (size too large: %ld)\n", dev->name, length);
			stats->rx_dropped++;
#ifdef BOOTLOADER
                 } else if ((skb = skb_alloc(length)) == NULL) {
#else

		 } else if ((skb = dev_alloc_skb(length + 2)) == NULL) {
#endif
			 ERR_PRINT("%s: rx dropped (memory unavailable)\n", dev->name);
	        	 stats->rx_dropped++;
		 } else {

		  	 skb->dev = dev;
#ifdef BOOTLOADER
                         skb->len = length - 4;
                         memcpy(skb->data, data, length-4);
                         skb_put(skb);

                         /* Update stats */
                         stats->rx_packets++;
                         stats->rx_bytes += length;
#else

		         skb_reserve(skb, 2);
		         memcpy(skb_put(skb, length), data, length-4); 
		         skb->protocol = eth_type_trans(skb, dev);
     			 skb->ip_summed = CHECKSUM_NONE;

		         /* Send the packet to kernel */
		         netif_rx(skb); 
 
	        	 /* Update stats */
		       	 stats->rx_packets++;
		         stats->rx_bytes += length;
		       	 dev->last_rx = jiffies;
#endif /*BOOTLOADER*/
  	 	}
  		desc_ptr->desc0 = DescOwnByDma;
  	}

	private->last_rxidx = count; /* This is the last desc we read */

	/* make sure rx is not suspended */
	em86xx_write_reg(EM86XX_RPDR_REG, 0x0);
  	return 0;
}

/* Resetting rx descriptor: error recovery for "rx buf unavailable" */
static void em86xx_eth_reset_rx_buf(struct net_device *dev)
{
	register int i;
	volatile struct em86xx_desc *desc_ptr = NULL;

	/* Reset rx desc */
	desc_ptr = (struct em86xx_desc *)(rxdsc); 
	for (i = 0; i < num_rxdesc; i++, desc_ptr++) 
		desc_ptr->desc0 = DescOwnByDma; 
  	
  	em86xx_write_reg(EM86XX_RPDR_REG,  0x0);
	em86xx_set_reg(EM86XX_CR_REG, DmaRxStart); 
}

#ifndef BOOTLOADER
/* Get the stats information */
static struct net_device_stats *em86xx_eth_stats(struct net_device *dev)
{
	if (dev != NULL) 
    		return(&(((EM86XX_ETH_PRIV *)dev->priv)->stats));
  	else
    		return(NULL);
}
#endif

/* Ethernet interrupt handler */
#ifdef BOOTLOADER
static void em86xx_eth_intr_handler(int irq, void *dev_id)
#else
static void em86xx_eth_intr_handler(int irq, void *dev_id, struct pt_regs *regs)
#endif
{
       unsigned long status;
       struct net_device *dev = (struct net_device *)dev_id;

#ifdef BOOTLOADER
        if( memcmp(dev->name,DRIVER, sizeof(DRIVER)) != 0) 
                return;
#else
        if (dev_id != (void *)&em86xx_eth_dev)
                return;
#endif
        /* Check status */
        status = em86xx_read_reg(EM86XX_SR_REG);

        //clear all interrupt requests
        em86xx_write_reg(EM86XX_SR_REG, status) ;

        if (status & DmaIntNormal) { 
                if (status & DmaIntRxCompleted)  
                        em86xx_eth_rx((struct net_device *)dev_id);
                else
                        ERR_PRINT("%s: unhandled NIS 0x%08lx\n", dev->name, status);
        } 
 
	if (status & DmaIntAbnormal) { 
                if (status & DmaIntRxNoBuffer) { 
                        ERR_PRINT("%s: receive buffer unavailable 0x%08lx\n", dev->name, status);
                        em86xx_eth_reset_rx_buf((struct net_device *)dev_id);
                } else if (status & DmaIntRxStopped) {
                        ERR_PRINT("%s: receive process stopped\n", dev->name);
                        em86xx_set_reg(EM86XX_CR_REG, DmaRxStart); 
                } else if (status & DmaIntTxStopped ) {
                        ERR_PRINT("%s: transmit process stopped\n", dev->name);
                        em86xx_write_reg(EM86XX_CR_REG, DmaRxStart); 
                } else if (status & DmaIntTxUnderflow) {
                        ERR_PRINT("%s: transmit underflow\n", dev->name);
                } else if (status & DmaIntEarlyTx ) {
                        ERR_PRINT("%s: Early transmit interrupt\n", dev->name);
                } else if (status & DmaIntBusError ) {
                        ERR_PRINT("%s: Fatal bus error\n", dev->name);
                } else {
                        ERR_PRINT("%s: unhandled AIS 0x%08lx\n", dev->name, status);
                }

        }
 
	if (((status & DmaIntAbnormal) == 0) && ((status & DmaIntNormal) == 0)) {
                ERR_PRINT("%s: Unknown SR 0x%08lx\n", dev->name, status);
        }

        return;

}

#ifndef BOOTLOADER
/* Handling ioctl call */
static int em86xx_eth_ioctl(struct net_device *dev, struct ifreq *ifr, int cmd)
{
	/* TODO: Not implemented yet */
	struct mii_ioctl_data *data = (struct mii_ioctl_data *)&ifr->ifr_data;

    	switch(cmd) {
	case SIOCGMIIREG:		/* Read MII PHY register. */
		data->val_out = em86xx_mii_read(data->phy_id & 0x1f, data->reg_num & 0x1f);
		return 0;
	case SIOCSMIIREG:		/* Write MII PHY register. */
		em86xx_mii_write(data->phy_id, data->reg_num, data->val_in);
		return 0;
	default:
		return -EOPNOTSUPP;
	}
  	return(-EIO);
}
#endif

/* Kernel level ethernet initialization */
static int em86xx_eth_init(struct net_device *dev)
{
	DBG_PRINT("em86xx_eth_init\n");

	if (dev == NULL)
  		goto failed;

	rxdsc = (volatile struct em86xx_desc *)NON_CACHED((u32)eth_rxdsc);
	txdsc = (volatile struct em86xx_desc *)NON_CACHED((u32)eth_txdsc);
	rxbuf = (volatile unsigned char *)NON_CACHED((u32)eth_rxbuf);
	txbuf = (volatile unsigned char *)NON_CACHED((u32)eth_txbuf);

	/* Ethernet device initialization */
#ifdef BOOTLOADER
        /* Fill in the fields of the device structure with ethernet-generic values.*/
//      dev->type               = ARPHRD_ETHER;
        dev->mtu                = ETH_DATA_LEN; /* eth_mtu */
        dev->addr_len           = ETH_ALEN;
        memset(dev->broadcast, 0xff, ETH_ALEN);
        dev->hard_header_len = ETH_HLEN;
#else
	ether_setup(dev);
#endif

	if (em86xx_set_mac(dev))
  		goto failed;

	/* Hook up with handlers */
#ifdef BOOTLOADER
        dev->send_packet         = em86xx_eth_tx;
        dev->open                = em86xx_eth_open;
        dev->close               = em86xx_eth_close;
#else
	dev->get_stats		 = em86xx_eth_stats;
	dev->hard_start_xmit	 = em86xx_eth_tx;
	dev->open		 = em86xx_eth_open;
	dev->stop		 = em86xx_eth_close;
	dev->set_multicast_list  = em86xx_eth_set_multicast_list;
	dev->do_ioctl		 = em86xx_eth_ioctl;
	dev->set_mac_address	 = em86xx_eth_set_macaddr;

	dev->tx_queue_len = num_txdesc; 
	dev->flags &= ~IFF_MULTICAST;
	dev->flags |= IFF_DEBUG;
#endif
	return 0;

failed:
	return(-EIO);
}

static int em86xx_eth_reset_desc(struct net_device *dev, int *reset)
{
	EM86XX_ETH_PRIV *private;

	if (dev == NULL)
  		return(-EIO);
	else
		private = (EM86XX_ETH_PRIV *)dev->priv;

	if (*reset)
  		memset(dev->priv, 0, sizeof(EM86XX_ETH_PRIV));

	*reset = 0;

	/* Clear all tx/rx buffer memory */
	memset((void *)(rxbuf), 0, num_rxdesc * R_BUF_SIZE);
	memset((void *)(txbuf), 0, num_txdesc * T_BUF_SIZE);

	/* Initialize the indices */
	private->last_rxidx = 0;
	private->next_txidx = 0;

	/* Intialize the descriptors */
	em86xx_eth_setup_desc();

	return 0;
}

/* Driver installation: this is where the thing starts */
#ifdef BOOTLOADER
int em86xx_eth_probe(struct net_device *dev)
#else
static int __init em86xx_eth_startup(void)
#endif
{
#ifndef BOOTLOADER
	struct net_device *dev 	= &em86xx_eth_dev;
#endif
	int err = 0, i;
	EM86XX_ETH_PRIV *private= NULL;
#ifndef STATIC_BUF_ALLOC
	unsigned long max_num_rxdsc, max_num_txdsc;
	unsigned long rxbuf_size, txbuf_size;
#endif

	if (dev_count != 0) {
		err = -EIO;
  		goto failed;
	} 

#ifndef STATIC_BUF_ALLOC	
	/* Validating module parameters */
	max_num_rxdsc = max_num_txdsc = PAGE_SIZE / (2 * sizeof(struct em86xx_desc));
#ifdef BOOTLOADER
	if (((num_rxdesc < MIN_NUM_RDESC) || (num_rxdesc > max_num_rxdsc)) ||
	    ((num_txdesc < MIN_NUM_TDESC) || (num_txdesc > max_num_txdsc))) {
		err = -EIO;
		goto failed;
	} else if ((desc_page = NON_CACHED(malloc(PAGE_SIZE))) == 0) {
		err = -ENOMEM;
		goto failed;
	}
#else
	if (((num_rxdesc < MIN_NUM_RDESC) || (num_rxdesc > max_num_rxdsc)) ||
	    ((num_txdesc < MIN_NUM_TDESC) || (num_txdesc > max_num_txdsc))) {
		err = -EIO;
		goto failed;
	} else if ((desc_page = __get_free_page(GFP_KERNEL | GFP_DMA)) == 0) {
		err = -ENOMEM;
		goto failed;
	}
#endif

	/* Split a page for both rx/tx descriptor */
	eth_rxdsc = (struct em86xx_desc *)desc_page;
	eth_txdsc = (struct em86xx_desc *)(desc_page + ((1 << PAGE_SHIFT) / 2));

	/* Calculate the size needed for tx/rx -- aligned by pages */
	rxbuf_size = num_rxdesc * R_BUF_SIZE;
	txbuf_size = num_txdesc * T_BUF_SIZE;
	if ((rxbuf_size & (PAGE_SIZE - 1)) != 0)
		rxbuf_size = (rxbuf_size & PAGE_MASK) + PAGE_SIZE;
	if ((txbuf_size & (PAGE_SIZE - 1)) != 0)
		txbuf_size = (txbuf_size & PAGE_MASK) + PAGE_SIZE;
	DBG_PRINT("txbuf_size = 0x%08lx, rxbuf_size = 0x%08lx\n", txbuf_size, rxbuf_size);

#ifndef BOOTLOADER
	/* Calculate the order needed for tx/rx */
	for (rxbuf_order = 0; rxbuf_order < MAX_ORDER; rxbuf_order++) {
		if ((PAGE_SIZE * (1 << rxbuf_order)) >= rxbuf_size)
			break;
	}

	for (txbuf_order = 0; txbuf_order < MAX_ORDER; txbuf_order++) {
		if ((PAGE_SIZE * (1 << txbuf_order)) >= txbuf_size)
			break;
	}

	if ((rxbuf_order >= MAX_ORDER) || (txbuf_order >= MAX_ORDER)) {
		err = -ENOMEM;
		goto failed;
	}

	DBG_PRINT("txbuf_order = 0x%08lx, rxbuf_order = 0x%08lx\n", txbuf_order, rxbuf_order);
#endif
		
#ifdef BOOTLOADER
	if ((rxbuf_pages = NON_CACHED(malloc(rxbuf_size))) == 0) {
		err = -ENOMEM;
		goto failed;
	} else if ((txbuf_pages = NON_CACHED(malloc(txbuf_size))) == 0) {
		err = -ENOMEM;
		goto failed;
	}
#else
	if ((rxbuf_pages = __get_free_pages(GFP_KERNEL | GFP_DMA, rxbuf_order)) == 0) {
		err = -ENOMEM;
		goto failed;
	} else if ((txbuf_pages = __get_free_pages(GFP_KERNEL | GFP_DMA, txbuf_order)) == 0) {
		err = -ENOMEM;
		goto failed;
	}
#endif

	eth_rxbuf  = (unsigned char *)rxbuf_pages;
	eth_txbuf  = (unsigned char *)txbuf_pages;
#endif

#ifndef BOOTLOADER
	SET_MODULE_OWNER(&em86xx_eth_dev);
#endif

	MSG_PRINT("Ethernet driver for EM86XX (v1.0)");
	dev->priv = NULL;

	/* Set up default mac address and others */
	max_link_loop 	= DEF_LINK_LOOP;
	ld_count 	= 1;
	reset_flag 	= 1;

	/* Get a device name: normally it'll be eth0 */
#ifdef BOOTLOADER
        memcpy( dev->name, DRIVER, sizeof(DRIVER));

        /* Register ISR */
        dev->irq = IRQ_ETHERNET;
        em86xx_request_irq(IRQ_ETHERNET, em86xx_eth_intr_handler, dev);
#else
	if ((err = dev_alloc_name(&em86xx_eth_dev, "eth%d")) < 0)
  		goto failed;

	/* Register ISR */
	if ((err = request_irq(IRQ_ETHERNET, em86xx_eth_intr_handler, 0, "ethernet", 
      	 	 &em86xx_eth_dev)) != 0) {
  		err = -EIO;
  		goto failed;
	} else
#endif 
  		MSG_PRINT(" on %s (IRQ: %d)", dev->name, IRQ_ETHERNET);


	if (em86xx_get_macaddr(dev->dev_addr)) {
  		err = -EIO;
  		goto failed;
	}

	MSG_PRINT("\n(MAC %02x", dev->dev_addr[0]);
	for (i = 1; i < 6; i++)
	  	MSG_PRINT(":%02x", dev->dev_addr[i]);
	MSG_PRINT(", tx_desc/rx_desc = %ld/%ld)\n", num_txdesc, num_rxdesc);

	/* Allocate memory for private data */
#ifdef BOOTLOADER
        dev->priv = private = (EM86XX_ETH_PRIV *)malloc(sizeof(EM86XX_ETH_PRIV));//, GFP_KERNEL);
#else
	dev->priv = private = (EM86XX_ETH_PRIV *)kmalloc(sizeof(EM86XX_ETH_PRIV), GFP_KERNEL);
#endif
	if (dev->priv == NULL) {
		err = -ENOMEM;
		goto failed;
	} 

	/* Point to driver initialization routine and register the device with kernel */
	dev->init = em86xx_eth_init;
#ifdef BOOTLOADER
        em86xx_eth_init(dev);
        dev->state = NETDEV_DOWN;
#else
	if (register_netdev(dev) != 0) {
  		err = -EIO;
  		goto failed;
	}
#endif

	MSG_PRINT("%s: driver installation completed.\n", dev->name);
	dev_count++;
	return 0;

failed:
  	if (dev->priv != NULL) {
#ifdef BOOTLOADER
                free(dev->priv);
#else
    		kfree(dev->priv);
#endif
    		dev->priv = NULL;
  	}
#ifndef STATIC_BUF_ALLOC
#ifdef BOOTLOADER
	if (desc_page   != 0)
		free((void *)CACHED(desc_page));
	if (rxbuf_pages != 0)
		free((void *)CACHED(rxbuf_pages));
	if (txbuf_pages != 0)
		free((void *)CACHED(txbuf_pages));
#else
	if (desc_page   != 0)
		free_page(desc_page);
	if (rxbuf_pages != 0)
		free_pages(rxbuf_pages, rxbuf_order);
	if (txbuf_pages != 0)
		free_pages(txbuf_pages, txbuf_order);
#endif
#endif
  	MSG_PRINT("%s: driver installation failed.\n", dev->name);
  	return(err);
}

#ifndef BOOTLOADER
/* Uninstallation of drive */
static void __exit em86xx_eth_shutdown(void)
{
	if (dev_count != 0) {
		/* Turn off IRQ and stop receive/transmit */
  		em86xx_write_reg(EM86XX_CR_REG, 0);
  		em86xx_write_reg(EM86XX_IER_REG, 0);

  		/* Unregister the device and ISR */
  		free_irq(IRQ_ETHERNET, &em86xx_eth_dev);
		unregister_netdev(&em86xx_eth_dev);

		/* Set desc base address registers to 0 */
		em86xx_write_reg(EM86XX_RLBAR_REG, 0);
		em86xx_write_reg(EM86XX_TLBAR_REG, 0);
#ifndef STATIC_BUF_ALLOC
#ifdef BOOTLOADER
		if (desc_page   != 0)
			free((void *)CACHED(desc_page));
		if (rxbuf_pages != 0)
			free((void *)CACHED(rxbuf_pages));
		if (txbuf_pages != 0)
			free((void *)CACHED(txbuf_pages));
#else
		if (desc_page   != 0)
			free_page(desc_page);
		if (rxbuf_pages != 0)
			free_pages(rxbuf_pages, rxbuf_order);
		if (txbuf_pages != 0)
			free_pages(txbuf_pages, txbuf_order);
#endif
#endif
		/* Free up memory */
		if (em86xx_eth_dev.priv != NULL) {
			kfree(em86xx_eth_dev.priv);
			em86xx_eth_dev.priv = NULL;
  		}
	} 
}

/* Register startup/shutdown routines */
module_init(em86xx_eth_startup);
module_exit(em86xx_eth_shutdown);
#endif

