/*-
 * Copyright (c) 2001 Atsushi Onoe
 * Copyright (c) 2002-2004 Sam Leffler, Errno Consulting
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * Alternatively, this software may be distributed under the terms of the
 * GNU General Public License ("GPL") version 2 as published by the Free
 * Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
#ifndef EXPORT_SYMTAB
#define	EXPORT_SYMTAB
#endif

__FBSDID("$FreeBSD: src/sys/net80211/ieee80211_node.c,v 1.13 2003/11/09 23:36:46 sam Exp $");
__KERNEL_RCSID(0, "$NetBSD: ieee80211_node.c,v 1.8 2003/11/02 01:29:05 dyoung Exp $");

/*
 * IEEE 802.11 node handling support.
 */
#include <linux/config.h>
#include <linux/version.h>
#include <linux/module.h>
#include <linux/skbuff.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/random.h>

#include "if_media.h"

#include <net80211/ieee80211_var.h>
#include <net80211/if_athproto.h>

static struct ieee80211_node *ieee80211_node_alloc(struct ieee80211com *);
static void ieee80211_node_free(struct ieee80211com *, struct ieee80211_node *);
static void ieee80211_node_copy(struct ieee80211com *,
		struct ieee80211_node *, const struct ieee80211_node *);
static u_int8_t ieee80211_node_getrssi(struct ieee80211com *,
		struct ieee80211_node *);

static void ieee80211_setup_node(struct ieee80211com *ic,
		struct ieee80211_node *ni, const u_int8_t *macaddr);
static void _ieee80211_free_node(struct ieee80211com *,
		struct ieee80211_node *);

MALLOC_DEFINE(M_80211_NODE, "node", "802.11 node state");

void
ieee80211_node_attach(struct ieee80211com *ic)
{

	/* XXX need unit */
	IEEE80211_NODE_LOCK_INIT(ic, ic->ic_ifp->if_xname);
	TAILQ_INIT(&ic->ic_node);
	ic->ic_node_alloc = ieee80211_node_alloc;
	ic->ic_node_free = ieee80211_node_free;
	ic->ic_node_copy = ieee80211_node_copy;
	ic->ic_node_getrssi = ieee80211_node_getrssi;
	ic->ic_scangen = 1;
	/* default inactivity timer setings */
	ic->ic_inact_init = IEEE80211_INACT_INIT;
	ic->ic_inact_auth = IEEE80211_INACT_AUTH;
	ic->ic_inact_run = IEEE80211_INACT_RUN;

	if (ic->ic_max_aid == 0)
		ic->ic_max_aid = IEEE80211_AID_DEF;
	else if (ic->ic_max_aid > IEEE80211_AID_MAX)
		ic->ic_max_aid = IEEE80211_AID_MAX;
	MALLOC(ic->ic_aid_bitmap, u_int32_t *,
		howmany(ic->ic_max_aid, 32) * sizeof(u_int32_t),
		M_DEVBUF, M_NOWAIT | M_ZERO);
	if (ic->ic_aid_bitmap == NULL) {
		/* XXX no way to recover */
		printf("%s: no memory for AID bitmap!\n", __func__);
		ic->ic_max_aid = 0;
	}
}

void
ieee80211_node_lateattach(struct ieee80211com *ic)
{
	struct ieee80211_node *ni;
	struct ieee80211_rsnparms *rsn;

	ni = (*ic->ic_node_alloc)(ic);
	KASSERT(ni != NULL, ("unable to setup inital BSS node"));
	/*
	 * Setup "global settings" in the bss node so that
	 * each new station automatically inherits them.
	 */
	rsn = &ni->ni_rsn;
	/* WEP, TKIP, and AES-CCM are always supported */
	rsn->rsn_ucastcipherset |= 1<<IEEE80211_CIPHER_WEP;
	rsn->rsn_ucastcipherset |= 1<<IEEE80211_CIPHER_TKIP;
	rsn->rsn_ucastcipherset |= 1<<IEEE80211_CIPHER_AES_CCM;
	if (ic->ic_caps & IEEE80211_C_AES)
		rsn->rsn_ucastcipherset |= 1<<IEEE80211_CIPHER_AES_OCB;
	if (ic->ic_caps & IEEE80211_C_CKIP)
		rsn->rsn_ucastcipherset |= 1<<IEEE80211_CIPHER_CKIP;
	/*
	 * Default unicast cipher to WEP for 802.1x use.  If
	 * WPA is enabled the management code will set these
	 * values to reflect.
	 */
	rsn->rsn_ucastcipher = IEEE80211_CIPHER_WEP;
	rsn->rsn_ucastkeylen = 104 / NBBY;
	/*
	 * WPA says the multicast cipher is the lowest unicast
	 * cipher supported.  But we skip WEP which would
	 * otherwise be used based on this criteria.
	 */
	rsn->rsn_mcastcipher = IEEE80211_CIPHER_TKIP;
	rsn->rsn_mcastkeylen = 128 / NBBY;

	/*
	 * We support both WPA-PSK and 802.1x; the one used
	 * is determined by the authentication mode and the
	 * setting of the PSK state.
	 */
	rsn->rsn_keymgmtset = WPA_ASE_8021X_UNSPEC | WPA_ASE_8021X_PSK;
	rsn->rsn_keymgmt = WPA_ASE_8021X_PSK;

	ni->ni_chan = IEEE80211_CHAN_ANYC;
	ni->ni_authmode = IEEE80211_AUTH_OPEN;
	ni->ni_txpower = IEEE80211_TXPOWER_MAX;
	ieee80211_crypto_resetkey(ic, &ni->ni_ucastkey, IEEE80211_KEYIX_NONE);
	ic->ic_bss = ni;
	ic->ic_auth = ieee80211_authenticator_get(ni->ni_authmode);
}

void
ieee80211_node_detach(struct ieee80211com *ic)
{

	if (ic->ic_bss != NULL) {
		(*ic->ic_node_free)(ic, ic->ic_bss);
		ic->ic_bss = NULL;
	}
	ieee80211_free_allnodes(ic);
	IEEE80211_NODE_LOCK_DESTROY(ic);
	if (ic->ic_aid_bitmap != NULL)
		FREE(ic->ic_aid_bitmap, M_DEVBUF);
}

/* 
 * Port authorize/unauthorize interfaces for use by an authenticator.
 */

void
ieee80211_node_authorize(struct ieee80211com *ic, struct ieee80211_node *ni)
{
	ni->ni_flags |= IEEE80211_NODE_AUTH;
}
EXPORT_SYMBOL(ieee80211_node_authorize);

void
ieee80211_node_unauthorize(struct ieee80211com *ic, struct ieee80211_node *ni)
{
	ni->ni_flags &= ~IEEE80211_NODE_AUTH;
}
EXPORT_SYMBOL(ieee80211_node_unauthorize);

/*
 * Set/change the channel.  The rate set is also updated as
 * to insure a consistent view by drivers.
 */
static inline void
ieee80211_set_chan(struct ieee80211com *ic,
	struct ieee80211_node *ni, struct ieee80211_channel *chan)
{
	ni->ni_chan = chan;
	ni->ni_rates = ic->ic_sup_rates[ieee80211_chan2mode(ic, chan)];
}

/*
 * AP scanning support.
 */

#ifdef IEEE80211_DEBUG
static void
dump_chanlist(const u_char chans[])
{
	const char *sep;
	int i;

	sep = " ";
	for (i = 0; i < IEEE80211_CHAN_MAX; i++)
		if (isset(chans, i)) {
			printf("%s%u", sep, i);
			sep = ", ";
		}
}
#endif /* IEEE80211_DEBUG */

/*
 * Initialize the active channel set based on the set
 * of available channels and the current PHY mode.
 */
static void
ieee80211_reset_scan(struct ieee80211com *ic)
{

	/* XXX ic_des_chan should be handled with ic_chan_active */
	if (ic->ic_des_chan != IEEE80211_CHAN_ANYC) {
		memset(ic->ic_chan_scan, 0, sizeof(ic->ic_chan_scan));
		setbit(ic->ic_chan_scan,
			ieee80211_chan2ieee(ic, ic->ic_des_chan));
	} else
		memcpy(ic->ic_chan_scan, ic->ic_chan_active,
			sizeof(ic->ic_chan_active));
	/* NB: hack, setup so next_scan starts with the first channel */
	if (ic->ic_bss->ni_chan == IEEE80211_CHAN_ANYC)
		ieee80211_set_chan(ic, ic->ic_bss,
			&ic->ic_channels[IEEE80211_CHAN_MAX]);
#ifdef IEEE80211_DEBUG
	if (ieee80211_msg_scan(ic)) {
		printf("%s: scan set:", __func__);
		dump_chanlist(ic->ic_chan_scan);
		printf(" start chan %u\n",
			ieee80211_chan2ieee(ic, ic->ic_bss->ni_chan));
	}
#endif /* IEEE80211_DEBUG */
}

/*
 * Begin an active scan.
 */
void
ieee80211_begin_scan(struct ieee80211com *ic, int reset)
{

#if defined(ATH_SUPERG_SCAN) 

	/* 
	 * Non-standard channels are NOT scanned in IEEE80211_MODE_AUTO
	 * by default.
	 *
	 * To automatically scan turbo channels, we rotate through 
	 * the various supported turbo modes during a scan. 
	 */
	   
	if (ic->ic_desmode == IEEE80211_MODE_AUTO) {
		u_int16_t nm; 

		for (nm = ic->ic_curmode + 1; nm != ic->ic_curmode;) { 

			if ((ic->ic_modecaps & (1<<nm)) &&
			    ((nm == IEEE80211_MODE_TURBO_A) ||
			     (nm == IEEE80211_MODE_TURBO_G) ||
			     (nm == IEEE80211_MODE_AUTO))) {
				IEEE80211_DPRINTF(ic, IEEE80211_MSG_SCAN,
						  ("scan: switching to mode "
						   "%d\n", nm));
				(void) ieee80211_setmode(ic, nm);
				break;
			}

			if (nm >= IEEE80211_MODE_MAX)
				nm = 0;
			else
				nm++;
		}
	}
#endif


	/*
	 * In all but hostap mode scanning starts off in
	 * an active mode before switching to passive.
	 */
	if (ic->ic_opmode != IEEE80211_M_HOSTAP) {
		ic->ic_flags |= IEEE80211_F_ASCAN;
		ic->ic_stats.is_scan_active++;
	} else
		ic->ic_stats.is_scan_passive++;
	IEEE80211_DPRINTF(ic, IEEE80211_MSG_SCAN, ("begin %s scan\n",
		(ic->ic_flags & IEEE80211_F_ASCAN) ?  "active" : "passive"));
	/*
	 * Clear scan state and flush any previously seen
	 * AP's.  Note that the latter assumes we don't act
	 * as both an AP and a station, otherwise we'll
	 * potentially flush state of stations associated
	 * with us.
	 */
	ieee80211_reset_scan(ic);
	if (reset)
		ieee80211_free_allnodes(ic);

	/* Scan the next channel. */
	ieee80211_next_scan(ic);
}

/*
 * Switch to the next channel marked for scanning.
 */
int
ieee80211_next_scan(struct ieee80211com *ic)
{
	struct ieee80211_channel *chan;

	/*
	 * Insure any previous mgt frame timeouts don't fire.
	 * This assumes the driver does the right thing in
	 * flushing anything queued in the driver and below.
	 */
	ic->ic_mgt_timer = 0;

	chan = ic->ic_bss->ni_chan;
	do {
		if (++chan > &ic->ic_channels[IEEE80211_CHAN_MAX])
			chan = &ic->ic_channels[0];
		if (isset(ic->ic_chan_scan, ieee80211_chan2ieee(ic, chan))) {
			clrbit(ic->ic_chan_scan, ieee80211_chan2ieee(ic, chan));
			IEEE80211_DPRINTF(ic, IEEE80211_MSG_SCAN,
			    ("%s: chan %d->%d\n", __func__,
			    ieee80211_chan2ieee(ic, ic->ic_bss->ni_chan),
			    ieee80211_chan2ieee(ic, chan)));
			ieee80211_set_chan(ic, ic->ic_bss, chan);
			ieee80211_new_state(ic, IEEE80211_S_SCAN, -1);
			return 1;
		}
	} while (chan != ic->ic_bss->ni_chan);
	ieee80211_end_scan(ic);
	return 0;
}
EXPORT_SYMBOL(ieee80211_next_scan);

void
ieee80211_create_ibss(struct ieee80211com* ic, struct ieee80211_channel *chan)
{
	struct ieee80211_node *ni;

	ni = ic->ic_bss;
	IEEE80211_DPRINTF(ic, IEEE80211_MSG_SCAN, ("creating ibss\n"));
	ic->ic_flags |= IEEE80211_F_SIBSS;
	ieee80211_set_chan(ic, ic->ic_bss, chan);
	IEEE80211_ADDR_COPY(ni->ni_macaddr, ic->ic_myaddr);
	IEEE80211_ADDR_COPY(ni->ni_bssid, ic->ic_myaddr);
	if (ic->ic_opmode == IEEE80211_M_IBSS)
		ni->ni_bssid[0] |= 0x02;	/* local bit for IBSS */
	ni->ni_esslen = ic->ic_des_esslen;
	memcpy(ni->ni_essid, ic->ic_des_essid, ni->ni_esslen);
	ni->ni_rssi = 0;
	ni->ni_rstamp = 0;
	ni->ni_tstamp.tsf = 0;
	ni->ni_intval = ic->ic_lintval;
	ni->ni_capinfo = IEEE80211_CAPINFO_IBSS;
	if (ic->ic_flags & IEEE80211_F_PRIVACY)
		ni->ni_capinfo |= IEEE80211_CAPINFO_PRIVACY;
	if (ic->ic_phytype == IEEE80211_T_FH) {
		ni->ni_fhdwell = 200;	/* XXX */
		ni->ni_fhindex = 1;
	}
	ni->ni_erp = 0;
	ieee80211_new_state(ic, IEEE80211_S_RUN, -1);
}

static int
ieee80211_match_bss(struct ieee80211com *ic, struct ieee80211_node *ni)
{
        u_int8_t rate;
        int fail;

	fail = 0;
	if (isclr(ic->ic_chan_active, ieee80211_chan2ieee(ic, ni->ni_chan)))
		fail |= 0x01;
	if (ic->ic_des_chan != IEEE80211_CHAN_ANYC &&
	    ni->ni_chan != ic->ic_des_chan)
		fail |= 0x01;
	if (ic->ic_opmode == IEEE80211_M_IBSS) {
		if ((ni->ni_capinfo & IEEE80211_CAPINFO_IBSS) == 0)
			fail |= 0x02;
	} else {
		if ((ni->ni_capinfo & IEEE80211_CAPINFO_ESS) == 0)
			fail |= 0x02;
	}
	if (ic->ic_flags & IEEE80211_F_PRIVACY) {
		if ((ni->ni_capinfo & IEEE80211_CAPINFO_PRIVACY) == 0)
			fail |= 0x04;
	} else {
		/* XXX does this mean privacy is supported or required? */
		if (ni->ni_capinfo & IEEE80211_CAPINFO_PRIVACY)
			fail |= 0x04;
	}
	rate = ieee80211_fix_rate(ic, ni, IEEE80211_F_DONEGO);
	if (rate & IEEE80211_RATE_BASIC)
		fail |= 0x08;
	if (ic->ic_des_esslen != 0 &&
	    (ni->ni_esslen != ic->ic_des_esslen ||
	     memcmp(ni->ni_essid, ic->ic_des_essid, ic->ic_des_esslen) != 0))
		fail |= 0x10;
	if ((ic->ic_flags & IEEE80211_F_DESBSSID) &&
	    !IEEE80211_ADDR_EQ(ic->ic_des_bssid, ni->ni_bssid))
		fail |= 0x20;
#ifdef IEEE80211_DEBUG
	if (ieee80211_msg_scan(ic)) {
		printf(" %c %s", fail ? '-' : '+',
		    ether_sprintf(ni->ni_macaddr));
		printf(" %s%c", ether_sprintf(ni->ni_bssid),
		    fail & 0x20 ? '!' : ' ');
		printf(" %3d%c", ieee80211_chan2ieee(ic, ni->ni_chan),
			fail & 0x01 ? '!' : ' ');
		printf(" %+4d", ni->ni_rssi);
		printf(" %2dM%c", (rate & IEEE80211_RATE_VAL) / 2,
		    fail & 0x08 ? '!' : ' ');
		printf(" %4s%c",
		    (ni->ni_capinfo & IEEE80211_CAPINFO_ESS) ? "ess" :
		    (ni->ni_capinfo & IEEE80211_CAPINFO_IBSS) ? "ibss" :
		    "????",
		    fail & 0x02 ? '!' : ' ');
		printf(" %3s%c ",
		    (ni->ni_capinfo & IEEE80211_CAPINFO_PRIVACY) ?
		    "wep" : "no",
		    fail & 0x04 ? '!' : ' ');
		ieee80211_print_essid(ni->ni_essid, ni->ni_esslen);
		printf("%s\n", fail & 0x10 ? "!" : "");
	}
#endif
	return fail;
}

/*
 * Complete a scan of potential channels.
 */
void
ieee80211_end_scan(struct ieee80211com *ic)
{
	struct ieee80211_node *ni, *nextbs, *selbs;
	int i, fail;

	IEEE80211_DPRINTF(ic, IEEE80211_MSG_SCAN, ("end %s scan\n",
		(ic->ic_flags & IEEE80211_F_ASCAN) ?  "active" : "passive"));

	ic->ic_flags &= ~IEEE80211_F_ASCAN;
	ni = TAILQ_FIRST(&ic->ic_node);

	ieee80211_notify_scan_done(ic);

	if (ic->ic_opmode == IEEE80211_M_HOSTAP) {
		/* XXX off stack? */
		u_char occupied[IEEE80211_CHAN_BYTES];
		/*
		 * The passive scan to look for existing AP's completed,
		 * select a channel to camp on.  Identify the channels
		 * that already have one or more AP's and try to locate
		 * an unnoccupied one.  If that fails, pick a random
		 * channel from the active set.
		 */
		for (; ni != NULL; ni = nextbs) {
			ieee80211_ref_node(ni);
			nextbs = TAILQ_NEXT(ni, ni_list);
			setbit(occupied, ieee80211_chan2ieee(ic, ni->ni_chan));
			ieee80211_free_node(ic, ni);
		}
		for (i = 0; i < IEEE80211_CHAN_MAX; i++)
			if (isset(ic->ic_chan_active, i) && isclr(occupied, i))
				break;
		if (i == IEEE80211_CHAN_MAX) {
			get_random_bytes(&fail, sizeof(fail));
			fail &= 3;		/* random 0-3 */
			for (i = 0; i < IEEE80211_CHAN_MAX; i++)
				if (isset(ic->ic_chan_active, i) && fail-- == 0)
					break;
		}
		ieee80211_create_ibss(ic, &ic->ic_channels[i]);
		return;
	}

	/*
	 * When manually sequencing the state machine; scan just once
	 * regardless of whether we have a candidate or not.  The
	 * controlling application is expected to setup state and
	 * initiate an association.
	 */
	if (ic->ic_roaming == IEEE80211_ROAMING_MANUAL)
		return;
	/*
	 * Automatic sequencing; look for a candidate and
	 * if found join the network.
	 */
	if (ni == NULL) {
		IEEE80211_DPRINTF(ic, IEEE80211_MSG_SCAN,
			("%s: no scan candidate\n", __func__));
  notfound:
		if (ic->ic_opmode == IEEE80211_M_IBSS &&
		    (ic->ic_flags & IEEE80211_F_IBSSON) &&
		    ic->ic_des_esslen != 0) {
			ieee80211_create_ibss(ic, ic->ic_ibss_chan);
			return;
		}
		/*
		 * Reset the list of channels to scan and start again.
		 */
                ieee80211_reset_scan(ic);
                ieee80211_next_scan(ic);

		return;
	}
	selbs = NULL;
	IEEE80211_DPRINTF(ic, IEEE80211_MSG_SCAN,
		("\tmacaddr          bssid         chan  rssi rate flag  wep  essid\n"));
	for (; ni != NULL; ni = nextbs) {
		ieee80211_ref_node(ni);
		nextbs = TAILQ_NEXT(ni, ni_list);
		if (ni->ni_fails) {
			/*
			 * The configuration of the access points may change
			 * during my scan.  So delete the entry for the AP
			 * and retry to associate if there is another beacon.
			 */
			IEEE80211_DPRINTF(ic, IEEE80211_MSG_SCAN,
				("%s: skip scan candidate %s, fails %u\n",
				__func__, ether_sprintf(ni->ni_macaddr),
				ni->ni_fails));
			if (ni->ni_fails++ > 2)
				ieee80211_free_node(ic, ni);
			continue;
		}
		if (ieee80211_match_bss(ic, ni) == 0) {
			if (selbs == NULL)
				selbs = ni;
			else if (ni->ni_rssi > selbs->ni_rssi) {
				ieee80211_unref_node(&selbs);
				selbs = ni;
			} else
				ieee80211_unref_node(&ni);
		} else {
			ieee80211_unref_node(&ni);
		}
	}
	if (selbs == NULL)
		goto notfound;
	if (!ieee80211_sta_join(ic, selbs)) {
		ieee80211_unref_node(&selbs);
		goto notfound;
	}
}

/*
 * Join the specified IBSS/BSS network.  The node is assumed to
 * be passed in with a held reference.
 */
int
ieee80211_sta_join(struct ieee80211com *ic, struct ieee80211_node *selbs)
{

	/* XXX leak existing state in ic_bss? */
	(*ic->ic_node_copy)(ic, ic->ic_bss, selbs);
	/*
	 * Set the erp state (mostly the slot time) to deal with
	 * the auto-select case; this should be redundant if the
	 * mode is locked.
	 */ 
	ic->ic_curmode = ieee80211_chan2mode(ic, ic->ic_bss->ni_chan);
	ieee80211_reset_erp(ic,ic->ic_curmode);
	if (ic->ic_opmode == IEEE80211_M_IBSS) {
		ieee80211_fix_rate(ic, ic->ic_bss, IEEE80211_F_DOFRATE |
		    IEEE80211_F_DONEGO | IEEE80211_F_DODEL);
		if (ic->ic_bss->ni_rates.rs_nrates == 0) {
			selbs->ni_fails++;
			return 0;
		}
		ieee80211_unref_node(&selbs);
		/*
		 * Discard scan set; the nodes have a refcnt of zero
		 * and have not asked the driver to setup private
		 * node state.  Let them be repopulated on demand either
		 * through transmission (ieee80211_find_txnode) or receipt
		 * of a probe response (to be added).
		 */
		ieee80211_free_allnodes(ic);
		ieee80211_new_state(ic, IEEE80211_S_RUN, -1);
	} else {
		ieee80211_unref_node(&selbs);
		ieee80211_new_state(ic, IEEE80211_S_AUTH, -1);
	}
	return 1;
}

/*
 * Leave the specified IBSS/BSS network.  The node is assumed to
 * be passed in with a held reference.
 */
void
ieee80211_sta_leave(struct ieee80211com *ic, struct ieee80211_node *ni)
{

	node_cleanup(ic, ni);
	ieee80211_notify_node_leave(ic, ni);
}

static struct ieee80211_node *
ieee80211_node_alloc(struct ieee80211com *ic)
{
	struct ieee80211_node *ni;
	MALLOC(ni, struct ieee80211_node *, sizeof(struct ieee80211_node),
		M_80211_NODE, M_NOWAIT | M_ZERO);
	return ni;
}

/*
 * Reclaim any resources in a node and reset any critical
 * state.  Typically nodes are free'd immediately after,
 * but in some cases the storage may be reused so we need
 * to insure consistent state (should probably fix that).
 */
void
node_cleanup(struct ieee80211com *ic, struct ieee80211_node *ni)
{
#define	N(a)	(sizeof(a)/sizeof(a[0]))
	int i;

	ni->ni_flags = 0;
	ni->ni_associd = 0;
	ni->ni_esslen = 0;
	/* XXX ni_savedq */
	if (ni->ni_challenge != NULL) {
		FREE(ni->ni_challenge, M_DEVBUF);
		ni->ni_challenge = NULL;
	}
	if (ni->ni_wpa_ie) {
		FREE(ni->ni_wpa_ie, M_DEVBUF);
		ni->ni_wpa_ie = NULL;
	}
	if (ni->ni_wme_ie) {
		FREE(ni->ni_wme_ie, M_DEVBUF);
		ni->ni_wme_ie = NULL;
	}
	for (i = 0; i < N(ni->ni_rxfrag); i++)
		if (ni->ni_rxfrag[i] != NULL) {
			kfree_skb(ni->ni_rxfrag[i]);
			ni->ni_rxfrag[i] = NULL;
		}
	ieee80211_crypto_delkey(ic, &ni->ni_ucastkey);
#undef N
}

static void
ieee80211_node_free(struct ieee80211com *ic, struct ieee80211_node *ni)
{
	node_cleanup(ic, ni);
	FREE(ni, M_80211_NODE);
}

static void
iedup(u_int8_t **iedst, const u_int8_t *iesrc)
{
	MALLOC(*iedst, void *, iesrc[1], M_DEVBUF, M_NOWAIT);
	if (*iedst != NULL)
		memcpy(*iedst, iesrc, iesrc[1]);
}

/*
 * Copy state from one node to another.  The semantics are
 * unclear for non-refcnt'd data; we leave them in the src
 * and clear the dst.  This is what is needed for the one
 * (current) use of this operation (setting up ic_bss after
 * an AP scan).  In practice the fields in question should
 * always be unused.
 */
static void
ieee80211_node_copy(struct ieee80211com *ic,
	struct ieee80211_node *dst, const struct ieee80211_node *src)
{
#define	N(a)	(sizeof(a)/sizeof(a[0]))
	int i;

	node_cleanup(ic, dst);
	*dst = *src;
	dst->ni_challenge = NULL;
	if (dst->ni_wpa_ie != NULL)
		iedup(&dst->ni_wpa_ie, src->ni_wpa_ie);
	if (dst->ni_wme_ie != NULL)
		iedup(&dst->ni_wme_ie, src->ni_wme_ie);
	for (i = 0; i < N(dst->ni_rxfrag); i++)
		dst->ni_rxfrag[i] = NULL;
	/* NB: unicast key index reset by ieee80211_crypto_delkey */
#undef N
}

static u_int8_t
ieee80211_node_getrssi(struct ieee80211com *ic, struct ieee80211_node *ni)
{
	return ni->ni_rssi;
}

static void
ieee80211_setup_node(struct ieee80211com *ic,
	struct ieee80211_node *ni, const u_int8_t *macaddr)
{
	int hash;

	IEEE80211_DPRINTF(ic, IEEE80211_MSG_NODE,
		("%s %s\n", __func__, ether_sprintf(macaddr)));
	IEEE80211_ADDR_COPY(ni->ni_macaddr, macaddr);
	hash = IEEE80211_NODE_HASH(macaddr);
	skb_queue_head_init(&ni->ni_savedq);
	ieee80211_node_initref(ni);		/* mark referenced */
	ieee80211_crypto_resetkey(ic, &ni->ni_ucastkey, IEEE80211_KEYIX_NONE);
	ni->ni_inact = ic->ic_inact_init;

	IEEE80211_NODE_LOCK_BH(ic);
	TAILQ_INSERT_TAIL(&ic->ic_node, ni, ni_list);
	LIST_INSERT_HEAD(&ic->ic_hash[hash], ni, ni_hash);
	/* 
	 * Note we don't enable the inactive timer when acting
	 * as a station.  Nodes created in this mode represent
	 * AP's identified while scanning.  If we time them out
	 * then several things happen: we can't return the data
	 * to users to show the list of AP's we encountered, and
	 * more importantly, we'll incorrectly deauthenticate
	 * ourself because the inactivity timer will kick us off. 
	 */
	if (ic->ic_opmode != IEEE80211_M_STA)
		ic->ic_inact_timer = IEEE80211_INACT_WAIT;
	IEEE80211_NODE_UNLOCK_BH(ic);
}

struct ieee80211_node *
ieee80211_alloc_node(struct ieee80211com *ic, const u_int8_t *macaddr)
{
	struct ieee80211_node *ni = (*ic->ic_node_alloc)(ic);
	if (ni != NULL)
		ieee80211_setup_node(ic, ni, macaddr);
	else
		ic->ic_stats.is_rx_nodealloc++;
	return ni;
}

struct ieee80211_node *
ieee80211_dup_bss(struct ieee80211com *ic, const u_int8_t *macaddr)
{
	struct ieee80211_node *ni = (*ic->ic_node_alloc)(ic);
	if (ni != NULL) {
		ieee80211_setup_node(ic, ni, macaddr);
		/*
		 * Inherit from ic_bss.
		 */
		ni->ni_authmode = ic->ic_bss->ni_authmode;
		ni->ni_txpower = ic->ic_bss->ni_txpower;
		ni->ni_vlan = ic->ic_bss->ni_vlan;	/* XXX?? */
		IEEE80211_ADDR_COPY(ni->ni_bssid, ic->ic_bss->ni_bssid);
		ieee80211_set_chan(ic, ni, ic->ic_bss->ni_chan);
		ni->ni_rsn = ic->ic_bss->ni_rsn;
	} else
		ic->ic_stats.is_rx_nodealloc++;
	return ni;
}

struct ieee80211_node *
#ifdef IEEE80211_DEBUG_REFCNT
_ieee80211_find_node_debug(struct ieee80211com *ic,
	const u_int8_t *macaddr, const char *func, int line)
#else
_ieee80211_find_node(struct ieee80211com *ic,
	const u_int8_t *macaddr)
#endif
{
	struct ieee80211_node *ni;
	int hash;

	IEEE80211_NODE_LOCK_ASSERT(ic);

	hash = IEEE80211_NODE_HASH(macaddr);
	LIST_FOREACH(ni, &ic->ic_hash[hash], ni_hash) {
		if (IEEE80211_ADDR_EQ(ni->ni_macaddr, macaddr)) {
			ieee80211_node_incref(ni); /* mark referenced */
#ifdef IEEE80211_DEBUG_REFCNT
			IEEE80211_DPRINTF(ic, IEEE80211_MSG_NODE,
			    ("%s (%s:%u) %s refcnt %d\n", __func__, func, line,
			     ether_sprintf(ni->ni_macaddr),
			     ieee80211_node_refcnt(ni)));
#endif
			return ni;
		}
	}
	return NULL;
}
#ifdef IEEE80211_DEBUG_REFCNT
#define	_ieee80211_find_node(nt, mac) \
	_ieee80211_find_node_debug(nt, mac, func, line)
#endif

struct ieee80211_node *
#ifdef IEEE80211_DEBUG_REFCNT
ieee80211_find_node_debug(struct ieee80211com *ic,
	const u_int8_t *macaddr, const char *func, int line)
#else
ieee80211_find_node(struct ieee80211com *ic, const u_int8_t *macaddr)
#endif
{
	struct ieee80211_node *ni;

	IEEE80211_NODE_LOCK(ic);
	ni = _ieee80211_find_node(ic, macaddr);
	IEEE80211_NODE_UNLOCK(ic);
	return ni;
}
#ifdef IEEE80211_DEBUG_REFCNT
EXPORT_SYMBOL(ieee80211_find_node_debug);
#else
EXPORT_SYMBOL(ieee80211_find_node);
#endif

/*
 * Return a reference to the appropriate node for sending
 * a data frame.  This handles node discovery in adhoc networks.
 */
struct ieee80211_node *
#ifdef IEEE80211_DEBUG_REFCNT
ieee80211_find_txnode_debug(struct ieee80211com *ic, const u_int8_t *macaddr,
	const char *func, int line)
#else
ieee80211_find_txnode(struct ieee80211com *ic, const u_int8_t *macaddr)
#endif
{
	struct ieee80211_node *ni;

	/*
	 * The destination address should be in the node table
	 * unless we are operating in station mode or this is a
	 * multicast/broadcast frame.
	 */
	if (ic->ic_opmode == IEEE80211_M_STA || IEEE80211_IS_MULTICAST(macaddr))
		return ic->ic_bss;

	/* XXX can't hold lock across dup_bss 'cuz of recursive locking */
	IEEE80211_NODE_LOCK(ic);
	ni = _ieee80211_find_node(ic, macaddr);
	IEEE80211_NODE_UNLOCK(ic);
	if (ni == NULL &&
	    (ic->ic_opmode == IEEE80211_M_IBSS ||
	     ic->ic_opmode == IEEE80211_M_AHDEMO)) {
		/*
		 * Fake up a node; this handles node discovery in
		 * adhoc mode.  Note that for the driver's benefit
		 * we we treat this like an association so the driver
		 * has an opportunity to setup it's private state.
		 *
		 * XXX need better way to handle this; issue probe
		 *     request so we can deduce rate set, etc.
		 */
		ni = ieee80211_dup_bss(ic, macaddr);
		if (ni != NULL) {
			/* XXX no rate negotiation; just dup */
			ni->ni_rates = ic->ic_bss->ni_rates;
			if (ic->ic_newassoc)
				(*ic->ic_newassoc)(ic, ni, 1);
			/* XXX not sure how 802.1x works w/ IBSS */
			ieee80211_node_authorize(ic, ni);
		}
	}
	return ni;
}
#ifdef IEEE80211_DEBUG_REFCNT
EXPORT_SYMBOL(ieee80211_find_txnode_debug);
#else
EXPORT_SYMBOL(ieee80211_find_txnode);
#endif

/*
 * Like find but search based on the channel too.
 */
struct ieee80211_node *
#ifdef IEEE80211_DEBUG_REFCNT
ieee80211_find_node_with_channel_debug(struct ieee80211com *ic,
	const u_int8_t *macaddr, struct ieee80211_channel *chan,
	const char *func, int line)
#else
ieee80211_find_node_with_channel(struct ieee80211com *ic,
	const u_int8_t *macaddr, struct ieee80211_channel *chan)
#endif
{
	struct ieee80211_node *ni;
	int hash;

	hash = IEEE80211_NODE_HASH(macaddr);
	IEEE80211_NODE_LOCK(ic);
	LIST_FOREACH(ni, &ic->ic_hash[hash], ni_hash) {
		if (IEEE80211_ADDR_EQ(ni->ni_macaddr, macaddr) &&
		    ni->ni_chan == chan) {
			ieee80211_node_incref(ni);/* mark referenced */
#ifdef IEEE80211_DEBUG_REFCNT
			IEEE80211_DPRINTF(ic, IEEE80211_MSG_NODE,
			    ("%s (%s:%u) %s refcnt %d\n", __func__, func, line,
			     ether_sprintf(ni->ni_macaddr),
			     ieee80211_node_refcnt(ni)));
#endif
			break;
		}
	}
	IEEE80211_NODE_UNLOCK(ic);
	return ni;
}

/*
 * Like find but search based on the ssid too.
 */
struct ieee80211_node *
#ifdef IEEE80211_DEBUG_REFCNT
ieee80211_find_node_with_ssid_debug(struct ieee80211com *ic,
	const u_int8_t *macaddr, u_int ssidlen, const u_int8_t *ssid,
	const char *func, int line)
#else
ieee80211_find_node_with_ssid(struct ieee80211com *ic,
	const u_int8_t *macaddr, u_int ssidlen, const u_int8_t *ssid)
#endif
{
	struct ieee80211_node *ni;
	int hash;

	hash = IEEE80211_NODE_HASH(macaddr);
	IEEE80211_NODE_LOCK(ic);
	LIST_FOREACH(ni, &ic->ic_hash[hash], ni_hash) {
		if (IEEE80211_ADDR_EQ(ni->ni_macaddr, macaddr) &&
		    ni->ni_esslen == ic->ic_des_esslen &&
		    memcmp(ni->ni_essid, ic->ic_des_essid, ni->ni_esslen) == 0) {
			ieee80211_node_incref(ni);/* mark referenced */
#ifdef IEEE80211_DEBUG_REFCNT
			IEEE80211_DPRINTF(ic, IEEE80211_MSG_NODE,
			    ("%s (%s:%u) %s refcnt %d\n", __func__, func, line,
			     ether_sprintf(ni->ni_macaddr),
			     ieee80211_node_refcnt(ni)));
#endif
			break;
		}
	}
	IEEE80211_NODE_UNLOCK(ic);
	return ni;
}

static void
_ieee80211_free_node(struct ieee80211com *ic, struct ieee80211_node *ni)
{

	KASSERT(ni != ic->ic_bss, ("freeing bss node"));

	IEEE80211_DPRINTF(ic, IEEE80211_MSG_NODE,
		("%s %s\n", __func__, ether_sprintf(ni->ni_macaddr)));
	IEEE80211_AID_CLR(ni->ni_associd, ic->ic_aid_bitmap);
	TAILQ_REMOVE(&ic->ic_node, ni, ni_list);
	if (TAILQ_EMPTY(&ic->ic_node))
		ic->ic_inact_timer = 0;
	LIST_REMOVE(ni, ni_hash);
	if (_IF_QLEN(&ni->ni_savedq) != 0) {
		/*
		 * Drain power save queue.
		 */
		IF_DRAIN(&ni->ni_savedq);
		if (ic->ic_set_tim)
			(*ic->ic_set_tim)(ic, ni->ni_associd, 0);
	}
	(*ic->ic_node_free)(ic, ni);
}

void
#ifdef IEEE80211_DEBUG_REFCNT
ieee80211_free_node_debug(struct ieee80211com *ic,
	struct ieee80211_node *ni, const char *func, int line)
#else
ieee80211_free_node(struct ieee80211com *ic, struct ieee80211_node *ni)
#endif
{
	KASSERT(ni != ic->ic_bss, ("freeing ic_bss"));

#ifdef IEEE80211_DEBUG_REFCNT
	IEEE80211_DPRINTF(ic, IEEE80211_MSG_NODE,
		("%s (%s:%u) %s refcnt %d\n", __func__, func, line,
		 ether_sprintf(ni->ni_macaddr), ieee80211_node_refcnt(ni)-1));
#endif
	if (ieee80211_node_dectestref(ni)) {
		IEEE80211_NODE_LOCK_BH(ic);
		_ieee80211_free_node(ic, ni);
		IEEE80211_NODE_UNLOCK_BH(ic);
	}
}
#ifdef IEEE80211_DEBUG_REFCNT
EXPORT_SYMBOL(ieee80211_free_node_debug);
#else
EXPORT_SYMBOL(ieee80211_free_node);
#endif

void
ieee80211_free_allnodes(struct ieee80211com *ic)
{
	struct ieee80211_node *ni;

	IEEE80211_DPRINTF(ic, IEEE80211_MSG_NODE, ("free all nodes\n"));
	IEEE80211_NODE_LOCK_BH(ic);
	while ((ni = TAILQ_FIRST(&ic->ic_node)) != NULL) {
		if (ni->ni_associd != 0) {
			if (ic->ic_auth->ia_node_leave != NULL)
				ic->ic_auth->ia_node_leave(ic, ni);
			IEEE80211_AID_CLR(ni->ni_associd, ic->ic_aid_bitmap);
		}
		_ieee80211_free_node(ic, ni);
	}
	ieee80211_reset_erp(ic,ic->ic_curmode);
	IEEE80211_NODE_UNLOCK_BH(ic);

	if (ic->ic_bss != NULL)
		node_cleanup(ic, ic->ic_bss);	/* for station mode */
}

/*
 * Timeout inactive nodes.  Note that we cannot hold the node
 * lock while sending a frame as this would lead to a LOR.
 * Instead we use a generation number to mark nodes that we've
 * scanned and drop the lock and restart a scan if we have to
 * time out a node.  Since we are single-threaded by virtue of
 * controlling the inactivity timer we can be sure this will
 * process each node only once.
 */
void
ieee80211_timeout_nodes(struct ieee80211com *ic)
{
	struct ieee80211_node *ni;
	u_int gen = ic->ic_scangen++;		/* NB: ok 'cuz single-threaded*/

restart:
	IEEE80211_NODE_LOCK(ic);
	TAILQ_FOREACH(ni, &ic->ic_node, ni_list) {
		if (ni->ni_scangen == gen)	/* previously handled */
			continue;
		ni->ni_scangen = gen;
		/*
		 * Free fragment skb if not needed anymore
		 * (last fragment older than 1s).
		 * XXX doesn't belong here
		 */
		if (ni->ni_rxfrag[0] && jiffies > ni->ni_rxfragstamp + HZ) {
			kfree_skb(ni->ni_rxfrag[0]);
			ni->ni_rxfrag[0] = NULL;
		}
		if (ni->ni_inact && --ni->ni_inact <= 0) {
			IEEE80211_DPRINTF(ic, IEEE80211_MSG_NODE,
			    ("station %s timed out due to inactivity\n",
			    ether_sprintf(ni->ni_macaddr)));
			/*
			 * Send a deauthenticate frame.
			 *
			 * Drop the node lock before sending the
			 * deauthentication frame in case the driver takes     
			 * a lock, as this will result in a LOR between the     
			 * node lock and the driver lock.
			 */
			IEEE80211_NODE_UNLOCK(ic);
			IEEE80211_SEND_MGMT(ic, ni,
			    IEEE80211_FC0_SUBTYPE_DEAUTH,
			    IEEE80211_REASON_AUTH_EXPIRE);
			ieee80211_node_leave(ic, ni);
			ic->ic_stats.is_node_timeout++;
			goto restart;
		}
	}
	if (!TAILQ_EMPTY(&ic->ic_node))
		ic->ic_inact_timer = IEEE80211_INACT_WAIT;
	IEEE80211_NODE_UNLOCK(ic);
}

void
ieee80211_iterate_nodes(struct ieee80211com *ic, ieee80211_iter_func *f, void *arg)
{
	struct ieee80211_node *ni;

	IEEE80211_NODE_LOCK(ic);
	TAILQ_FOREACH(ni, &ic->ic_node, ni_list)
		(*f)(arg, ni);
	IEEE80211_NODE_UNLOCK(ic);
}
EXPORT_SYMBOL(ieee80211_iterate_nodes);

void
ieee80211_dump_node(struct ieee80211_node *ni)
{
	int i;

	printf("0x%p: mac %s refcnt %d\n", ni,
		ether_sprintf(ni->ni_macaddr), ieee80211_node_refcnt(ni));
	printf("\tscangen %u authmode %u flags 0x%x\n",
		ni->ni_scangen, ni->ni_authmode, ni->ni_flags);
	printf("\tassocid 0x%x txpower %u vlan %u\n",
		ni->ni_associd, ni->ni_txpower, ni->ni_vlan);
	printf ("rxfragstamp %u\n", ni->ni_rxfragstamp);
	for (i=0; i<17; i++) {
		printf("\t%d: txseq %u rxseq %u fragno %u\n", i, 
		       ni->ni_txseqs[i],
		       ni->ni_rxseqs[i] >> IEEE80211_SEQ_SEQ_SHIFT,
		       ni->ni_rxseqs[i] & IEEE80211_SEQ_FRAG_MASK);
	}
	printf("\trstamp %u rssi %u intval %u capinfo 0x%x\n",
		ni->ni_rstamp, ni->ni_rssi, ni->ni_intval, ni->ni_capinfo);
	printf("\tbssid %s essid \"%.*s\" channel %u:0x%x\n",
		ether_sprintf(ni->ni_bssid),
		ni->ni_esslen, ni->ni_essid,
		ni->ni_chan->ic_freq, ni->ni_chan->ic_flags);
	printf("\tfails %u inact %u txrate %u\n",
		ni->ni_fails, ni->ni_inact, ni->ni_txrate);
}

void
ieee80211_dump_nodes(struct ieee80211com *ic)
{
	struct ieee80211_node *ni;

	IEEE80211_NODE_LOCK(ic);
	TAILQ_FOREACH(ni, &ic->ic_node, ni_list)
		ieee80211_dump_node(ni);
	IEEE80211_NODE_UNLOCK(ic);
}

/*
 * Handle a station joining an 11g network.
 */
static void
ieee80211_node_join_11g(struct ieee80211com *ic, struct ieee80211_node *ni)
{

	/*
	 * Station isn't capable of short slot time.  Bump
	 * the count of long slot time stations and disable
	 * use of short slot time.  Note that the actual switch
	 * over to long slot time use may not occur until the
	 * next beacon transmission (per sec. 7.3.1.4 of 11g).
	 */

	if ((ni->ni_capinfo & IEEE80211_CAPINFO_SHORT_SLOTTIME) == 0) {
		ic->ic_longslotsta++;
		IEEE80211_DPRINTF(ic, IEEE80211_MSG_ASSOC,
			("station %s needs long slot time, "
			"count %d\n",
			ether_sprintf(ni->ni_macaddr),
			ic->ic_longslotsta));
		/* XXX vap's w/ conflicting needs won't work */
		if (ic->ic_curmode != IEEE80211_MODE_TURBO_G) 
			ieee80211_set_shortslottime(ic, 0);
	}
	/*
	 * If the new station is not an ERP station
	 * then bump the counter and enable protection
	 * if configured.
	 */
	if (!ieee80211_iserp_rateset(ic, &ni->ni_rates)) {
		ic->ic_nonerpsta++;
		IEEE80211_DPRINTF(ic, IEEE80211_MSG_ASSOC,
			("station %s is !ERP, %d non-ERP stations "
			"associated\n",
			ether_sprintf(ni->ni_macaddr),
			ic->ic_nonerpsta));
		/*
		 * If protection is configured, enable it.
		 */
		if (ic->ic_protmode != IEEE80211_PROT_NONE) {
			IEEE80211_DPRINTF(ic, IEEE80211_MSG_ASSOC,
			    ("enable use of protection\n"));
			ic->ic_flags |= IEEE80211_F_USEPROT;
		}
		/*
		 * If station does not support short preamble
		 * then we must enable use of Barker preamble.
		 */
		if ((ni->ni_capinfo & IEEE80211_CAPINFO_SHORT_PREAMBLE) == 0) {
			IEEE80211_DPRINTF(ic, IEEE80211_MSG_ASSOC,
			    ("station %s needs long preamble\n",
			    ether_sprintf(ni->ni_macaddr)));
			ic->ic_flags |= IEEE80211_F_USEBARKER;
			ic->ic_flags &= ~IEEE80211_F_SHPREAMBLE;
		}
	} else
		ni->ni_flags |= IEEE80211_NODE_ERP;
}

void
ieee80211_node_join(struct ieee80211com *ic, struct ieee80211_node *ni, int resp)
{
	int newassoc;

	if (ni->ni_associd == 0) {
		u_int16_t aid;

		/*
		 * It would be good to search the bitmap
		 * more efficiently, but this will do for now.
		 */
		for (aid = 1; aid < ic->ic_max_aid; aid++) {
			if (!IEEE80211_AID_ISSET(aid,
			    ic->ic_aid_bitmap))
				break;
		}
		if (aid >= ic->ic_max_aid) {
			IEEE80211_SEND_MGMT(ic, ni, resp,
			    IEEE80211_REASON_ASSOC_TOOMANY);
			ieee80211_node_leave(ic, ni);
			return;
		}
		ni->ni_associd = aid | 0xc000;
		IEEE80211_AID_SET(ni->ni_associd, ic->ic_aid_bitmap);
		newassoc = 1;
		if ((ic->ic_curmode == IEEE80211_MODE_11G) ||
		    (ic->ic_curmode == IEEE80211_MODE_TURBO_G))
			ieee80211_node_join_11g(ic, ni);
	} else
		newassoc = 0;

	IEEE80211_DPRINTF(ic, IEEE80211_MSG_ASSOC | IEEE80211_MSG_DEBUG,
		("station %s %s associated at aid %d\n",
		ether_sprintf(ni->ni_macaddr),
		newassoc ? "newly" : "already",
		IEEE80211_NODE_AID(ni)));

	/* give driver a chance to setup state like ni_txrate */
	if (ic->ic_newassoc)
		(*ic->ic_newassoc)(ic, ni, newassoc);
	IEEE80211_SEND_MGMT(ic, ni, resp, IEEE80211_STATUS_SUCCESS);
	/* tell the authenticator about new station */
	if (ic->ic_auth->ia_node_join != NULL)
		(*ic->ic_auth->ia_node_join)(ic, ni);
	ieee80211_notify_node_join(ic, ni, newassoc);
}

/*
 * Handle a station leaving an 11g network.
 */
static void
ieee80211_node_leave_11g(struct ieee80211com *ic, struct ieee80211_node *ni)
{

	KASSERT((ic->ic_curmode == IEEE80211_MODE_11G) ||
		(ic->ic_curmode == IEEE80211_MODE_TURBO_G),
		("not in 11g, curmode %x", ic->ic_curmode));

	/*
	 * If a long slot station do the slot time bookkeeping.
	 */

	if ((ni->ni_capinfo & IEEE80211_CAPINFO_SHORT_SLOTTIME) == 0) {
		KASSERT(ic->ic_longslotsta > 0,
		    ("bogus long slot station count %d", ic->ic_longslotsta));
		ic->ic_longslotsta--;
		IEEE80211_DPRINTF(ic, IEEE80211_MSG_ASSOC,
			("long slot time station %s leaves, count now %d\n",
			ether_sprintf(ni->ni_macaddr), ic->ic_longslotsta));
		if (ic->ic_longslotsta == 0) {
			/*
			 * Re-enable use of short slot time if supported
			 * and not operating in IBSS mode (per spec).
			 */
			if ((ic->ic_caps & IEEE80211_C_SHSLOT) &&
			    ic->ic_opmode != IEEE80211_M_IBSS) {
				IEEE80211_DPRINTF(ic, IEEE80211_MSG_ASSOC,
					("re-enable use of short slot time\n"));
				ieee80211_set_shortslottime(ic, 1);
			}
		}
	}
	/*
	 * If a non-ERP station do the protection-related bookkeeping.
	 */
	if ((ni->ni_flags & IEEE80211_NODE_ERP) == 0) {
		KASSERT(ic->ic_nonerpsta > 0,
		    ("bogus non-ERP station count %d", ic->ic_nonerpsta));
		ic->ic_nonerpsta--;
		IEEE80211_DPRINTF(ic, IEEE80211_MSG_ASSOC,
			("non-ERP station %s leaves, count now %d\n",
			ether_sprintf(ni->ni_macaddr), ic->ic_nonerpsta));
		if (ic->ic_nonerpsta == 0) {
			IEEE80211_DPRINTF(ic, IEEE80211_MSG_ASSOC,
				("disable use of protection\n"));
			ic->ic_flags &= ~IEEE80211_F_USEPROT;
			/* XXX verify mode? */
			if (ic->ic_caps & IEEE80211_C_SHPREAMBLE) {
				IEEE80211_DPRINTF(ic, IEEE80211_MSG_ASSOC,
					("re-enable use of short preamble\n"));
				ic->ic_flags |= IEEE80211_F_SHPREAMBLE;
				ic->ic_flags &= ~IEEE80211_F_USEBARKER;
			}
		}
	}
}

/*
 * Handle bookkeeping for station deauthentication/disassociation
 * when operating as an ap.
 */
void
ieee80211_node_leave(struct ieee80211com *ic, struct ieee80211_node *ni)
{

	KASSERT(ic->ic_opmode == IEEE80211_M_HOSTAP,
		("not in ap mode, mode %u", ic->ic_opmode));

	/*
	 * If node wasn't previously associated all
	 * we need to do is reclaim the reference.
	 */
	if (ni->ni_associd == 0)
		goto done;
	/*
	 * Tell the authenticator the station is leaving.
	 * Note that we must do this before yanking the
	 * association id as the authenticator uses the
	 * associd to locate it's state block.
	 */
	if (ic->ic_auth->ia_node_leave != NULL)
		(*ic->ic_auth->ia_node_leave)(ic, ni);
	IEEE80211_AID_CLR(ni->ni_associd, ic->ic_aid_bitmap);
	ni->ni_associd = 0;

	if ((ic->ic_curmode == IEEE80211_MODE_11G) ||
	    (ic->ic_curmode == IEEE80211_MODE_TURBO_G))
		ieee80211_node_leave_11g(ic, ni);

	/*
	 * Cleanup station state.  In particular clear various
	 * state that might otherwise be reused if the node
	 * is reused before the reference count goes to zero
	 * (and memory is reclaimed).
	 */
	ieee80211_sta_leave(ic, ni);
done:
	ieee80211_free_node(ic, ni);
}
EXPORT_SYMBOL(ieee80211_node_leave);

/*
 * Set the short slot time state and notify the driver.
 */
void
ieee80211_set_shortslottime(struct ieee80211com *ic, int onoff)
{
	if (onoff)
		ic->ic_flags |= IEEE80211_F_SHSLOT;
	else
		ic->ic_flags &= ~IEEE80211_F_SHSLOT;
	/* notify driver */
	if (ic->ic_updateslot != NULL)
		(*ic->ic_updateslot)(ic->ic_dev);
}
