/*
 *  EtherDump 2.0
 *  IPv4 packet sniffer using raw sockets
 *
 *  Changes by Peter Willis since 2.0:
 *    * Changed configuration option to reflect new name is
 *      "etherdump", not "packetdump". -p is now -e.
 *    * Added basic [ipv4] filtering rules.
 *    * Improved tcpdump output.
 *    * If etherdump was run as a program named tcpdump, defaults
 *      to tcpdump-like output.
 *    * Added -i to specify interface.
 *    * If EtherDump is executed as "tcpdump", tcpdump-like output
 *      is the default output type.
 *
 *  Changes by Erik Andersen since 2.0:
 *    * Converted to getopt instead of a million strcmps
 *    * cleanup
 *
 *  Todo:
 *    * Cache the /etc/protocols and /etc/services files and write
 *      native versions of the getproto* and getserv* functions so
 *      sniffing is faster.
 *
 *  Copyright (C) 2004  Christophe Devine
 *  Copyright (C) 2004  Peter Willis <psyphreak@phreaker.net>
 *  Copyright (C) 2004  Erik Andersen <andersen@codepoet.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <netpacket/packet.h>
#include <netinet/in.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <arpa/inet.h>
#include <net/if.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <netdb.h>
#include <sys/time.h>
#include <ctype.h>
#include <getopt.h>
#include <stdlib.h>

#define ETH_P_ALL 0x0003
#define ETH_P_IP  0x0800

#include <string.h>
#include <stdio.h>
#include <errno.h>
#include <time.h>

#define debug 0

struct ip_hdr
{
    unsigned char  hlv;         /* +00 - header len. & version  */
    unsigned char  tos;         /* +01 - type of service        */
    unsigned short tot_len;     /* +02 - total packet length    */
    unsigned short id;          /* +04 - identification         */
    unsigned short frag_off;    /* +06 - fragment offset field  */
    unsigned char  ttl;         /* +08 - time to live           */
    unsigned char  protocol;    /* +09 - ip protocol            */
    unsigned short check;       /* +10 - ip checksum            */
    unsigned long saddr;    /* +12 - source address         */
    unsigned long daddr;    /* +16 - destination address    */
};

struct icmp_hdr
{
    unsigned char  type;        /* +00 - message type           */
    unsigned char  code;        /* +01 - type sub-code          */
    unsigned short checksum;    /* +02 - icmp checksum          */
    unsigned short id;          /* +04 - identification         */
    unsigned short sequence;    /* +06 - sequence number        */
};

struct tcp_hdr
{
    unsigned short source;      /* +00 - source port            */
    unsigned short dest;        /* +02 - destination port       */
    unsigned long  seq;         /* +04 - sequence number        */
    unsigned long  ack_seq;     /* +08 - ack seq. number        */
    unsigned char  unused;      /* +12 - unused field           */
    unsigned char  flags;       /* +13 - tcp flags              */
     /* flags format: RESERVED|RESERVED|URG|ACK|PSH|RST|SYN|FIN */
    unsigned short window;      /* +14 - tcp window             */
    unsigned short check;       /* +16 - tcp checksum           */
    unsigned short urp_ptr;     /* +18 - urgent pointer         */
};

struct udp_hdr
{
    unsigned short source;      /* +00 - source port            */
    unsigned short dest;        /* +02 - destination port       */
    unsigned short len;         /* +04 - message length         */
    unsigned short check;       /* +06 - udp checksum           */
};

struct packet_filter_rule
{
	int proto;
	unsigned char set_proto;
	unsigned short source;		/* source port */
	unsigned char set_source;
	unsigned short dest;		/* destination port */
	unsigned char set_dest;
	unsigned long saddr;		/* source address */
	unsigned char set_saddr;
	unsigned long daddr;		/* destination address */
	unsigned char set_daddr;
	unsigned char negate;
	unsigned char complete;
}; /* struct size: 4 + 2 + 2 + 4 + 4 + 4 + 4  =  ~24 bytes */


int show_usage() {
	printf( "\n"
	"  usage: etherdump [options]\n"
	"\n"
	"   options:\n"
	"    -o,--output\t\toutput to a file instead of stdout\n"
	"    -r,--raw\t\toutput raw frames in binary\n"
	"    -h,--hex\t\toutput raw frames in ASCII hex\n"
	"    -f,--filter\t\tfilter printed packets based on some rules\n"
	"    -t,--tcpdump\tprint packets out in tcpdump-like format\n"
	"    -e,--etherdump\tprint packetd out in etherdump format\n"
	"    \t\t\t[default]\n"
	"    -i,--interface\tinterface to listen for traffic on\n"
    "\n");
	return(0);
}

int parse_filter(int argc, char **argv, int i, struct packet_filter_rule **p_filters, int p_filter_idx) {
	for (;i<argc;i++) {
		if (
	    		(strncmp(argv[i], "not", strlen(argv[i])) == 0) ||
			 (strncmp(argv[i], "!", strlen(argv[i])) == 0)
		) {
			p_filters[p_filter_idx]->negate = 1;
		} else
		if (
			(strncmp(argv[i], "proto", strlen(argv[i])) == 0) ||
			 (strncmp(argv[i], "protocol", strlen(argv[i])) == 0)
		) {
			struct protoent *pf_proto;
			if ( ++i >= argc ) {
				show_usage();
				return(-1);
			}
			if ( (pf_proto = getprotobyname(argv[i])) == NULL ) {
				if ( (pf_proto = getprotobynumber(atoi(argv[i]))) == NULL ) { /* bad protocol */
					show_usage();
					return(-1);
				}
			}
			if (debug) printf("setting proto %i\n", pf_proto->p_proto);
			p_filters[p_filter_idx]->proto = pf_proto->p_proto;
			p_filters[p_filter_idx]->set_proto = 1;
		} else
		if (
			(strncmp(argv[i], "sport", strlen(argv[i])) == 0) ||
			 (strncmp(argv[i], "sourceport", strlen(argv[i])) == 0)
		) {
			struct protoent *pf_proto;
			struct servent *pf_serv;
			char pf_svcname[33];
			if ( (! p_filters[p_filter_idx]->set_proto) || (++i >= argc) ) {
				show_usage();
				return(-1);
			}
			memset(pf_svcname, 0, sizeof(pf_svcname));
			if ( (pf_proto = getprotobynumber(p_filters[p_filter_idx]->proto)) == NULL ) {
				if (debug) printf("bad proto\n");
				show_usage();
				return(-1);
			}
			strncpy(pf_svcname, pf_proto->p_name, sizeof(pf_svcname)-1);
			if ( (pf_serv = getservbyname(argv[i], pf_svcname)) == NULL ) {
				if (debug) printf("bad name\n");
				if ( (pf_serv = getservbyport(htons(atoi(argv[i])), pf_svcname)) == NULL ) { /* bad port */
					show_usage();
					return(-1);
				}
			}
			if (debug) printf("setting sport %i\n", ntohs(pf_serv->s_port));
			p_filters[p_filter_idx]->source = ntohs(pf_serv->s_port);
			p_filters[p_filter_idx]->set_source = 1;
		} else
		if (
			(strncmp(argv[i], "dport", strlen(argv[i])) == 0) ||
			(strncmp(argv[i], "destinationport", strlen(argv[i])) == 0)
		) {
			struct protoent *pf_proto;
			struct servent *pf_serv;
			char pf_svcname[33];
			if (debug) printf("at dport\n");
			if (! p_filters[p_filter_idx]->set_proto) {
				show_usage();
				return(-1);
			}
			if ( ++i >= argc ) {
				show_usage();
				return(-1);
			}
			memset(pf_svcname, 0, sizeof(pf_svcname));
			if ( (pf_proto = getprotobynumber(p_filters[p_filter_idx]->proto)) == NULL ) {
				show_usage();
				return(-1);
			}
			strncpy(pf_svcname, pf_proto->p_name, sizeof(pf_svcname)-1);
			if ( (pf_serv = getservbyname(argv[i], pf_svcname) ) == NULL ) {
				if ( (pf_serv = getservbyport(htons(atoi(argv[i])), pf_svcname) ) == NULL ) { /* bad port */
					show_usage();
					return(-1);
				}
			}
			if (debug) printf("setting dport %i\n", ntohs(pf_serv->s_port));
			p_filters[p_filter_idx]->dest = ntohs(pf_serv->s_port);
			p_filters[p_filter_idx]->set_dest = 1;
		} else
		if (
			(strncmp(argv[i], "src", strlen(argv[i])) == 0) ||
			(strncmp(argv[i], "source", strlen(argv[i])) == 0)
		) {
			struct in_addr pf_addr;
			if ( (++i >= argc) || ( inet_aton(argv[i], &pf_addr) == 0 ) ) {
				show_usage();
				return(-1);
			}
			if (debug) printf("setting src %s\n", inet_ntoa(pf_addr));
			p_filters[p_filter_idx]->saddr = pf_addr.s_addr;
			p_filters[p_filter_idx]->set_saddr = 1;
		} else
		if (
			(strncmp(argv[i], "dst", strlen(argv[i])) == 0) ||
			(strncmp(argv[i], "destination", strlen(argv[i])) == 0)
		) {
			struct in_addr pf_addr;
			if ( (++i >= argc) || ( inet_aton(argv[i], &pf_addr) == 0 ) ) {
				show_usage();
				return(-1);
			}
			if (debug) printf("setting dst %s\n", inet_ntoa(pf_addr));
			p_filters[p_filter_idx]->daddr = pf_addr.s_addr;
			p_filters[p_filter_idx]->set_daddr = 1;
		}
	}
	p_filters[p_filter_idx]->complete = 1;
	p_filter_idx++;
	return(p_filter_idx);
}

static const struct option etherdump_long_options[] = {
	{ "output",	1, NULL, 'o' },
	{ "filter",	1, NULL, 'f' },
	{ "interface", 1, NULL, 'i' },
	{ "raw",	0, NULL, 'r' },
	{ "hex",	0, NULL, 'h' },
	{ "tcpdump",	0, NULL, 't' },
	{ "etherdump",	0, NULL, 'e' },
	{ 0, 0, 0, 0 }
};

int etherdump_main( int argc, char *argv[] )
{
    unsigned char buffer[4096];
    int raw_sock, n;

    struct sockaddr_ll sll;
    struct packet_mreq mr;
    struct ifreq ifr;
    int socktype = SOCK_DGRAM;

    int i;
    struct protoent *proto;
    char *outputfile = NULL;
    FILE *F_logfd = stdout; /* guess. :P */
    int logfd = 1; /* default to stdout */
    int printformat = 0; /* 0 for packetdump format, 1 for tcpdump */
    int dumptype = 0; /* 0 for disabled, 1 for ASCII hex, 2 for raw binary */
    char interface[16];

    struct icmp_hdr *icmp;
    struct udp_hdr *udp;
    struct tcp_hdr *tcp;
    struct ip_hdr *ip;
    struct in_addr src;
    struct in_addr dst;
    struct tm *lt;
    struct timeval tv;
    struct sockaddr_ll from;
    int fromlen;
    int next_packet;
    unsigned char tcpflag_urg=0, tcpflag_ack=0, tcpflag_psh=0, tcpflag_rst=0, tcpflag_syn=0, tcpflag_fin=0;
    char srcaddr[16];
    char dstaddr[16];
    char tmpproto[16];
    unsigned short sport, dport;

    int p_filter_idx = 0;
    int max_p_filters = 32;
    struct packet_filter_rule p_filters[max_p_filters];
    struct packet_filter_rule *p_filters_pointer[max_p_filters];

	/* set default filter struct entries */
	for (i=0;i<max_p_filters;i++) {
		p_filters[i].proto = -1;
		p_filters[i].set_proto = 0;
		p_filters[i].source = 0;
		p_filters[i].set_source = 0;
		p_filters[i].dest = 0;
		p_filters[i].set_dest = 0;
		p_filters[i].saddr = 0;
		p_filters[i].set_saddr = 0;
		p_filters[i].daddr = 0;
		p_filters[i].set_daddr = 0;
		p_filters[i].negate = 0;
		p_filters[i].complete = 0;
		p_filters_pointer[i] = &p_filters[i];
	}

    /* check the arguments */
    if( argc < 2 ) {
	show_usage();
	return( 1 );
    }

    if (strstr(argv[0], "tcpdump") != NULL)
	    printformat = 1;

    while ((i = getopt_long (argc, argv, "htro:ei:f:", // fhrtio:
		    etherdump_long_options, NULL)) > 0)
    {
	 switch (i) {
	    /* for some reason, "case 't'" causes a seg fault if it's not above the 'i' or 'e' part... strange huh? */
	    case 'h':
		socktype = SOCK_RAW;
		dumptype = 1;
		break;
	    case 't':
		socktype = SOCK_DGRAM;
		dumptype = 0;
		printformat = 1;
		break;
	    case 'r':
		socktype = SOCK_RAW;
		dumptype = 2;
		break;
	    case 'o':
		outputfile = optarg;
		break;
	    case 'e':
		socktype = SOCK_DGRAM;
		dumptype = 0;
		printformat = 0;
		break;
	    case 'i':
		memset(interface, '\0', sizeof(interface));
		strncpy(interface, optarg /*argv[optind]*/, sizeof(interface));
		break;
	    case 'f':
		if ( (p_filter_idx = parse_filter(argc, argv, optind-1, p_filters_pointer, p_filter_idx)) == -1) {
			return(1);
		}
		break;
	}
    }

    if (strlen(interface) < 1) {
	show_usage();
	return(1);
    }

    /* create the raw socket */
    if( ( raw_sock = socket( PF_PACKET, socktype, htons( ETH_P_ALL ) ) ) < 0 ) {
	perror( "socket" );
	return( 1 );
    }

    /* find the interface index */
    memset( &ifr, 0, sizeof( ifr ) );
    strncpy( ifr.ifr_name, interface, sizeof( ifr.ifr_name ) );
    if( ioctl( raw_sock, SIOCGIFINDEX, &ifr ) < 0 ) {
	perror( "ioctl(SIOCGIFINDEX)" );
	return( 1 );
    }

    /* bind the raw socket to the interface */
    memset( &sll, 0, sizeof( sll ) );
    sll.sll_family   = AF_PACKET;
    sll.sll_ifindex  = ifr.ifr_ifindex;
    sll.sll_protocol = htons( ETH_P_ALL );
    if( bind( raw_sock, (struct sockaddr *) &sll, sizeof( sll ) ) < 0 ) {
	perror( "bind" );
	return( 1 );
    }

    /* enable promiscuous mode */
    memset( &mr, 0, sizeof( mr ) );
    mr.mr_ifindex = ifr.ifr_ifindex;
    mr.mr_type    = PACKET_MR_PROMISC;
    if( setsockopt( raw_sock, SOL_PACKET, PACKET_ADD_MEMBERSHIP, &mr, sizeof( mr ) ) < 0 ) {
	perror( "setsockopt" );
	return( 1 );
    }

    /* (optionally) open a log file */
    if (outputfile != NULL) {
	if ( (logfd = open(outputfile, O_WRONLY|O_CREAT|O_TRUNC, S_IRUSR|S_IWUSR )) == -1) {
	    perror ( "open" );
	    return( 1 );
	}
	if ( (F_logfd = fdopen(logfd, "w")) == NULL ) {
	    perror ( "fdopen" );
	    return( 1 );
	}
    }

    while( 1 ) {

	next_packet = 0;
	memset(buffer, '\0', sizeof(buffer));

	/* wait for packets */
	fromlen = sizeof( from );
	if( ( n = recvfrom( raw_sock, buffer, 4096, 0, (struct sockaddr *) &from, &fromlen ) ) < 0 ) {
	    if( errno == ENETDOWN ) {
		if (debug) fprintf(stderr, "sleeping for 30 secs\n");
		sleep( 30 );
		continue;
	    } else {
		perror( "recvfrom" );
		return( 1 );
	    }
	}

	/* skip duplicate packets on the loopback interface */
	if( from.sll_pkttype == PACKET_OUTGOING && ! strcmp( interface, "lo" ) ) {
	    if (debug)
		    fprintf(stderr, "skipping loopback duplicate packet\n");
	    continue;
	}

	if (dumptype == 1) { /* dump ASCII hex of raw frames in 16 byte lines */

	    for (i=0; i<=n; i++) {
		if ( (i % 16) == 0 ) {
		    fprintf(F_logfd, i == 0 ? "%.4x" : "\n%.4x", i);
		    fflush(NULL);
		}
		fprintf(F_logfd, " %.2X", buffer[i]);
		fflush(NULL);
	    }
	    fprintf(F_logfd, "\n");
	    fflush(NULL);

	} else if (dumptype == 2) { /* dump raw frames in binary */

		write(logfd, buffer, n);

	} else { /* display certain packet types all nifty-like */

		/* we're only interested in standard IPv4 packets */
	    if( ntohs( from.sll_protocol ) != ETH_P_IP ) {
		continue;
	    }

	    /* have a look inside the IP header */
	    ip = (struct ip_hdr *) buffer;
	    src.s_addr = ip->saddr;
	    dst.s_addr = ip->daddr;
	    memset(srcaddr, '\0', sizeof(srcaddr)); /* I'm */
	    memset(dstaddr, '\0', sizeof(dstaddr)); /* paranoid. */
	    strncpy(srcaddr, inet_ntoa(src), sizeof(srcaddr));
	    strncpy(dstaddr, inet_ntoa(dst), sizeof(dstaddr));

	    /* get the time */
	    gettimeofday(&tv, NULL);
	    lt = localtime( &tv.tv_sec );

	    switch( ip->protocol )
	    {
		case 6: /* SOL_TCP */
		    tcp = (struct tcp_hdr *) ( (unsigned char *) ip + ( (ip->hlv & 0x0F) << 2 ) );
		    sport = htons( tcp->source );
		    dport = htons( tcp->dest );
		    /* grab flags */
		    tcpflag_urg = (tcp->flags & (1<<5)) ? 1 : 0;
		    tcpflag_ack = (tcp->flags & (1<<4)) ? 1 : 0;
		    tcpflag_psh = (tcp->flags & (1<<3)) ? 1 : 0;
		    tcpflag_rst = (tcp->flags & (1<<2)) ? 1 : 0;
		    tcpflag_syn = (tcp->flags & (1<<1)) ? 1 : 0;
		    tcpflag_fin = (tcp->flags & (1<<0)) ? 1 : 0;
		    goto common;

		case 17: /* SOL_UDP */
		    udp = (struct udp_hdr *) ( (unsigned char *) ip + ( (ip->hlv & 0x0F) << 2 ) );
		    sport = htons( udp->source );
		    dport = htons( udp->dest );
		    goto common;

		case 1: /* SOL_ICMP */
		    icmp = (struct icmp_hdr *) ( (unsigned char *) ip + ( (ip->hlv & 0x0F) << 2 ) );
		    sport = icmp->type;
		    dport = icmp->code;
		    goto common;

		common:
		    	/* filtering!!! */
			for (i=0;i<p_filter_idx;i++) {
				if ( p_filters[i].set_proto ) { /* check protocol */
					if ( 
						(p_filters[i].negate && ip->protocol == p_filters[i].proto) ||
						 ( ip->protocol != p_filters[i].proto )
					) {
						if (debug)
							fprintf(stderr, "protocol %i does not match proto %i\n", ip->protocol, p_filters[i].proto);
						next_packet=1;
						break;
					}
				}
				if ( p_filters[i].set_source ) { /* check source port */
					if (
						( p_filters[i].negate && sport == p_filters[i].source ) ||
						 ( sport == p_filters[i].source )
					) {
						if (debug)
							fprintf(stderr, "source port %i does not match source port %i\n", sport, p_filters[i].source);
						next_packet=1;
						break;
					}
				}
				if ( p_filters[i].set_dest ) { /* check dest port */
					if (
						( p_filters[i].negate && dport == p_filters[i].dest) ||
						 ( dport != p_filters[i].dest )
					) {
						if (debug)
							fprintf(stderr, "destination port %i does not match destination port %i\n", dport, p_filters[i].dest);
						next_packet=1;
						break;
					}
				}
				if ( p_filters[i].set_saddr ) {
					if (
						( p_filters[i].negate && ip->saddr == p_filters[i].saddr ) ||
						 ( ip->saddr != p_filters[i].saddr )
					) { /* check dst address */
						struct in_addr something;
						char address1[16], address2[16];
						something.s_addr = p_filters[i].saddr;
						strcpy(address1, inet_ntoa(src));
						strcpy(address2, inet_ntoa(something));
						if (debug) printf("saddr %s does not match src %s\n", address1, address2);
						next_packet=1;
						break;
				    	}
				}
				if ( p_filters[i].set_daddr ) {
					if (
						( p_filters[i].negate && ip->daddr == p_filters[i].daddr ) ||
						 ( ip->daddr != p_filters[i].daddr )
					) { /* check dst address */
						struct in_addr something;
						char address1[16], address2[16];
						something.s_addr = p_filters[i].daddr;
						strcpy(address1, inet_ntoa(dst));
						strcpy(address2, inet_ntoa(something));
						if (debug) printf("daddr %s does not match dst %s\n", address1, address2);
						next_packet=1;
						break;
				    	}
				}
		    }

		    /* print out !!! */
		    if (next_packet)
			    break;

		    if (printformat == 1) { /* tcpdump */

			fprintf(F_logfd, "%02d:%02d:%02d.%d %s.%d > %s.%d: ", lt->tm_hour, lt->tm_min,
				lt->tm_sec, (int)tv.tv_usec, srcaddr, sport, dstaddr, dport);
		    	if (ip->protocol == 6) { /* tcp-specific printing */
				if (tcpflag_syn)
				    fprintf(F_logfd, "S");
			    	if (tcpflag_fin)
				    fprintf(F_logfd, "F");
			    	if (tcpflag_psh)
				    fprintf(F_logfd, "P");
			    	if (tcpflag_rst)
				    fprintf(F_logfd, "R");
			    	if ( ! (tcpflag_syn||tcpflag_fin||tcpflag_psh||tcpflag_rst) )
				    fprintf(F_logfd, ".");
				//printf(" ack %li", htons(tcp->ack_seq)); // fix this pls? :|
			} else if (ip->protocol == 17) { /* udp-specific printing */
				fprintf(F_logfd, "udp %i", n);
			}
		    	fprintf(F_logfd, "\n");
		    	fflush(NULL);

		    } else { /* etherdump */

		     /* get the ip protocol */
		     proto = getprotobynumber(ip->protocol);
		     memset(tmpproto, '\0', sizeof(tmpproto));
		     strncpy(tmpproto, proto->p_name, sizeof(tmpproto));
		     /* convert to upper case */
		     for (i=0; i < (int)sizeof(tmpproto); i++) {
				tmpproto[i] = toupper(tmpproto[i]);
		     }
			fprintf(F_logfd, "%02d-%02d %02d:%02d:%02d %4s: %15s : %-5d -> %15s : %-5d  %4d\n",
				lt->tm_mon, lt->tm_mday, lt->tm_hour, lt->tm_min, lt->tm_sec,
				tmpproto, srcaddr, sport, dstaddr, dport, n);
			fflush(NULL);

		    }
		    break;

		default :
		    fprintf(F_logfd, " unsupported IP protocol %d from %s to %s\n", ip->protocol,
			    srcaddr, dstaddr);
		    break;
	    }
	}
    }

    return( 0 );
}
