/*
 * include/asm-arm/arch-em86xx/hardware.h
 *
 * Copyright 2002-2004, Sigma Designs, Inc 
 *
 * This file contains the hardware definitions for EM86XX
 *
 * by Ho Lee 01/27/2003
 */

#ifndef __ASM_ARCH_HARDWARE_H
#define __ASM_ARCH_HARDWARE_H

#include <asm/arch/emhwlib_registers.h>
#include <asm/arch/board/hardware.h>

/* macro to get at IO space when running virtually */
/*  but we are not here... */
#define IO_ADDRESS(x) (x)

/* ------------------------------------------------------------------------
 *  EM86XX Memory Map as PCI Slave : relative to PCI base address
 * ------------------------------------------------------------------------
 */

// DMA Access
#if (defined(CONFIG_ARCH_MAMBO) && (CONFIG_ARCH_MAMBO_REV == 1))
#define EM86XX_PCI_SLAVEDMA_1BYTE_BASE      0x00000000
#define EM86XX_PCI_SLAVEDMA_2BYTE_BASE      0x00001000
#define EM86XX_PCI_SLAVEDMA_3BYTE_BASE      0x00002000
#define EM86XX_PCI_SLAVEDMA_4BYTE_BASE      0x00003000
#define EM86XX_PCI_SLAVEDMA_1BYTE_REV_BASE  0x00004000
#define EM86XX_PCI_SLAVEDMA_2BYTE_REV_BASE  0x00005000
#define EM86XX_PCI_SLAVEDMA_3BYTE_REV_BASE  0x00006000
#define EM86XX_PCI_SLAVEDMA_4BYTE_REV_BASE  0x00007000
#elif (defined(CONFIG_ARCH_MAMBO) && (CONFIG_ARCH_MAMBO_REV >= 2)) || defined(CONFIG_ARCH_TANGO)
#define EM86XX_PCI_SLAVEDMA_4BYTE_BASE      0x00000000
#define EM86XX_PCI_SLAVEDMA_4BYTE_REV_BASE  0x00004000
#endif

// Tiemout
#if (defined(CONFIG_ARCH_MAMBO) && (CONFIG_ARCH_MAMBO_REV == 1))
#define EM86XX_PCI_TIMEOUT_VALUE            0x00008000
#define EM86XX_PCI_TIMEOUT_STATUS           0x00008004
#define EM86XX_PCI_TIMER_COUNTER            0x00008008
#define EM86XX_PCI_TIMER_TEST_REGISTER      0x0000800c
#define EM86XX_PCI_WAKEUP_REGISTER          0x00008010
#elif (defined(CONFIG_ARCH_MAMBO) && (CONFIG_ARCH_MAMBO_REV >= 2)) || defined(CONFIG_ARCH_TANGO)
#define EM86XX_PCI_SLAVEABORT_VALUE         0x00008000
#define EM86XX_PCI_SLAVEABORT_STATUS        0x00008004
#define EM86XX_PCI_SLAVEABORT_CLEAR         0x0000800c
#define EM86XX_PCI_WAKEUP_REGISTER          0x00008010
#endif

// Slave configuration
#define EM86XX_PCI_SLAVECFG_BASE            0x00009000
#define EM86XX_PCI_REGION_0_BASE            0x00009000
#define EM86XX_PCI_REGION_1_BASE            0x00009004
#define EM86XX_PCI_REGION_2_BASE            0x00009008
#define EM86XX_PCI_REGION_3_BASE            0x0000900c
#define EM86XX_PCI_REGION_4_BASE            0x00009010
#define EM86XX_PCI_REGION_5_BASE            0x00009014
#define EM86XX_PCI_REGION_6_BASE            0x00009018
#define EM86XX_PCI_REGION_7_BASE            0x0000901c
#define EM86XX_PCI_PCI_IRQ                  0x00009020

/* ------------------------------------------------------------------------
 *  EM86XX Memory Map and Registers as Host
 * ------------------------------------------------------------------------
 */

/*
 * EM86XX Register Base Addresses
 */

// Configuration Memory
#define MEMORY_BASE_CONFIG          0x00000000  /* configuration space */
#define REG_BASE_SYSTEM             0x00010000  /* System Block */
#define REG_BASE_HOST               0x00020000  /* Host Interface */    
#define REG_BASE_DRAMCTRL0          0x00030000  /* DRAM Controller 0 */
#define REG_BASE_DRAMCTRL1          0x00040000  /* DRAM Controller 1 */
#define REG_BASE_DRAMCTRL2          0x00050000  /* DRAM Controller 2 */
#define REG_BASE_CPU                0x00060000  /* CPU Block */
#define REG_BASE_VIDEOOUT           0x00070000  /* Video Output */
#define REG_BASE_MPEG0              0x00080000  /* MPEG Engine 0 */
#define REG_BASE_MPEG1              0x00090000  /* MPEG Engine 1 */
#define REG_BASE_TSDEMUX            0x000a0000  /* Transport Demux */
#define REG_BASE_AUDIO0             0x000c0000  /* Audio Block 0 */
#define REG_BASE_AUDIO1             0x000d0000  /* Audio Block 1 */

// RISC Memories
#define MEMORY_BASE_RISC            0x00100000
#define REG_BASE_MPEG0_RISC         0x00100000  /* MPEG Engine 0 RISC */
#define REG_BASE_MPEG1_RISC         0x00120000  /* MPEG Engine 1 RISC */
#define REG_BASE_TSDEMUX_RISC       0x00140000  /* Transport Demux RISC */
#define REG_BASE_AUDIO0_DSP         0x00180000  /* Audio Block 0 DSP */
#define REG_BASE_AUDIO1_DSP         0x001a0000  /* Audio Block 1 DSP */

// DRAM Memories
#if (defined(CONFIG_ARCH_MAMBO) && (CONFIG_ARCH_MAMBO_REV == 1))
#define MEMORY_BASE_DRAMCTRL0       0x10000000  /* Memory controlled by DRAM controller 0 */
#define MEMORY_BASE_DRAMCTRL1       0x20000000  /* Memory controlled by DRAM controller 1 */
#define MEMORY_BASE_DRAMCTRL2       0x30000000  /* Memory controlled by DRAM controller 2 */
#elif (defined(CONFIG_ARCH_MAMBO) && (CONFIG_ARCH_MAMBO_REV >= 2)) || defined(CONFIG_ARCH_TANGO)
#define MEMORY_BASE_DRAMCTRL0       0x90000000  /* Memory controlled by DRAM controller 0 */
#define MEMORY_BASE_DRAMCTRL1       0xa0000000  /* Memory controlled by DRAM controller 1 */
#define MEMORY_BASE_DRAMCTRL2       0xb0000000  /* Memory controlled by DRAM controller 2 */
#endif

#define MEMORY_BASE_DRAMCTRL0_NC    0x10000000  /* Memory controlled by DRAM controller 0 */
#define MEMORY_BASE_DRAMCTRL1_NC    0x20000000  /* Memory controlled by DRAM controller 1 */
#define MEMORY_BASE_DRAMCTRL2_NC    0x30000000  /* Memory controlled by DRAM controller 2 */

#define EM86XX_DRAM_C2NC(x)         ((x) & 0x7fffffff)
#define EM86XX_DRAM_NC2C(x)         ((x) | 0x80000000)

// Host Memories
#define MEMROY_BASE_HOST            0x40000000  /* Host memory */
#define MEMORY_BASE_HOST_SFLASH     0x40000000  /* Serial Flash */
#define MEMORY_BASE_HOST_PB0        0x44000000  /* Peripheral Bus CS #0 Memory */
#define MEMORY_BASE_HOST_PB1        0x45000000  /* Peripheral Bus CS #1 Memory */
#define MEMORY_BASE_HOST_PB2        0x46000000  /* Peripheral Bus CS #2 Memory */
#define MEMORY_BASE_HOST_PB3        0x47000000  /* Peripheral Bus CS #3 Memory */
#define MEMORY_BASE_HOST_PB_CS(n)   (MEMORY_BASE_HOST_PB0 + (0x01000000 * (n)))

// PCI Memories
#define MEMORY_BASE_PCI_CONFIG      0x50000000  /* PCI configuration */
#define MEMORY_BASE_PCI_IO          0x58000000  /* PCi I/O space */
#define MEMORY_BASE_PCI_MEMORY      0x60000000  /* PCI Memory Base */

/*
 * System Block 
 *
 * Base Address : REG_BASE_SYSTEM
 */

// GPIO 
#define SYS_gpio_base               0x0500

#define GPIO_DIR_INPUT(gpio)        ((1 << (16 + (gpio))))
#define GPIO_DIR_OUTPUT(gpio)       ((1 << (16 + (gpio))) | (1 << (gpio)))
#define GPIO_DATA_SET(gpio)         ((1 << (16 + (gpio))) | (1 << (gpio)))
#define GPIO_DATA_CLEAR(gpio)       ((1 << (16 + (gpio))))

/*
 * Host Interface  
 *
 * Base Address : REG_BASE_HOST
 */

// Peripheral bus Registers
#define HOST_pb0_base               0x0000
#define HOST_pb1_base               0x0200
#define HOST_pb2_base               0x0400
#define HOST_pb3_base               0x0600
#define HOST_pb_base_cs(n)          (HOST_pb0_base + (0x0200 * (n)))

// Peripheral bus CS #0 : IDE 
// Peripheral bus CS #1 : IDE 
#define PB_ide_data                 (PB_ide_base + IDE_data)
#define PB_ide_error                (PB_ide_base + IDE_error)
#define PB_ide_count                (PB_ide_base + IDE_count)
#define PB_ide_start_sector         (PB_ide_base + IDE_start_sector)
#define PB_ide_cylinder_lo          (PB_ide_base + IDE_cylinder_lo)
#define PB_ide_cylinder_hi          (PB_ide_base + IDE_cylinder_hi)
#define PB_ide_head_device          (PB_ide_base + IDE_head_device)
#define PB_ide_cmd_stat0            (PB_ide_base + IDE_cmd_stat)

#define PB_ide_irq_stat             (PB_ide_base + IDE_irq_stat)
#define PB_ide_cmd_stat1            (PB_ide_base + IDE_cmd_stat__)

#define REG_BASE_HOST_ISAIDE        (REG_BASE_HOST + PB_ide_base)       // ATAPI registers

// Peripheral bus configuration
#define PB_timing_slot(n)           (PB_timing0 + (0x04 * (n)))
#define PB_use_timing_slot(n)       (PB_use_timing0 + (0x04 * (n)))
#define PB_cs_config                (PB_CS_config)

// Switch Box 
#if defined(CONFIG_ARCH_MAMBO)
#define SBOX_base                   0x1000
#define SBOX_reset                  (SBOX_base + 0x00)
#define SBOX_route                  (SBOX_base + 0x04)
#elif defined(CONFIG_ARCH_TANGO)
#define SBOX_base                   0x9000
#if (CONFIG_ARCH_TANGO10_REV == 1)
#define SBOX_reset                  (SBOX_base + 0x80)
#define SBOX_route                  (SBOX_base + 0x88)
#define SBOX_mbus_w0                (SBOX_base + 0x90)
#define SBOX_mbus_w1                (SBOX_base + 0x94)
#define SBOX_pci_master             (SBOX_base + 0x98)
#define SBOX_pci_slave              (SBOX_base + 0x9c)
#define SBOX_cipher                 (SBOX_base + 0xa0)
#define SBOX_ide_isa                (SBOX_base + 0xa4)
#define SBOX_ide_dvd                (SBOX_base + 0xa8)
#else
#define SBOX_reset                  (SBOX_base + 0xa0)
#define SBOX_route                  (SBOX_base + 0xa8)
#define SBOX_mbus_w0                (SBOX_base + 0x80)
#define SBOX_mbus_w1                (SBOX_base + 0x84)
#define SBOX_pci_master             (SBOX_base + 0x88)
#define SBOX_pci_slave              (SBOX_base + 0x8c)
#define SBOX_cipher                 (SBOX_base + 0x90)
#define SBOX_ide_isa                (SBOX_base + 0x94)
#define SBOX_ide_dvd                (SBOX_base + 0x98)
#endif
#endif

// Semaphore
#if defined(CONFIG_ARCH_TANGO)
#define MISCSEM_base                0x9000
#define MISC_semaphore0             (MISCSEM_base + 0x40)
#define MISC_semaphore1             (MISCSEM_base + 0x44)
#define MISC_semaphore2             (MISCSEM_base + 0x48)
#define MISC_semaphore3             (MISCSEM_base + 0x4c)
#define MISC_semaphore4             (MISCSEM_base + 0x50)
#define MISC_semaphore5             (MISCSEM_base + 0x54)
#define MISC_semaphore6             (MISCSEM_base + 0x58)
#define MISC_semaphore7             (MISCSEM_base + 0x5c)
#define MISC_semaphore8             (MISCSEM_base + 0x60)
#define MISC_semaphore9             (MISCSEM_base + 0x64)
#define MISC_semaphore10            (MISCSEM_base + 0x68)
#define MISC_semaphore11            (MISCSEM_base + 0x6c)
#define MISC_semaphore12            (MISCSEM_base + 0x70)
#define MISC_semaphore13            (MISCSEM_base + 0x74)
#define MISC_semaphore14            (MISCSEM_base + 0x78)
#define MISC_semaphore15            (MISCSEM_base + 0x7c)
#endif

// MBUS interface
#define MIF_w0_base                 MIF_W0_ADD
#define MIF_w1_base                 MIF_W1_ADD
#define MIF_r0_base                 MIF_R0_ADD
#define MIF_r1_base                 MIF_R1_ADD

#define MIF_add_offset              0x00
#define MIF_cnt_offset              0x04
#define MIF_add2_skip_offset        0x08
#define MIF_cmd_offset              0x0c

#define MIF_w0_add                  (MIF_w0_base + 0x00)
#define MIF_w0_cnt                  (MIF_w0_base + 0x04)
#define MIF_w0_add2_skip            (MIF_w0_base + 0x08)
#define MIF_w0_cmd                  (MIF_w0_base + 0x0c)

#define MIF_w1_add                  (MIF_w1_base + 0x00)
#define MIF_w1_cnt                  (MIF_w1_base + 0x04)
#define MIF_w1_add2_skip            (MIF_w1_base + 0x08)
#define MIF_w1_cmd                  (MIF_w1_base + 0x0c)

#define MIF_r0_add                  (MIF_r0_base + 0x00)
#define MIF_r0_cnt                  (MIF_r0_base + 0x04)
#define MIF_r0_add2_skip            (MIF_r0_base + 0x08)
#define MIF_r0_cmd                  (MIF_r0_base + 0x0c)

#define MIF_r1_add                  (MIF_r1_base + 0x00)
#define MIF_r1_cnt                  (MIF_r1_base + 0x04)
#define MIF_r1_add2_skip            (MIF_r1_base + 0x08)
#define MIF_r1_cmd                  (MIF_r1_base + 0x0c)

// PCI Timeout (slave)
#define PCI_timeout_base            0x8000
#define PCI_timeout_value           (PCI_timeout_base + 0x00)
#define PCI_timeout_status          (PCI_timeout_base + 0x04)
#define PCI_timer_counter           (PCI_timeout_base + 0x08)
#define PCI_timer_test_register     (PCI_timeout_base + 0x0c)
#define PCI_wakeup_register         (PCI_timeout_base + 0x10)

// PCI SLAVE (directly accessed, slave)
#define PCI_slavecfg_base           0x9000
#define PCI_region_base             (PCI_slavecfg_base)
#define PCI_region_0_base           (PCI_region_base + 0x00)
#define PCI_region_1_base           (PCI_region_base + 0x04)
#define PCI_region_2_base           (PCI_region_base + 0x08)
#define PCI_region_3_base           (PCI_region_base + 0x0c)
#define PCI_region_4_base           (PCI_region_base + 0x10)
#define PCI_region_5_base           (PCI_region_base + 0x14)
#define PCI_region_6_base           (PCI_region_base + 0x18)
#define PCI_region_7_base           (PCI_region_base + 0x1c)
#define PCI_slavecfg_irq            (PCI_slavecfg_base + 0x20)

// PCI host
/* This file holds the definitive definition of PCI_chip_is_host for kernel
 * code. Undef'ing any previous definition.
 */
#undef PCI_chip_is_host

#if defined(CONFIG_ARCH_MAMBO)
#define PCI_chip_is_host            0xe000
#elif defined(CONFIG_ARCH_TANGO)
#define PCI_chip_is_host            0xfe90
#endif


// PCI device configuration
#define PCI_devcfg_base             0xfe00
#define PCI_devcfg_reg0             (PCI_REG0)
#define PCI_devcfg_reg1             (PCI_REG1)
#define PCI_devcfg_reg2             (PCI_REG2)
#define PCI_devcfg_reg3             (PCI_REG3)
#define PCI_devcfg_config           (PCI_CONFIG)

// PCI master access
#define PCI_master_base             0xfe00
#define PCI_master_read_addr        (PCI_master_base + 0xc0)
#define PCI_master_read_counter     (PCI_master_base + 0xc4)
#define PCI_master_read_enable      (PCI_master_base + 0xc8)
#define PCI_master_read_reverse     (PCI_master_base + 0xcc)
#define PCI_master_write_addr       (PCI_master_base + 0xd8)
#define PCI_master_write_counter    (PCI_master_base + 0xdc)
#define PCI_master_write_enable     (PCI_master_base + 0xe0)
#define PCI_master_burst            (PCI_master_base + 0xe4)

// PCI host functionality
#define PCI_host_base               0xfe00
#define PCI_host_host_reg1          (PCI_host_base + 0xd0)
#define PCI_host_host_reg2          (PCI_host_base + 0xd4)
#define PCI_host_host_reg3          (PCI_host_base + 0x80)
#define PCI_host_host_reg4          (PCI_host_base + 0x84)

#ifdef CONFIG_ARCH_TANGO

// Host Cipher : DES
#define TDMX_cipher_base            0x5000
#define TDMX_ciper_des_key_1_1      (TDMX_cipher_base + 0x00)
#define TDMX_ciper_des_key_1_2      (TDMX_cipher_base + 0x04)
#define TDMX_ciper_des_key_2_1      (TDMX_cipher_base + 0x08)
#define TDMX_ciper_des_key_2_2      (TDMX_cipher_base + 0x0c)
#define TDMX_ciper_des_key_3_1      (TDMX_cipher_base + 0x10)
#define TDMX_ciper_des_key_3_2      (TDMX_cipher_base + 0x14)
#define TDMX_ciper_des_iv_1         (TDMX_cipher_base + 0x18)
#define TDMX_ciper_des_iv_2         (TDMX_cipher_base + 0x1c)
#define TDMX_ciper_des_flags        (TDMX_cipher_base + 0x20)

// Host Cipher : AES
#define TDMX_ciper_aes_key_1        (TDMX_cipher_base + 0x80)
#define TDMX_ciper_aes_key_2        (TDMX_cipher_base + 0x84)
#define TDMX_ciper_aes_key_3        (TDMX_cipher_base + 0x88)
#define TDMX_ciper_aes_key_4        (TDMX_cipher_base + 0x8c)
#define TDMX_ciper_aes_key_5        (TDMX_cipher_base + 0x90)
#define TDMX_ciper_aes_key_6        (TDMX_cipher_base + 0x94)
#define TDMX_ciper_aes_key_7        (TDMX_cipher_base + 0x98)
#define TDMX_ciper_aes_key_8        (TDMX_cipher_base + 0x9c)
#define TDMX_ciper_aes_aes_flags    (TDMX_cipher_base + 0xa0)
#define TDMX_ciper_aes_iv_1         (TDMX_cipher_base + 0xa4)
#define TDMX_ciper_aes_iv_2         (TDMX_cipher_base + 0xa8)
#define TDMX_ciper_aes_iv_3         (TDMX_cipher_base + 0xac)
#define TDMX_ciper_aes_iv_4         (TDMX_cipher_base + 0xb0)
#define TDMX_ciper_aes_iv_5         (TDMX_cipher_base + 0xb4)
#define TDMX_ciper_aes_iv_6         (TDMX_cipher_base + 0xb8)
#define TDMX_ciper_aes_iv_7         (TDMX_cipher_base + 0xbc)
#define TDMX_ciper_aes_iv_8         (TDMX_cipher_base + 0xc0)

// IDE controller : IDE controller configuration registers
#define REG_BASE_HOST_BMIDE         (REG_BASE_HOST + IDECTRL_pri_cmdblock)
#define REG_BASE_HOST_BMIDE_DMA     (REG_BASE_HOST + IDECTRL_bmic)

#endif

/*
 * DRAM Controllers
 *
 * Base Address : REG_BASE_DRAMCTRLx
 */

// MBUS unit configuration
#define DRAM_mbus_base              0x0200
#define DRAM_mbus_w0_cfg            (DRAM_mbus_base + 0x00)
#define DRAM_mbus_w1_cfg            (DRAM_mbus_base + 0x04)
#define DRAM_mbus_w2_cfg            (DRAM_mbus_base + 0x08)
#define DRAM_mbus_w3_cfg            (DRAM_mbus_base + 0x0c)
#define DRAM_mbus_w4_cfg            (DRAM_mbus_base + 0x10)
#define DRAM_mbus_w5_cfg            (DRAM_mbus_base + 0x14)
#define DRAM_mbus_w6_cfg            (DRAM_mbus_base + 0x18)

#define DRAM_mbus_r0_cfg            (DRAM_mbus_base + 0x40)
#define DRAM_mbus_r1_cfg            (DRAM_mbus_base + 0x44)
#define DRAM_mbus_r2_cfg            (DRAM_mbus_base + 0x48)
#define DRAM_mbus_r3_cfg            (DRAM_mbus_base + 0x4c)
#define DRAM_mbus_r4_cfg            (DRAM_mbus_base + 0x50)
#define DRAM_mbus_r5_cfg            (DRAM_mbus_base + 0x54)
#define DRAM_mbus_r6_cfg            (DRAM_mbus_base + 0x58)
#define DRAM_mbus_r7_cfg            (DRAM_mbus_base + 0x5c)
#define DRAM_mbus_r8_cfg            (DRAM_mbus_base + 0x60)
#define DRAM_mbus_r9_cfg            (DRAM_mbus_base + 0x64)

// VBUS unit configuration
#define DRAM_vbus_base              0x0300
#define DRAM_vbus_r0_cfg            (DRAM_vbus_base + 0x00)
#define DRAM_vbus_r1_cfg            (DRAM_vbus_base + 0x04)
#define DRAM_vbus_r2_cfg            (DRAM_vbus_base + 0x08)
#define DRAM_vbus_r3_cfg            (DRAM_vbus_base + 0x0c)
#define DRAM_vbus_r4_cfg            (DRAM_vbus_base + 0x10)
#define DRAM_vbus_r5_cfg            (DRAM_vbus_base + 0x14)
#define DRAM_vbus_r6_cfg            (DRAM_vbus_base + 0x18)
#define DRAM_vbus_r7_cfg            (DRAM_vbus_base + 0x1c)

// GBUS to LBUS bridge
#define DRAM_g2l_base               0xff00
#define DRAM_g2l_bist_busy          (DRAM_g2l_base + 0xe0)
#define DRAM_g2l_bist_pass          (DRAM_g2l_base + 0xe4)
#define DRAM_g2l_bist_mask          (DRAM_g2l_base + 0xe8)
#define DRAM_g2l_reset_control      (DRAM_g2l_base + 0xfc)

/*
 * CPU Block
 *
 * Base Address : REG_BASE_CPU
 */

// CPU local ram
#define CPU_localmem_base           0x0000 

// UART - UART 0
#define CPU_uart0_base              CPU_UART0_base
#define CPU_uart0_rxd               (CPU_uart0_base + 0x00)
#define CPU_uart0_txd               (CPU_uart0_base + 0x04)
#define CPU_uart0_inten             (CPU_uart0_base + 0x08)
#define CPU_uart0_intid             (CPU_uart0_base + 0x0c)
#define CPU_uart0_fifoctl           (CPU_uart0_base + 0x10)
#define CPU_uart0_linectl           (CPU_uart0_base + 0x14)
#define CPU_uart0_modemctl          (CPU_uart0_base + 0x18)
#define CPU_uart0_linestat          (CPU_uart0_base + 0x1c)
#define CPU_uart0_modemstat         (CPU_uart0_base + 0x20)
#define CPU_uart0_scratch           (CPU_uart0_base + 0x24)
#define CPU_uart0_clkdiv            (CPU_uart0_base + 0x28)
#define CPU_uart0_clkset            (CPU_uart0_base + 0x2c)

#define CPU_uart0_gpio_dir          (CPU_uart0_base + 0x30)
#define CPU_uart0_gpio_data         (CPU_uart0_base + 0x34)
#define CPU_uart0_gpio_mode_w       (CPU_uart0_base + 0x38)
#define CPU_uart0_gpio_mode_r       (CPU_uart0_base + 0x3c)

#define CPU_UART0_BASE              (REG_BASE_CPU + CPU_uart0_base)     /* UART 0 */

// UART - UART 1
#define CPU_uart1_base              CPU_UART1_base
#define CPU_uart1_rxd               (CPU_uart1_base + 0x00)
#define CPU_uart1_txd               (CPU_uart1_base + 0x04)
#define CPU_uart1_inten             (CPU_uart1_base + 0x08)
#define CPU_uart1_intid             (CPU_uart1_base + 0x0c)
#define CPU_uart1_fifoctl           (CPU_uart1_base + 0x10)
#define CPU_uart1_linectl           (CPU_uart1_base + 0x14)
#define CPU_uart1_modemctl          (CPU_uart1_base + 0x18)
#define CPU_uart1_linestat          (CPU_uart1_base + 0x1c)
#define CPU_uart1_modemstat         (CPU_uart1_base + 0x20)
#define CPU_uart1_scratch           (CPU_uart1_base + 0x24)
#define CPU_uart1_clkdiv            (CPU_uart1_base + 0x28)
#define CPU_uart1_clkset            (CPU_uart1_base + 0x2c)

#define CPU_uart1_gpio_dir          (CPU_uart1_base + 0x30)
#define CPU_uart1_gpio_data         (CPU_uart1_base + 0x34)
#define CPU_uart1_gpio_mode_w       (CPU_uart1_base + 0x38)
#define CPU_uart1_gpio_mode_r       (CPU_uart1_base + 0x3c)

#define CPU_UART1_BASE              (REG_BASE_CPU + CPU_uart1_base)     /* UART 0 */

#define UART_GPIO_DIR_INPUT(gpio)      ((1 << (8 + (gpio))))
#define UART_GPIO_DIR_OUTPUT(gpio)     ((1 << (8 + (gpio))) | (1 << (gpio)))
#define UART_GPIO_DATA_SET(gpio)       ((1 << (8 + (gpio))) | (1 << (gpio)))
#define UART_GPIO_DATA_CLEAR(gpio)     ((1 << (8 + (gpio))))

// Timer - Timer 0
#define CPU_timer0_base             0xc500
#define CPU_timer0_load             (CPU_timer0_base + 0x00)
#define CPU_timer0_value            (CPU_timer0_base + 0x04)
#define CPU_timer0_ctrl             (CPU_timer0_base + 0x08)
#define CPU_timer0_clr              (CPU_timer0_base + 0x0c)

#define CPU_TIMER0_BASE             (REG_BASE_CPU + CPU_timer0_base)    /* TIMER 0 */

// Timer - Timer 1
#define CPU_timer1_base             0xc600
#define CPU_timer1_load             (CPU_timer1_base + 0x00)
#define CPU_timer1_value            (CPU_timer1_base + 0x04)
#define CPU_timer1_ctrl             (CPU_timer1_base + 0x08)
#define CPU_timer1_clr              (CPU_timer1_base + 0x0c)

#define CPU_TIMER1_BASE             (REG_BASE_CPU + CPU_timer1_base)    /* TIMER 1 */

// Interupt controller - IRQ
#define CPU_irq_base                CPU_irq_status

// Interupt controller - FIQ
#define CPU_fiq_base                CPU_fiq_status

// RESET
#define CPU_reset                   G2L_RESET_CONTROL

/*
 * Video Output
 *
 * Base Address : REG_BASE_VIDEOOUT
 */

/*
 * MPEG Engine
 *
 * Base Address : REG_BASE_MPEGx
 */

/*
 * Transport Demux
 *
 * Base Address : REG_BASE_TSDEMUX
 */

/*
 * Audio Block
 *
 * Base Address : REG_BASE_AUDIOx
 */

/*
 * Others
 */

#define UART_NR                 2   // number of UART port. 
                                    // needed by drivers/char/serial_em86xx.c

/*
 * PCI
 */

#if defined(CONFIG_PCI_EM86XX_HOST_EM86XX)
#define PCIBIOS_MIN_IO          MEMORY_BASE_PCI_IO
#define PCIBIOS_MIN_MEM         (MEMORY_BASE_PCI_MEMORY + 0x00100000)   /* avoid 0x00000000 on bus addr */
#define PCIBIOS_MIN_MEM_EM86XX  (MEMORY_BASE_PCI_MEMORY + 0x10000000)   /* base address of EM86xx PCI slave */
#elif defined(CONFIG_PCI_EM86XX_HOST_FPGA)
#include <asm/arch/pcifpga.h>
#define PCIBIOS_MIN_IO          PCIFPGA_IO_BASE
#define PCIBIOS_MIN_MEM         PCIFPGA_MEMORY_BASE
#endif

// set to 1 if the Phy Addr == Bus Addr, otherwise 0
#define PCI_DMA_BUS_IS_PHYS     0

/*
 * Clock 
 *
 * Only use timer 0
 */

// PLL input clock, typically 27 Mhz
#define EM86XX_EXT_CLOCK        27000000    

#ifndef __ASSEMBLY__
#include <asm/io.h>

static __inline__ unsigned long __get_clock(unsigned int unit)
{
    unsigned long clock;
    unsigned int pll_reg, pll_mult, pll_div;

    pll_reg = __raw_readl(REG_BASE_SYSTEM + SYS_clkgen0_pll);
    pll_mult = (pll_reg & 0x03ff);
    pll_div = (pll_reg >> 16) & 0x003f;

    clock = (EM86XX_EXT_CLOCK / unit) * (pll_mult + 2) / (pll_div + 2);

    return clock;
}

static __inline__ unsigned int __get_clock_mhz(void)
{
    return __get_clock(2) / 1000000;
}

/*
 * Others
 */

#define HARD_RESET_NOW()                

#endif  // __ASSEMBLY__

#endif

