/**********
This library is free software; you can redistribute it and/or modify it under
the terms of the GNU Lesser General Public License as published by the
Free Software Foundation; either version 2.1 of the License, or (at your
option) any later version. (See <http://www.gnu.org/copyleft/lesser.html>.)

This library is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
more details.

You should have received a copy of the GNU Lesser General Public License
along with this library; if not, write to the Free Software Foundation, Inc.,
59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
**********/
// "liveMedia"
// Copyright (c) 1996-2004 Live Networks, Inc.  All rights reserved.
// A file source that is a plain byte stream (rather than frames)
// Implementation

#if defined(__WIN32__) || defined(_WIN32)
#include <io.h>
#include <fcntl.h>
#endif

#include "ByteStreamFileSource.hh"
#include "GroupsockHelper.hh"

#if defined(VXWORKS)
#include <LfxNew.h>
#endif

////////// ByteStreamFileSource //////////

ByteStreamFileSource::ByteStreamFileSource(UsageEnvironment& env, FILE* fid,
					   unsigned preferredFrameSize,
					   unsigned playTimePerFrame)
  : FramedFileSource(env, fid), fPreferredFrameSize(preferredFrameSize),
    fPlayTimePerFrame(playTimePerFrame), fLastPlayTime(0) {
}

ByteStreamFileSource::~ByteStreamFileSource() {
  fclose(fFid);
}

ByteStreamFileSource*
ByteStreamFileSource::createNew(UsageEnvironment& env, char const* fileName,
				unsigned preferredFrameSize,
				unsigned playTimePerFrame) {
  do {
    FILE* fid;

    // Check for a special case file name: "stdin"
    if (strcmp(fileName, "stdin") == 0) {
      fid = stdin;
#if defined(__WIN32__) || defined(_WIN32)
      _setmode(_fileno(stdin), _O_BINARY); // convert to binary mode
#endif
    } else { 
     fid = fopen(fileName, "rb");
      if (fid == NULL) {
	env.setResultMsg("unable to open file \"",fileName, "\"");
	break;
      }
    }

    return new ByteStreamFileSource(env, fid, preferredFrameSize,
				    playTimePerFrame);
  } while (0);

  return NULL;
}

#ifdef BSD
static struct timezone Idunno;
#else
static int Idunno;
#endif

// tea 10/09/03: 
// * moving back file pointer to beginning of file to implement loop 
//
void ByteStreamFileSource::reset() {
  if (ferror(fFid)) {
    handleClosure(this);
    return;
  }
  if (fseek(fFid, 0, SEEK_SET)) {
    envir().setResultMsg("unable to seek to beginning of file");
  }
}


void ByteStreamFileSource::doGetNextFrame() {
  if (feof(fFid) || ferror(fFid)) {
    handleClosure(this);
    return;
  }

  // Try to read as many bytes as will fit in the buffer provided
  // (or "fPreferredFrameSize" if less)
  if (fPreferredFrameSize > 0 && fPreferredFrameSize < fMaxSize) {
    fMaxSize = fPreferredFrameSize;
  }
  fFrameSize = fread(fTo, 1, fMaxSize, fFid);

  // Set the 'presentation time':
  if (fPlayTimePerFrame > 0 && fPreferredFrameSize > 0) {
    if (fPresentationTime.tv_sec == 0 && fPresentationTime.tv_usec == 0) {
      // This is the first frame, so use the current time:
      gettimeofday(&fPresentationTime, &Idunno);
    } else {
      // Increment by the play time of the previous data:
      unsigned uSeconds	= fPresentationTime.tv_usec + fLastPlayTime;
      fPresentationTime.tv_sec += uSeconds/1000000;
      fPresentationTime.tv_usec = uSeconds%1000000;
    }

    // Remember the play time of this data:
    fLastPlayTime = (fPlayTimePerFrame*fFrameSize)/fPreferredFrameSize;
    fDurationInMicroseconds = fLastPlayTime;
  } else {
    // We don't know a specific play time duration for this data,
    // so just record the current time as being the 'presentation time':
    gettimeofday(&fPresentationTime, &Idunno);
  }

  // Switch to another task, and inform the reader that he has data:
  nextTask() = envir().taskScheduler().scheduleDelayedTask(0,
				(TaskFunc*)FramedSource::afterGetting, this);
}
